#-------------------------------------------------------------------------------
#  afpSpool.pm:
#
#  Module to process AFP spool and generate store process requests
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -doc_cold -sp:afpSpool.pm -sa:<json string> samples\insure.pdf
#
#  On Unix:    ./afp2web   -q -c -doc_cold -sp:afpSpool.pm -sa:<json string> samples/insure.pdf
#
#  where <json string> (given in ScriptArguments) must have following structure
#  {
#      'RequestId': <processSpool.RequestId>,
#      'DocumentType': <AFP | dump>,
#      'DocumentData': <processSpool.ProcessingQueues.DocumentData logical location json>,
#      'storeProcessRequest': <processSpool.ProcessingQueues.storeProcessRequest logical location json>,
#      'WorkBasketId': <processSpool.WorkBasketId>
#      'TransactionSourceID': <processSpool.TransactionSourceID>
#      'DocumentSourceXMLRef': <processSpool.DocumentSourceXMLRef>
#      'DocumentSourceID': <processSpool.DocumentSourceID>
#      'ProcessTypeXmlRef': <processSpool.ProcessTypeXmlRef>
#      'ProcessTypeID': <processSpool.ProcessTypeID>
#      'DocumentTypeXmlRef': <processSpool.DocumentTypeXmlRef>
#      'IndexAliasList': <processSpool.IndexAliasList>
#  }
#
#  Author  : Fa. Maas
#
#  $V100   2019-01-31    Initial Release
#
#-------------------------------------------------------------------------------
#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;
    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }
        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }
    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
    unshift( @INC, $sScriptFilePathTmp . "/a2w" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/site/lib" );
}

use a2w::Config;
use a2w::ConfigConstants;
use a2w::Kernel;

use a2w::request::ProcessSpool;
use a2w::request::StoreProcess;

use Time::HiRes;
use Digest::MD5;
use File::Path;
use File::Copy; # For renaming files across filesystems
use Storable qw( dclone );

use a2w::core::log::Logger; # Log engine

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){
    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Fetch start time
    $tStart = time;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
	$sDocType          = $a2wConfigPar->getAttribute( "OutputFormat" );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    #---- Fetch AFP2web version
    $sA2WVersion = $a2wKernelPar->getVersion();

    #---- Instantiate Logger
    $theLogger = undef;
    $bDebugLog = $FALSE;
    if ( $bLog == $TRUE ){
        #---- Set log on log engine
        $theLogger = a2w::core::log::Logger->getSingleton();
        $theLogger->setStartTime( $sStartTime );

        #---- Register modules that has to be logged
        $theLogger->registerClasses(   "a2w::Main"
                                     . ",a2w::request::ProcessSpool"
                                     . ",a2w::request::StoreProcess"
                                     . ",a2w::process::afp"
                                     . ",a2w::process::dump"
                                   );

        #---- Open log file
        my $sLogFilenameTmp = $sSpoolFilename;
        $sLogFilenameTmp =~ s/^.*[\\\/]//;
        $sLogFilenameTmp .= ".sf.log";
        $theLogger->setFilename( $sLogFilenameTmp );
        if ( $theLogger->open( $sOutputFilePath ) == $FALSE ){
            return ( -1, "[ERROR] Unable to open log file (" . $sOutputFilePath . $sLogFilenameTmp . ")" );
        }
        $bLog = $theLogger->isRegistered( "a2w::Main" );

        if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sfdbg") >= 0 ){
            $theLogger->setLevel( $a2w::core::log::Logger::LEVEL_DEBUG );
            $bDebugLog = $TRUE;
        }

        $theLogger->logFunctionName( "a2w::Main", "initialize()" );
        $theLogger->logMessage( "Running $sScriptProcTmp..." );
        $theLogger->logMessage( "initialize(): Processing " . $sSpoolFilename );
        $theLogger->logMessage( "initialize(): Args: $sScriptArgsTmp, OutputFilePath: $sOutputFilePath" );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Create process spool request
    my $iRetTmp = 0;
    $reqProcessSpool = undef; # Process spool request object
    ( $iRetTmp, $reqProcessSpool ) = _createProcessSpoolRequest( $sScriptArgsTmp );
    if ( $iRetTmp < 0 ){ return ( $iRetTmp, $reqProcessSpool ); }

    #---- Validate process spool request
    ( $iRetTmp, $sMsgTmp ) = _validateProcessSpoolRequest( $reqProcessSpool );
    if ( $iRetTmp < 0 ){ return ( $iRetTmp, $sMsgTmp ); }
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "initialize(): ReqID: " . $reqProcessSpool->getRequestId() . ", LayoutType: " . $reqProcessSpool->getInputFileDocumentType() ); }

    #---- Fetch request id
    $sRequestId = $reqProcessSpool->getRequestId();

    # Initalize document and page count
    $iPageCount = 0;
    $iDocCount  = 0;

    #---- Set protocol file open status
    $iProtocolOpened = 0;

    #---- Protocol filename
    $sProtocolFilename = "";

    #---- Statistic filename
    $sStatisticFilename = "";

    #---- Initialize document data specified output path
    $sDocDataOutputPath = "";

    #---- Initialize store process specified output path
    $sStoreProcOutputPath = "";

    #---- Initialize document data specified output path with request id added
    $sDDOPWithReqId = "";

    #---- Initialize MD5 context
    $md5Context = undef;

    #---- Initialize logical location references
    $llDocumentData = $reqProcessSpool->getDocumentData();    # Logical location of output document
    $llStoreProcess = $reqProcessSpool->getStoreProcess();    # Logical location of store process request

    #---- Initialize last store process filename
    $sLastStoreProcessFilename = "";

    #---- Time stamp
    # (sec,min,hour,mday,mon,year,wday,yday,isdst) = localtime( time );
    @StartTime = localtime( $tStart );
    $sStartTimeStamp = sprintf( "%04d%02d%02d_%02d%02d%02d",
                              ( @StartTime[ 5 ] + 1900 ),
                              ( @StartTime[ 4 ] + 1 ),
                              @StartTime[ 3 ],
                              @StartTime[ 2 ],
                              @StartTime[ 1 ],
                              @StartTime[ 0 ]
                            );

    #---- Declare and Initialize the function references
    $frefInitialize     = undef;
    $frefInitializeDoc  = undef;
    $frefInitializePage = undef;
    $frefAfp2web        = undef;
    $frefFinalizePage   = undef;
    $frefFinalizeDoc    = undef;
    $frefFinalize       = undef;

    $frefGetSPProtocolFilename   = undef;
    $frefGetVSNRProtocolFilename = undef;

    #---- Dump Layout should be used for dumping page content (for testing purposes)
    $sLayoutTypeTmp = $reqProcessSpool->getInputFileDocumentType();
    if ( lc( $sLayoutTypeTmp ) eq "dump" ){

        #---- Dynamically load the specific module according to the Layout Type
        require a2w::process::dump;

        #---- Initialize the function references with proper definitions
        $frefInitialize     = \&a2w::process::dump::initialize;
        $frefInitializeDoc  = \&a2w::process::dump::initializeDoc;
        $frefInitializePage = \&a2w::process::dump::initializePage;
        $frefAfp2web        = \&a2w::process::dump::afp2web;
        $frefFinalizePage   = \&a2w::process::dump::finalizePage;
        $frefFinalizeDoc    = \&a2w::process::dump::finalizeDoc;
        $frefFinalize       = \&a2w::process::dump::finalize;
    }
    #---- Default is AFP (MO:DCA) Spool file Process
    else {
        
        #---- Dynamically load the specific module according to the Layout Type
        require a2w::process::afp;

        #---- Initialize the function references with proper definitions
        $frefInitialize     = \&a2w::process::afp::initialize;
        $frefInitializeDoc  = \&a2w::process::afp::initializeDoc;
        $frefInitializePage = \&a2w::process::afp::initializePage;
        $frefAfp2web        = \&a2w::process::afp::afp2web;
        $frefFinalizePage   = \&a2w::process::afp::finalizePage;
        $frefFinalizeDoc    = \&a2w::process::afp::finalizeDoc;
        $frefFinalize       = \&a2w::process::afp::finalize;

        $frefGetSPProtocolFilename   = \&a2w::process::afp::getSkippedPagesProtocolFilename;
        $frefGetVSNRProtocolFilename = \&a2w::process::afp::getVSNRProtocolFilename;
    }

    my $iInitializeRetTmp = -1;
    if ( $frefInitialize != undef ){
        $iInitializeRetTmp = &$frefInitialize( $a2wConfigPar, $a2wKernelPar, $reqProcessSpool, $tStart );
    }
    return $iInitializeRetTmp;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "initializeDoc()" );
        $theLogger->logMessage( "Name=" . $a2wDocumentPar->getName() . " Id=" . $iDocumentId );
    }

    #---- Open protocol file and write header
    if ( $iProtocolOpened == 0 ){
        $iProtocolOpened = _openProtocolAndWriteHeader();
    }

    my $iInitializeDocRetTmp = -1;
    if ( $frefInitializeDoc != undef ){
        $iInitializeDocRetTmp = &$frefInitializeDoc( $a2wDocumentPar );
    }
    return $iInitializeDocRetTmp;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "initializePage()" ); }

    #---- Increment page count
    $iPageCount++;

    my $iInitializePageRetTmp = -1;
    if ( $frefInitializePage != undef ){
        $iInitializePageRetTmp = &$frefInitializePage( $a2wPagePar );
    }
    return $iInitializePageRetTmp;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "afp2web()" );
        $theLogger->logMessage( "PageId=" . $a2wPagePar->getParseId() );
    }

    my $iA2WRetTmp = -1;
    if ( $frefAfp2web != undef ){
        $iA2WRetTmp = &$frefAfp2web();
    }
    return $iA2WRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizePage()" ); }

    my $iFinalizePageRetTmp = -1;
    if ( $frefFinalizePage != undef ){
        $iFinalizePageRetTmp = &$frefFinalizePage();
    }
    return $iFinalizePageRetTmp;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizeDoc()" ); }

    if ( $a2wDocumentPar->getPageCount() <= 0 ){
	    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Document contain no pages, skipping ..." ); }
        return 0;
    }

    #---- Increment document count
    $iDocCount++;

    #---- Reset last store process request filename
    $sLastStoreProcessFilename = "";

    #---- Call process specific finalize doc
    my $iFinalizeDocRetTmp = -1;
    if ( $frefFinalizeDoc != undef ){
        $iFinalizeDocRetTmp = &$frefFinalizeDoc();
    }
    if ( $iFinalizeDocRetTmp < 0 ){
        return ( $iFinalizeDocRetTmp, "Unable to do spool type specific document finalize" );
    }

    #---- Process DocumentData logical location and set output path appropriately
    if ( $llDocumentData->{ 'FileCount' } ne "" ){
        if ( exists( $llDocumentData->{ 'FileCount' } ) ){
            if ( $llDocumentData->{ 'FileCount' } <= 0 ){
                $llDocumentData->{ 'FileCount' } = ~0;    # Maximum integer data type range
            }
        }
        if ( $llDocumentData->{ 'FileCount' } > 0
             && ( $iDocCount % $llDocumentData->{ 'FileCount' } ) == 1 ){
            #---- Extended to handle return value with error message
            my ( $iDDLLRetTmp, $sErrMsgTmp ) = _processLogicalLocation( $llDocumentData, \$sDocDataOutputPath );
            if ( $iDDLLRetTmp < 0 ){
                if ( $sErrMsgTmp eq "" ){
                    return ( $iDDLLRetTmp, "Unable to process document data logication details" );
                }
                else {
                    return( $iDDLLRetTmp, $sErrMsgTmp );
                }
            }
        }
    }

    #----------------------------------------------------------------------------------------------------
    # Bug:
    #   Output files are not renamed proerly when "FileCount" attribute is missing under
    #   under "DocumentData" logical location
    #
    # Reason:
    #   Output path ($sDocDataOutputPath) is evaluated as per "DocumentData" logical location's only
    #   when "FileCount" attribute is not empty or else it is not evaluated and defaults to empty
    #   string, which makes process to rename files wrongly
    #
    # Fix:
    #   Evaluate output path ($sDocDataOutputPath) even when "FileCount" attribute is empty
    #
    #----------------------------------------------------------------------------------------------------
    elsif ( $sDocDataOutputPath eq "" ){    # Evaluate for the first time only
        #---- Extended to handle return value with error message
        my ( $iDDLLRetTmp, $sErrMsgTmp ) = _processLogicalLocation( $llDocumentData, \$sDocDataOutputPath );
        if ( $iDDLLRetTmp < 0 ){
            if ( $sErrMsgTmp eq "" ){
                return ( $iDDLLRetTmp, "Unable to process document data logication details" );
            }
            else {
                return( $iDDLLRetTmp, $sErrMsgTmp );
            }
        }
    }

    #---- Assert store process logical location before use
    if ( keys( %{ $llStoreProcess } ) != 0 ){
        #---- Process StoreProcess logical location and set output path appropriately
        if ( $llStoreProcess->{ 'FileCount' } ne "" ){
            if ( exists( $llStoreProcess->{ 'FileCount' } ) ){
                if ( $llStoreProcess->{ 'FileCount' } <= 0 ){
                    $llStoreProcess->{ 'FileCount' } = ~0;    # Maximum integer data type range
                }
            }
            if ( $llStoreProcess->{ 'FileCount' } > 0
                 && ( $iDocCount % $llStoreProcess->{ 'FileCount' } ) == 1 ){
                my ( $iSPLLRetTmp, $sErrMsgTmp ) = _processLogicalLocation( $llStoreProcess, \$sStoreProcOutputPath );
                if ( $iSPLLRetTmp < 0 ){
                    if ( $sErrMsgTmp eq "" ){
                        return ( $iSPLLRetTmp, "Unable to process store process logication details" );
                    }
                    else {
                        return( $iSPLLRetTmp, $sErrMsgTmp );
                    }
                }
            }
        }
        elsif ( $sStoreProcOutputPath eq "" ){    # Evaluate for the first time only
            my ( $iSPLLRetTmp, $sErrMsgTmp ) = _processLogicalLocation( $llStoreProcess, \$sStoreProcOutputPath );
            if ( $iSPLLRetTmp < 0 ){
                if ( $sErrMsgTmp eq "" ){
                    return ( $iSPLLRetTmp, "Unable to process store process logication details" );
                }
                else {
                    return( $iSPLLRetTmp, $sErrMsgTmp );
                }
            }
        }
    }

    #---- Fetch output filename
    my $sOutputFilenameTmp = $a2wDocumentPar->getOutputFilename();
    my $sOutputFileExtTmp  = "";
    if ( $sOutputFilenameTmp =~ /(\..{3})$/ ){
        $sOutputFileExtTmp = $1;
    }
    my $sFQOutputFilenameTmp = $sOutputFilePath . $sOutputFilenameTmp;

    my $sFilenameIndexValueTmp = "";
    my $sFormattedOutputFilenameTmp = "";
    my $sDocNameTmp = $a2wDocumentPar->getName();
    if ( -e $sFQOutputFilenameTmp ){
        #
        # Rename output file based on "I_FILENAME" index, if exist else use filename pattern of logical location
        #
        $sFilenameIndexValueTmp = _fetchDocumentIndex( "I_FILENAME" );
        $sFormattedOutputFilenameTmp = $sFilenameIndexValueTmp;

        if ( $sFormattedOutputFilenameTmp eq "" ){
            #---- Fetch output filename based on pattern
            $sFormattedOutputFilenameTmp = _fetchPatternFormattedFilename( $llDocumentData
                                                                           , $sFQOutputFilenameTmp
                                                                           , $iDocCount
                                                                           , $sDocNameTmp
                                                                         );
        }

        #---- Build new output filename based on document data logical location
        #     given filename pattern and rename kernel generated output
        #
        my $sNewOutputFilenameTmp = $sDocDataOutputPath . $sFormattedOutputFilenameTmp . $sOutputFileExtTmp;

        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Renaming output file " . $sFQOutputFilenameTmp . " to " . $sNewOutputFilenameTmp ); }

        my $iRenameStatusTmp = move( $sFQOutputFilenameTmp, $sNewOutputFilenameTmp );
        if ( !$iRenameStatusTmp ){
            return ( -1, "Unable to rename " . $sFQOutputFilenameTmp . " as " . $sNewOutputFilenameTmp . ", Reason: " . $! );
        }
    }
    else {
        return ( -2, "Output file " . $sFQOutputFilenameTmp . " doesn't exist" );
    }

    #---- Write protocol document entry
    if ( $iProtocolOpened > 0 ){
        _writeProtocolDocumentEntry( $a2wDocumentPar->getId(),
                                     $sFormattedOutputFilenameTmp,
                                     1
                                   );
    }

    #---- Build store process request filename based on pattern
    #
    # Rename store process file based on "I_FILENAME" index, if exist else use filename pattern of logical location
    #
    my $sFormattedSPFilenameTmp = $sFilenameIndexValueTmp;

    if ( $sFormattedSPFilenameTmp eq "" ){
        #---- Fetch output filename based on pattern
        $sFormattedSPFilenameTmp = _fetchPatternFormattedFilename( $llStoreProcess
                                                                   #, ""
																   , $sFQOutputFilenameTmp
                                                                   , $iDocCount
                                                                   , $sDocNameTmp
                                                                 );
    }

    #---- Do not try to modify store process if logical location type is not DIR
    if ( lc( $llStoreProcess->{ 'Type' } ) ne "dir" ){
        $sFormattedSPFilenameTmp = "";
    }

    #---- Assert generated store process request filename instead of creating file with improper name
    if ( $sFormattedSPFilenameTmp ne "" ){
        my $sFQStoreProcessFilenameTmp = $sStoreProcOutputPath . $sFormattedSPFilenameTmp . ".xml";
        $sLastStoreProcessFilename = $sFQStoreProcessFilenameTmp;

        #---- Fetch page count
        my $iPageCountTmp = $a2wDocumentPar->getPageCount;

        #---- Fetch first and last page id ----#
        my $a2wPageTmp   = $a2wDocumentPar->getFirstPage();
        my $iPageFromTmp = 0;
        my $iPageToTmp   = 0;
        if ( $a2wPageTmp != 0 ){
            #---- Fetch first page id
            $iPageFromTmp = $a2wPageTmp->getParseId();

            #---- Loop through upto last page document
            do{
                #---- Fetch page id
                $iPageToTmp = $a2wPageTmp->getParseId();

                $a2wPageTmp = $a2wDocumentPar->getNextPage();
            } while ( $a2wPageTmp != 0 );
        }

        #---- Write store process request for current document
        _writeStoreProcessRequest( $sFQStoreProcessFilenameTmp,
                                   $iDocCount,
                                   $sFormattedOutputFilenameTmp,
                                   $iPageCountTmp,
                                   $iPageFromTmp,
                                   $iPageToTmp
                                 );
    }
    else {
        #---- Do not try to modify store process if logical location type is not DIR
        if ( lc( $llStoreProcess->{ 'Type' } ) eq "dir" ){
            return ( -2, "Empty store process request XML filename for document " . $iDocCount );
        }
    }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalize()" ); }

    #---- Fetch start time
    $tEnd = time;

    my $iFinalizeRetTmp = -1;
    if ( $frefFinalize != undef ){
        $iFinalizeRetTmp = &$frefFinalize();
    }
    if ( $iFinalizeRetTmp < 0 ){
        return ( $iFinalizeRetTmp, "Unable to do spool type specific finalize" );
    }

    #---- Modify last document's "StoreProcess" request
    #     a. The last document storeprocess request must have
    #        have "TransactionEnd" element with value "true"
    #        as given below
    #
    #        ns:StoreProcess->ImportProcess->Transaction->TransactionEnd
    #
    #     b. The last document storeprocess request must have
    #        "ProcessCount" element with value "output document count"
    #        as given below
    #
    #        ns:StoreProcess->ImportProcess->ProcessCount
    #
    if ( $sLastStoreProcessFilename ne "" ){
        my $MSPRetTmp = _modifyLastDocumentSP( $sLastStoreProcessFilename );
        if ( $MSPRetTmp < 0 ){
            return ( $MSPRetTmp, "Unable to modify last StoreProcess request ($sLastStoreProcessFilename)" );
        }
    }

    #---- Rename output path from "Working" as "Done" or as "Error" based on process completion status
    if ( $sDDOPWithReqId ne ""
         && -d $sDDOPWithReqId
       ){
        #---- Fetch exit status from kernel
        my $iA2WExitStatusTmp = $a2wKernelPar->getExitStatus();
		if ( $bLog == $TRUE ){ $theLogger->logMessage( "Kernel exit status: " . $iA2WExitStatusTmp ); }

        my $sRenameStatusTmp = "Done";
        if ( $iA2WExitStatusTmp < 0 ){ $sRenameStatusTmp = "Error"; }

        my $sDDOPWithStatusTmp = $sDDOPWithReqId;
        $sDDOPWithStatusTmp =~ s/Working/$sRenameStatusTmp/;

        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Renaming " . $sDDOPWithReqId . " as " . $sDDOPWithStatusTmp ); }

        my $iRenameStatusTmp = move( $sDDOPWithReqId, $sDDOPWithStatusTmp );
        if ( !$iRenameStatusTmp ){
            if ( $sRenameStatusTmp eq "Done" ){
                # Renaming "Working" as "Done" failed, so try to rename as "Error"
                if ( $bLog == $TRUE ){ $theLogger->logMessage( ": Failed, Reason: " . $! ); }
                $sDDOPWithStatusTmp = $sDDOPWithReqId;
                $sDDOPWithStatusTmp =~ s/Working/Error/;
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Renaming " . $sDDOPWithReqId . " as " . $sDDOPWithStatusTmp ); }
                $iRenameStatusTmp = move( $sDDOPWithReqId, $sDDOPWithStatusTmp );
            }
        }

        if ( !$iRenameStatusTmp ){
            return ( -1, "Unable to rename " . $sDDOPWithReqId . " as " . $sDDOPWithStatusTmp . ", Reason: " . $! );
        }
        if ( $bLog == $TRUE ){ $theLogger->logMessage( ": Ok" ); }
    }

    #---- Write protocol footer and close
    if ( $iProtocolOpened > 0 ){
        _writeFooterAndCloseProtocol();
    }

    #---- Write statistics file
    _writeStatisticsFile();

    #---- Skipped pages protocol filename
    my $sSkipPageProtocolFilenameTmp = "";
    if ( $frefGetSPProtocolFilename != undef ){
        $sSkipPageProtocolFilenameTmp = &$frefGetSPProtocolFilename();
    }

    #---- VSNR protocol filename
    my $sVSNRProtocolFilenameTmp = "";
    if ( $frefGetVSNRProtocolFilename != undef ){
        $sVSNRProtocolFilenameTmp = &$frefGetVSNRProtocolFilename();
    }

    return 0;
}

#-----------------------------------------------------------------------
# Create process spool request
#
# Parameter(s)
# 1. JSON string of process spool
#
#-----------------------------------------------------------------------
sub _createProcessSpoolRequest{
    #---- Fetch parameter(s)
    #
    # 1. JSON string of process spool
    #
    my $sPSJsonPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_createProcessSpoolRequest()" ); }

    #---- Parse json string (given in script argument)
    my $hrefPSTmp = undef;
    eval{
        require JSON::Tiny;
        $hrefPSTmp = JSON::Tiny::from_json( $sPSJsonPar );
    };
    if ( $@ ){
        return ( -2, "Unable to parse json string given in script argument ($sPSJsonPar). rc=$hrefPSTmp reason=" . $@ );
    }
    elsif ( $hrefPSTmp == undef ){
        return ( -2, "Unable to parse json string given in script argument ($sPSJsonPar). rc=$hrefPSTmp" );
    }

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "ProcessSpool: " );
		$theLogger->logHashMessage( $hrefPSTmp );
    }

    #---- Create process spool request object
    my $reqPSTmp = new a2w::request::ProcessSpool( $hrefPSTmp ); # Process spool hash
    if ( $reqPSTmp == undef ){ return ( -3, "Unable to allocate process spool request" ); }

    return ( 0, $reqPSTmp );
}

#-----------------------------------------------------------------------
# Validate process spool request
#
# Parameter(s)
# 1. Process spool request object
#
#-----------------------------------------------------------------------
sub _validateProcessSpoolRequest{
    #---- Fetch parameter(s)
    #
    # 1. Process spool request object
    #
    my $reqPSPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_validateProcessSpoolRequest()" ); }

    #---- Validate request id
    if ( $reqPSPar->getRequestId() eq "" ){ return ( -5, "RequestId missing in process spool request" ); }

    #---- Validate layout type
    if ( $reqPSPar->getInputFileDocumentType() eq "" ){ return ( -6, "DocumentType missing in process spool request" ); }

    #---- Validate document data logical location
    my $hrefDDTmp = $reqPSPar->getDocumentData();
    if ( $hrefDDTmp == undef ){ return ( -7, "DocumentData logical location missing in process spool request" ); }
    if ( !defined( $hrefDDTmp->{ 'DirectoryName' }) ){ return ( -7, "Missing DirectoryName in document data logical location" ); }

    my $hrefSPTmp = $reqPSPar->getStoreProcess();
    if ( $hrefSPTmp != undef ){
        if ( !defined( $hrefSPTmp->{ 'DirectoryName' }) ){ return ( -8, "Missing DirectoryName in store process logical location" ); }
    }

    return ( 0, undef );
}

#----------------------------------------------------------------------------------------
# Fetch document index list
#
# Processes document level indexes and returns a hash reference having
# index name as key and index value as value
#
#---------------------------------------------------------------------------------------
sub _fetchDocumentIndexes{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_fetchDocumentIndexes()" ); }

    # Initialize index hash
    my $idxlstDocumentTmp = {};

    #----- Assert document instance
    if ( $a2wDocumentPar == undef ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Invalid document handle to fetch indexes" ); }
        return $idxlstDocumentTmp;
    }
    else {
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "" ); }
    }

    # Iniitialize variables
    my $sIndexNameTmp  = undef;
    my $sIndexValueTmp = undef;

    #---- Fetch first Document Index
    my $a2wDocIndexTmp = $a2wDocumentPar->getFirstIndex();

    while ( $a2wDocIndexTmp != 0 ){
        #---- Get index name
        $sIndexNameTmp = $a2wDocIndexTmp->getName();

        if ( $sIndexNameTmp ne "" ){
            $sIndexValueTmp = $a2wDocIndexTmp->getValue();
            $idxlstDocumentTmp->{ $sIndexNameTmp } = $sIndexValueTmp;

            if ( $bLog == $TRUE ){ $theLogger->logMessage( $sIndexNameTmp . "=" . $sIndexValueTmp ); }
        }

        #---- Get next document index
        $a2wDocIndexTmp = $a2wDocumentPar->getNextIndex();
    }

    # $V102 Begin
    #---- Fetch first page
    my $a2wPageTmp = $a2wDocumentPar->getFirstPage();
    my $a2wPageIndexTmp = undef;

    #---- Loop through pages and collect indexes
    while ( $a2wPageTmp != 0 ){
        #---- Fetch first Document Index
        $a2wPageIndexTmp = $a2wPageTmp->getFirstIndex();

        while ( $a2wPageIndexTmp != 0 ){
            #---- Get index name
            $sIndexNameTmp = $a2wPageIndexTmp->getName();

            if ( $sIndexNameTmp ne "" ){
                $sIndexValueTmp = $a2wPageIndexTmp->getValue();
                $idxlstDocumentTmp->{ $sIndexNameTmp } = $sIndexValueTmp;

                if ( $bLog == $TRUE ){ $theLogger->logMessage( $sIndexNameTmp . "=" . $sIndexValueTmp ); }
            }

            #---- Get next document index
            $a2wPageIndexTmp = $a2wPageTmp->getNextIndex();
        }    # while ( $a2wPageIndexTmp != 0 ){

        #---- Fetch next page
        $a2wPageTmp = $a2wDocumentPar->getNextPage();
    }    # while ( $a2wPageTmp != 0 ){
    # $V102 End

    return $idxlstDocumentTmp;
}

#-----------------------------------------------------------------------
# Processes logical location and does following actions
#
# Parameter(s)
# 1. Logical location (Hash reference)
# 2. Output path (String reference)
#
# Returns
# >0  on success
# <=  on error
#
# Remarks
# 1. Modifies output filepath based on "DirectoryName" value
# 2. Appends output filepath based on "PathSuffixPattern" value
#
#-----------------------------------------------------------------------
sub _processLogicalLocation{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_processLogicalLocation()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Logical location (Hash reference)
    # 2. Output path (String reference)
    #
    my $hrefLogicalLocationPar = shift;
    my $srefOutputPathPar = shift;

    #---- Assert logical location type and do actions
    # $V11 Begin
    #if ( $hrefLogicalLocationPar->{ 'Type' } eq "DIR" ){
    if ( lc( $hrefLogicalLocationPar->{ 'Type' } ) eq "dummy" ){
        # do nothing
    }
    elsif ( lc( $hrefLogicalLocationPar->{ 'Type' } ) eq "dir" ){
        #---- Modify directory path
        if ( $hrefLogicalLocationPar->{ 'DirectoryName' } ){
            #---- Fetch output path
            $$srefOutputPathPar = $hrefLogicalLocationPar->{ 'DirectoryName' };

            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Output path: " . $$srefOutputPathPar ); }
        }
        else {
            if ( $bLog == $TRUE ){ $theLogger->logMessage( " Empty logical location path" ); }

            return -1;
        }

        #---- Modify path suffix pattern
        if ( $hrefLogicalLocationPar->{ 'PathSuffixPattern' } ne "" ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Path Suffix Pattern: " . $hrefLogicalLocationPar->{ 'PathSuffixPattern' } ); }

            my $sPSPTmp = lc( $hrefLogicalLocationPar->{ 'PathSuffixPattern' } );
            my $sFormattedPathTmp = $sPSPTmp;

            #---- Find and replace request id pattern
			my $sReqIdWorkingTmp = $sRequestId . "_Working";
            $sFormattedPathTmp =~ s/requestid/$sReqIdWorkingTmp/;

            #---- Find and replace filecount
            # Evaluate current sub directory as per logical location
            my $iCurrentSubDirTmp = 0;
            if ( exists( $hrefLogicalLocationPar->{ 'SubDirCount' } ) ){
                $iCurrentSubDirTmp = $hrefLogicalLocationPar->{ 'SubDirCount' };
                $iCurrentSubDirTmp++;
                $hrefLogicalLocationPar->{ 'SubDirCount' } = $iCurrentSubDirTmp;
            }
            else {
                # First time
                $hrefLogicalLocationPar->{ 'SubDirCount' } = 0;
            }
            my $sFileCountTmp = sprintf( "%06d", $iCurrentSubDirTmp );
            $sFormattedPathTmp =~ s/filecount/$sFileCountTmp/;

            #---- Check and add slash at end
            if (    $$srefOutputPathPar !~ /\\$/
                 && $$srefOutputPathPar !~ /\/$/
               ){
                $$srefOutputPathPar .= '/';
            }
            $$srefOutputPathPar .= $sFormattedPathTmp;

            #----- Fetch path upto request id
            if ( $$srefOutputPathPar =~ /(.*Working)/ ){
                $sDDOPWithReqId = $1;
            }
       }

        #---- Check and create this path
        if ( !(-d $$srefOutputPathPar ) ){
            eval mkpath( $$srefOutputPathPar );
            if ( $@ ){
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Output path ($$srefOutputPathPar) creation failed, reason: $@" ); }

                return ( -2, "Output path ($$srefOutputPathPar) creation failed, reason: $@" );
            }
            else {
                if ( $bLog == $TRUE ){ $theLogger->logMessage( "Output path ($$srefOutputPathPar) created successfully" ); }
            }
        }

        #---- Check and add slash at end
        if ( $$srefOutputPathPar !~ /\\$/
             && $$srefOutputPathPar !~ /\/$/
           ){
            $$srefOutputPathPar .= '/';
        }

        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Final Output path:>" . $$srefOutputPathPar . "<" ); }
    }
    else {
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "" ); }
        #---- Raise error for invalid logical location types
        return ( -3, "Invalid logical location type " . $hrefLogicalLocationPar->{ 'Type' } );
    }

    return 1;
}

#-----------------------------------------------------------------------
# Convert logical location as string
#
# Parameter
# 1. Logical location hash reference
#
#-----------------------------------------------------------------------
sub _logicalLocationToString{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_logicalLocationToString()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Logical location hash reference
    #
    my $hrefLogicalLocationPar = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logHashMessage( $hrefLogicalLocationPar ); }
    my $sLLStringTmp = "";

    #---- Fetch type
    my $sTypeTmp = $hrefLogicalLocationPar->{ 'Type' };
    if ( lc( $sTypeTmp ) eq "dir" ){
        # Type=DIR DirectoryName=X:\Archiver3 PathSuffixPattern=RequestId/FileCount FilenamePattern="%06d_%s",DOCID,TIMESTAMP FileCount=500
        $sLLStringTmp = "Type=DIR";
        if ( $hrefLogicalLocationPar->{ 'DirectoryName' } ne "" ){
            $sLLStringTmp .= " DirectoryName=" . $hrefLogicalLocationPar->{ 'DirectoryName' };
        }
        if ( $hrefLogicalLocationPar->{ 'PathSuffixPattern' } ne "" ){
            $sLLStringTmp .= " PathSuffixPattern=" . $hrefLogicalLocationPar->{ 'PathSuffixPattern' };
        }
        if ( $hrefLogicalLocationPar->{ 'FilenamePattern' } ne "" ){
            $sLLStringTmp .= " FilenamePattern=" . $hrefLogicalLocationPar->{ 'FilenamePattern' };
        }
        if ( $hrefLogicalLocationPar->{ 'FileCount' } ne "" ){
            $sLLStringTmp .= " FileCount=" . $hrefLogicalLocationPar->{ 'FileCount' };
        }
    }
    elsif ( lc( $sTypeTmp ) eq "dummy" ){
        # Type=DUMMY
        $sLLStringTmp = "Type=DUMMY";
    }
	#if ( $bLog == $TRUE ){ $theLogger->logMessage( "Logical location:>" . $sLLStringTmp . "<" ); }

    return $sLLStringTmp;
}

#-----------------------------------------------------------------------
# Fetch document index
#
# Parameter
# 1. Index name
#
# Returns index value if index found at document else empty string
# 
#-----------------------------------------------------------------------
sub _fetchDocumentIndex{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_fetchDocumentIndex()" ); }

    #---- Fetch parameter
    #
    # 1. Index name
    #
    my $sIndexNamePar = shift;

    #---- Fetch first document index
    my $sIndexNameTmp  = "";
    my $sIndexValueTmp = "";

    my $sSearchIndexTmp = lc( $sIndexNamePar );

    my $a2wIndexTmp = $a2wDocumentPar->getFirstIndex();

    while ( $a2wIndexTmp != 0 ){
        #---- Fetch index name
        $sIndexNameTmp = $a2wIndexTmp->getName();

        #---- Check for given index name
        if ( lc( $sIndexNameTmp ) eq $sSearchIndexTmp ){
           $sIndexValueTmp = $a2wIndexTmp->getValue();

           if ( $bLog == $TRUE ){ $theLogger->logMessage( "Found index:" . $sIndexNamePar . "=>" . $sIndexValueTmp . "<" ); }

           last; # leave while loop
        }

        #---- Fetch next PagePageGroupIndex item
        $a2wIndexTmp = $a2wDocumentPar->getNextIndex();
    }

    return $sIndexValueTmp;
}

#-----------------------------------------------------------------------
# Fetch output filename based on pattern
#
# Parameter
# 1. Logical location hash reference
# 2. Output filename
# 3. Document id
# 4. Simple document name
#
#-----------------------------------------------------------------------
sub _fetchPatternFormattedFilename{
    #---- Fetch parameter(s)
    #
    # 1. Logical location hash reference
    # 2. Output filename
    # 3. Document id
    # 4. Simple document name
    #
    my $hrefLogicalLocationPar = shift;
    my $sFQOutputFilenamePar = shift;
    my $iDocIdPar = shift;
    my $sSimpleDocNamePar = shift;    # $V110 Change

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_fetchPatternFormattedFilename( $sFQOutputFilenamePar . ", " . $iDocIdPar . ", " . $sSimpleDocNamePar )" ); }

    # Formatted filename
    my $sFFNTmp = undef;

    #---- Fetch type
    my $sTypeTmp = $hrefLogicalLocationPar->{ 'Type' };
    if ( lc( $sTypeTmp ) eq "dir" ){
        #---- Fetch output filename pattern
        my $sFilenamePatternTmp = $hrefLogicalLocationPar->{ 'FilenamePattern' };
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "FilenamePattern: " . $sFilenamePatternTmp ); }
    
        if ( $sFilenamePatternTmp ne "" ){
            my @FilenamePatternTmp = split( ',', $sFilenamePatternTmp  );

            my $sFilenameFormatTmp = @FilenamePatternTmp[ 0 ];
            $sFilenameFormatTmp =~ s/^\"//g;    # Remove leading double quotes
            $sFilenameFormatTmp =~ s/\"$//g;    # Remove trailing double quotes
            my $sFormattedFilenameTmp = $sFilenameFormatTmp;

            for ( my $ j = 1; $j < @FilenamePatternTmp; $j++ ){
                #---- Generate MD5 for output document data
                if ( lc( @FilenamePatternTmp[ $j ] ) eq "md5id" ){
                    if ( !( -e $sFQOutputFilenamePar ) ){
                        if ( $bLog == $TRUE ){ $theLogger->logMessage( $sFQOutputFilenamePar . " file doesn't exist to generate MD5 id" ); }

                        return undef;
                    }

                    my $sMD5IdTmp = _generateMD5Id( $sFQOutputFilenamePar );
                    if ( $sMD5IdTmp eq "" ){
                        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Unable to generate MD5 id for " . $sFQOutputFilenamePar ); }

                        return undef;
                    }

                    #---- Fill in MD5 id on filename
                    $sFormattedFilenameTmp =~ s/\%s/$sMD5IdTmp/;

                    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Filename after applying MD5ID: " . $sFormattedFilenameTmp ); }
                }

                #---- Generate current time stamp
                if ( lc( @FilenamePatternTmp[ $j ] ) eq "timestamp" ){
                    # (sec,min,hour,mday,mon,year,wday,yday,isdst) = localtime( time );
                    my @CurrentTimeTmp = localtime( time );
                    my $iYearTmp = @CurrentTimeTmp[ 5 ] + 1900;
                    my $sObjIdTimeStampTmp = sprintf( "%02d%02d%02d%02d%02d%02d",
                                                      ( $iYearTmp % 100 ),
                                                      ( @CurrentTimeTmp[ 4 ] + 1 ),
                                                      @CurrentTimeTmp[ 3 ],
                                                      @CurrentTimeTmp[ 2 ],
                                                      @CurrentTimeTmp[ 1 ],
                                                      @CurrentTimeTmp[ 0 ]
                                                    );
                    #---- Fill in timestamp on filename
                    $sFormattedFilenameTmp =~ s/\%s/$sObjIdTimeStampTmp/;

                    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Filename after applying timestamp: " . $sFormattedFilenameTmp ); }
                }

                #---- Generate document id
                if ( lc( @FilenamePatternTmp[ $j ] ) eq "docid" ){
                    my $sDocIdTmp = $iDocIdPar;
                    if ( $sFormattedFilenameTmp =~ /(\%.*d)/ ){
                        my $sDocIdFormatTmp = $1;    # Format as given in filename pattern
                        $sDocIdTmp = sprintf( $sDocIdFormatTmp, $iDocIdPar );
                    }

                    #---- Fill in document id
                    $sFormattedFilenameTmp =~ s/\%.*d/$sDocIdTmp/;

                    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Filename after applying document id: " . $sFormattedFilenameTmp ); }
                }

                #---- Generate spoolname
                if ( lc( @FilenamePatternTmp[ $j ] ) eq "spoolname" ){
                    #---- Fill in spool name
                    $sFormattedFilenameTmp =~ s/\%s/$sSimpleDocNamePar/;

                    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Filename after applying spool name: " . $sFormattedFilenameTmp ); }
                }

                $sFFNTmp = $sFormattedFilenameTmp;
            }    # for ( my $ j = 1; $j < @FilenamePatternTmp; $j++ )
        }    # if ( $sFilenamePatternTmp ne "" ){
        else {
            $sFFNTmp = $sSimpleDocNamePar;
            if ( $sFQOutputFilenamePar =~ /.*[\\\/](.*)$/ ){
			    $sFFNTmp = $1;
			    $sFFNTmp =~ s/(\..{3})$//; # remove extension
			}
        }
    }
    #---- Other than DIR type are not handled, so return undef
    #else{
    #}

    return $sFFNTmp;
}

#-----------------------------------------------------------------------
# Generate MD5 id
#
# Parameter(s)
# 1. Output filename
#
#-----------------------------------------------------------------------
sub _generateMD5Id{
    #---- Fetch parameter(s)
    #
    # 1. Output filename (string)
    #
    my $sOutputFilenamePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_generateMD5Id( $sOutputFilenamePar )" ); }

    #---- Assert parameter
    if ( $sOutputFilenamePar eq "" ){
        return "";
    }

    #---- Fetch file size
    my $iFileSizeTmp = -s $sOutputFilenamePar;
    if ( $iFileSizeTmp <= 0 ){
        return "";
    }

    #---- Open file
    my $iRetTmp = open( MD5CONTENT, "<" . $sOutputFilenamePar );
    if ( $iRetTmp == 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Failed to open ( " . $sOutputFilenamePar . " )" ); }
        return "";
    }

    #---- Read whole data in binary mode
    binmode( MD5CONTENT );
    my $byFileContentTmp = undef;
    my $iReadCountTmp = sysread( MD5CONTENT, $byFileContentTmp, $iFileSizeTmp );

    #---- Close file
    close( MD5CONTENT );

    #---- Assert data reading
    if ( $iReadCountTmp != $iFileSizeTmp ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Failed to read ( " . $sOutputFilenamePar . " )" ); }
        return "";
    }

    #---- Create MD5 context
    if ( $md5Context == undef ){
        $md5Context = Digest::MD5->new();
    }
    else {
        $md5Context->reset();
    }

    #---- Add data to MD5 context
    $md5Context->add( $byFileContentTmp );

    #---- Fetch MD5 id
    my $sContentMD5IdTmp = uc( $md5Context->hexdigest() );

    return $sContentMD5IdTmp;
}

#-----------------------------------------------------------------------
# Write store process request
#
# Parameter(s)
# 1. Output filename
# 2. Document id
# 3. Object id
# 4. Page count
# 5. Page from
# 6. Page to
#
#-----------------------------------------------------------------------
sub _writeStoreProcessRequest{
    #---- Fetch parameter(s)
    #
    # 1. Output filename (string)
    # 2. Document id (int)
    # 3. Object id (string)
    # 3. Page count (int)
    # 4. Page from (int)
    # 5. Page to (int)
    #
    my $sOutputFilenamePar = shift;
    my $iDocumentIdPar     = shift;
    my $sObjectIdPar       = shift;
    my $iPageCountPar      = shift;
    my $iPageFromPar       = shift;
    my $iPageToPar         = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_writeStoreProcessRequest( $sOutputFilenamePar . ", " . $iDocumentIdPar . ", " . $sObjectIdPar . ", " . $iPageCountPar . ", " . $iPageFromPar . ", " . $iPageToPar )" ); }

    #---- Assert parameter
    if ( $sOutputFilenamePar eq "" ){
        return -1;
    }

    # Create new store process
    my $reqStoreProcessTmp = new a2w::request::StoreProcess( $bLog );
    if ( $reqStoreProcessTmp == undef ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Unable to allocate store process request" ); }
        return -2;
    }

    # Set filename
    $reqStoreProcessTmp->setFilename( $sOutputFilenamePar );

    # Set session context
    $reqStoreProcessTmp->setSessionContext( sprintf( "%X", $$ ), "1" );

    # Set request id
    # ( Seconds, Micro-seconds ) = Time::HiRes::gettimeofday;
    my ( $tCurrentSecTmp, $tCurrentMicroSecTmp ) = Time::HiRes::gettimeofday;
    my $tCurrentMilliSecondsTmp = ( $tCurrentSecTmp * 1000 )            # Convert seconds to milli-seconds
                                  + ( $tCurrentMicroSecTmp / 1000 );    # Convert micro-seconds to milli-seconds
    my $sReqIdTmp = sprintf( "%s", $tCurrentMilliSecondsTmp );          # %s format is mandatory, %d is not working for this high value
    $sReqIdTmp =~ s/\.(.*)//g;                                          # Remove trailing floating digits
    $reqStoreProcessTmp->setRequestId( $sReqIdTmp );

    # Set transaction
    #---- Fetch transaction source id
    #
    #     Value fetched in below given priority
    #
    #     1. TransactionSourceID element's value AS IS from "processSpool" request
    #     2. A constant value of "3"
    #
    my $sTrnSrcIdTmp = $reqProcessSpool->getTransactionSourceID();
    $sTrnSrcIdTmp =~ s/^\s+//g;   # Strip leading spaces
    $sTrnSrcIdTmp =~ s/\s+$//g;   # Strip trailing spaces

    #---- Assert transaction source id value and use constant "3" if not specified
    if ( $sTrnSrcIdTmp eq "" ){
        $sTrnSrcIdTmp = "3";
    }
    if ( $iDocumentIdPar == 1 ){
        $reqStoreProcessTmp->setTransaction( $sRequestId
                                             , $sTrnSrcIdTmp
                                             , 1  # Transaction begin case
                                           );
    }
    else {
        $reqStoreProcessTmp->setTransaction( $sRequestId
                                             , $sTrnSrcIdTmp
                                             #, 1  # Transaction begin case
                                             #, 2  # Transaction end case
                                           );
    }

    # Set document consignment number
    my $sDCNTmp = sprintf( "%08d", $iDocumentIdPar );
    $reqStoreProcessTmp->setDocumentConsignmentNumber( $sDCNTmp );

    # Set skip process
    $reqStoreProcessTmp->setSkipProcess( "false" );

    # Fetch document indexes
    my $DocIndexListTmp = _fetchDocumentIndexes();

    # Set process type xml ref
    my $sProcessTypeXmlRefTmp = "";

    # Value of "I_PROCESS_TYPE_XML_REF" index is used as ProcessTypeXmlRef, if this
    # index doesn't exist, then "ProcessTypeXmlRef" value from ProcessSpool request
    # is used
    if ( exists $DocIndexListTmp->{ "I_PROCESS_TYPE_XML_REF" } ){
        $sProcessTypeXmlRefTmp = $DocIndexListTmp->{ "I_PROCESS_TYPE_XML_REF" };

        # Delete key from hash since it is not needed any more
        delete $DocIndexListTmp->{ "I_PROCESS_TYPE_XML_REF" };
    }
    elsif ( exists $DocIndexListTmp->{ "I_PROTYPEXMLREF" } ){
        $sProcessTypeXmlRefTmp = $DocIndexListTmp->{ "I_PROTYPEXMLREF" };

        # Delete key from hash since it is not needed any more
        delete $DocIndexListTmp->{ "I_PROTYPEXMLREF" };
    }
    else {
        $sProcessTypeXmlRefTmp = $reqProcessSpool->getProcessTypeXmlRef();
    }
    if ( $sProcessTypeXmlRefTmp eq "" ){
        $sProcessTypeXmlRefTmp = " ";
    }
    $reqStoreProcessTmp->setProcessTypeXmlRef( $sProcessTypeXmlRefTmp );

    # Set process type id
    my $sProcessTypeIdTmp = "";

    # Value of "I_PROCESS_TYPE_ID" index is used as ProcessTypeID, if this
    # index doesn't exist, then "ProcessTypeID" value from ProcessSpool request
    # is used
    if ( exists $DocIndexListTmp->{ "I_PROCESS_TYPE_ID" } ){
        $sProcessTypeIdTmp = $DocIndexListTmp->{ "I_PROCESS_TYPE_ID" };

        # Delete key from hash since it is not needed any more
        delete $DocIndexListTmp->{ "I_PROCESS_TYPE_ID" };
    }
    elsif ( exists $DocIndexListTmp->{ "I_PROTYPEID" } ){
        $sProcessTypeIdTmp = $DocIndexListTmp->{ "I_PROTYPEID" };

        # Delete key from hash since it is not needed any more
        delete $DocIndexListTmp->{ "I_PROTYPEID" };
    }
    else {
        $sProcessTypeIdTmp = $reqProcessSpool->getProcessTypeID();
    }
    if ( $sProcessTypeIdTmp eq "" ){
        $sProcessTypeIdTmp = " ";
    }
    $reqStoreProcessTmp->setProcessTypeID( $sProcessTypeIdTmp );

    # Set BO id
    my $sBOIDTmp = "";
    
    # Value of "I_VNR" index is used as BoID
    if( exists $DocIndexListTmp->{ "I_VNR" } ){
        $sBOIDTmp = $DocIndexListTmp->{ "I_VNR" };

        # Delete key from hash since it is not needed any more
        delete $DocIndexListTmp->{ "I_VNR"};
    }

    if ( $sBOIDTmp eq "" ){
        $sBOIDTmp = " ";
    }
    $reqStoreProcessTmp->setBoID( $sBOIDTmp );

    # Set target work basket id
    my $sTWBIdTmp = $reqProcessSpool->getWorkBasketId();
    if ( $sTWBIdTmp eq "" ){
        $sTWBIdTmp = " ";
    }
    $reqStoreProcessTmp->setTargetWorkBasketID( $reqProcessSpool->getWorkBasketId() );

    # Set priority
    $reqStoreProcessTmp->setPriority( "4" );

    # Set section id
    $reqStoreProcessTmp->setSectionID( "1" );

    # Set in out date
    my $sInOutDateTmp = "";
    
    # Value of " I_DATUM" is used as InOutDate, if this index doesn't
    # exist, a constant value "0001-01-01" us used
    my $sIDatumTmp = "";
    if ( exists $DocIndexListTmp->{ "I_DATUM"} ){
        $sIDatumTmp = $DocIndexListTmp->{ "I_DATUM" };
        $sInOutDataTmp = $sIDatumTmp;

        # Delete key from hash since it is not needed any more
        delete $DocIndexListTmp->{ "I_DATUM" };        
    }
    if ( $sInOutDataTmp eq "" ) {
        $sInOutDataTmp = "0001-01-01";
    }
    $reqStoreProcessTmp->setInOutDate( $sInOutDataTmp );

    # Fetch index alias list from Process Spool
    #
    # Here we fetch reference of alias and modify it which makes
    # entries to be lost for next document iteration, in order to
    # avoid it, we should duplicate the index alias list
    #
    my $hrefIndexAliasListTmp = dclone( $reqProcessSpool->getIndexAliasList() );
    my $arefIndexAliasListTmp = $hrefIndexAliasListTmp->{ 'IndexAlias' };
    my @arrIndexAliasListTmp = @{ $arefIndexAliasListTmp };

    # Declare and initialize index variables
    my $sIdxNameTmp      = "";
    my $sIdxAliasNameTmp = "";
    my $sIdxLevelTmp     = "";
    my $sIdxValueTmp     = "";

    # Loop through index alias list and add Process Attribute Elements
    my $iArrIdxTmp = 0;
    foreach $idxTmp ( @arrIndexAliasListTmp ){
        $sIdxLevelTmp = $idxTmp->{ 'IndexLevel' };

        #
        # Fixed minor bug in logging index name values while preparing
        # Additional document and process attribute list for StoreProcess
        #
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $idxTmp->{ 'IndexName' } . "=" . $idxTmp->{ 'AliasName' } . ", " . $sIdxLevelTmp ); }

        if ( lc( $sIdxLevelTmp ) eq "p" ){
            $sIdxNameTmp      = $idxTmp->{ 'IndexName' };
            $sIdxAliasNameTmp = $idxTmp->{ 'AliasName' };
            $sIdxValueTmp     = $DocIndexListTmp->{ $sIdxNameTmp };

            if ( $sIdxValueTmp ne "" ){
                $reqStoreProcessTmp->addAdditionalProcessAttributeElement( $sIdxAliasNameTmp, $sIdxValueTmp );
            }

            # Delete index alias from index alias list
            delete $arefIndexAliasListTmp[ $iArrIdxTmp ];
            delete $DocIndexListTmp->{ $sIdxNameTmp };
        }

        $iArrIdxTmp++;
    }

    # Set document count
    $reqStoreProcessTmp->setDocumentCount( "1" );

    # Set document way
    $reqStoreProcessTmp->setDocumentWay( "2" );

    # Set document type xml ref
    my $sDocTypeXmlRefTmp = "";

    # Value of "I_DOCUMENTTYPEXMLREF" index is used as DocumentTypeXmlRef, if this
    # index doesn't exist, then "DocumentTypeXmlRef" value from ProcessSpool request
    # is used
    if ( exists $DocIndexListTmp->{ "I_DOCUMENTTYPEXMLREF" } ){
        $sDocTypeXmlRefTmp = $DocIndexListTmp->{ "I_DOCUMENTTYPEXMLREF" };

        # Delete key from hash since it is not needed any more
        delete $DocIndexListTmp->{ "I_DOCUMENTTYPEXMLREF" };
    }
    elsif ( exists $DocIndexListTmp->{ "I_DOCTYPEXMLREF" } ){
        $sDocTypeXmlRefTmp = $DocIndexListTmp->{ "I_DOCTYPEXMLREF" };

        # Delete key from hash since it is not needed any more
        delete $DocIndexListTmp->{ "I_DOCTYPEXMLREF" };
    }
    else {
        $sDocTypeXmlRefTmp = $reqProcessSpool->getDocumentTypeXmlRef();
    }
    if ( $sDocTypeXmlRefTmp eq "" ){
        $sDocTypeXmlRefTmp = " ";
    }
    $reqStoreProcessTmp->setDocumentTypeXmlRef( $sDocTypeXmlRefTmp );

    # Set document type id
    my $sDocTypeIdTmp = "";

    # Value of "I_DOCUMENTTYPEID" index is used as DocumentTypeID, if this
    # index doesn't exist, then "DocumentTypeID" value from ProcessSpool request
    # is used
    if ( exists $DocIndexListTmp->{ "I_DOCUMENTTYPEID" } ){
        $sDocTypeIdTmp = $DocIndexListTmp->{ "I_DOCUMENTTYPEID" };

        # Delete key from hash since it is not needed any more
        delete $DocIndexListTmp->{ "I_DOCUMENTTYPEID" };
    }
    elsif ( exists $DocIndexListTmp->{ "I_DOCTYPEID" } ){
        $sDocTypeIdTmp = $DocIndexListTmp->{ "I_DOCTYPEID" };

        # Delete key from hash since it is not needed any more
        delete $DocIndexListTmp->{ "I_DOCTYPEID" };
    }
    else {
        $sDocTypeIdTmp = $reqProcessSpool->getDocumentTypeID();
    }
    if ( $sDocTypeIdTmp eq "" ){
        $sDocTypeIdTmp = " ";
    }
    $reqStoreProcessTmp->setDocumentTypeID( $sDocTypeIdTmp );

    # Set document content date
    # Value of " I_DATUM" index is used
    my $sDocumentContentDateTmp = $sIDatumTmp;
    if( $sDocumentContentDateTmp eq "" ){
        $sDocumentContentDateTmp = " ";
    }
    $reqStoreProcessTmp->setDocumentContentDate( $sDocumentContentDateTmp );

    # Set document subject
    if ( exists $DocIndexListTmp->{ "I_BETREFF" } ){
        my $sDocSubjectTmp = $DocIndexListTmp->{ "I_BETREFF" };

        # Delete key from hash since it is not needed any more
        delete $DocIndexListTmp->{ "I_BETREFF" };

        $reqStoreProcessTmp->setDocumentSubject( $sDocSubjectTmp );
    }
    
    # Set document source xml ref
    my $sDocSrcXMLRefTmp = $reqProcessSpool->getDocumentSourceXMLRef();
    if ( $sDocSrcXMLRefTmp eq "" ){
        $sDocSrcXMLRefTmp = " ";
    }
    $reqStoreProcessTmp->setDocumentSourceXmlRef( $sDocSrcXMLRefTmp );

    # Set document source id
    my $sDocSrcIdTmp = $reqProcessSpool->getDocumentSourceID();
    if ( $sDocSrcIdTmp eq "" ){
        $sDocSrcIdTmp = " ";
    }
    $reqStoreProcessTmp->setDocumentSourceID( $sDocSrcIdTmp );

    # Set page count
    $reqStoreProcessTmp->setPageCount( $iPageCountPar );

    # Set preferred object file format
    $reqStoreProcessTmp->setPreferredObjectFileFormat( $sDocType );

    # Set object id
    # <MD5 Id of output document data><Current time in YYMMDDhhmmss format>0008
    #
    $reqStoreProcessTmp->setObjectID( $sObjectIdPar );

    # Set object representation format
    $reqStoreProcessTmp->setObjectRepresentationFormat( $sDocType );

    # Set original format
    $reqStoreProcessTmp->setOriginalFormat( "true" );

    # Set sequence number
    $reqStoreProcessTmp->setSequenceNumber( "1" );

    # Loop through index alias list and add Indexes with level "D" to document attribute elements
    $iArrIdxTmp       = 0;
    $sIdxNameTmp      = "";
    $sIdxAliasNameTmp = "";
    $sIdxLevelTmp     = "";
    $sIdxValueTmp     = "";

    @arrIndexAliasListTmp = @{ $arefIndexAliasListTmp };
    foreach $idxTmp ( @arrIndexAliasListTmp ){
        $sIdxLevelTmp = $idxTmp->{ 'IndexLevel' };

        # Fixed minor bug in logging index name values while preparing
        # Additional document and process attribute list for StoreProcess
        #
        if ( $bLog == $TRUE ){ $theLogger->logMessage( $idxTmp->{ 'IndexName' } . "=" . $idxTmp->{ 'AliasName' } . ", " . $sIdxLevelTmp ); }

        if ( lc( $sIdxLevelTmp ) eq "d" ){
            $sIdxNameTmp      = $idxTmp->{ 'IndexName' };
            $sIdxAliasNameTmp = $idxTmp->{ 'AliasName' };
            $sIdxValueTmp     = $DocIndexListTmp->{ $sIdxNameTmp };

            # Add additional document attribute element
            if ( $sIdxValueTmp ne "" ){
                $reqStoreProcessTmp->addAdditionalDocumentAttributeElement( $sIdxAliasNameTmp, $sIdxValueTmp );
            }

            #Delete indexalias from indexalias list
            delete $arefIndexAliasListTmp[ $iArrIdxTmp ];
            delete $DocIndexListTmp->{ $sIdxNameTmp };
        }

        $iArrIdxTmp++;
    }    

    # Loop through rest of indexes and add them to document attribute elements
    foreach $sIdxNameTmp ( keys %{ $DocIndexListTmp } ){
        if ( $sIdxNameTmp ne "" ){
            $sIdxValueTmp = $DocIndexListTmp->{ $sIdxNameTmp };

            # Add additional document attribute element
            $reqStoreProcessTmp->addAdditionalDocumentAttributeElement( $sIdxNameTmp, $sIdxValueTmp );
        }
    }

    # Set spool filename
    $reqStoreProcessTmp->setSpoolFileName( $sSpoolFilename );

    # Set resource filename
    $reqStoreProcessTmp->setResourceFileName( $sRequestFilename );

    # Set page from
    $reqStoreProcessTmp->setPageFrom( $iPageFromPar );

    # Set page to
    $reqStoreProcessTmp->setPageTo( $iPageToPar );

    #---- Serialize store process request ----#
    $reqStoreProcessTmp->serialize();

    # Reset request reference
    $reqStoreProcessTmp = undef;

    return 1;
}

#-----------------------------------------------------------------------
# Modifies last written Store Process request for TransactionEnd element
#
# a. StoreProcess->ImportProcess->Transaction->TransactionEnd element must
#    be set with a value of "true" for last document of spool but AFP2web
#    doesn't knew the last document until spool parsing is over, so SF
#    will in general write the Store Process for all documents without
#    "TransactionEnd" element and later in "finalize" last written Store
#    Process request will be updated with "TransactionEnd" element
#
# b. StoreProcess->ImportProcess->ProcessCount element must be set with
#    value of "output document count" for last document of spool but AFP2web
#    doesn't knew the last document until spool parsing is over, so SF
#    will in general write the Store Process for all documents without
#    "ProcessCount" element and later in "finalize" last written Store
#    Process request will be updated with "ProcessCount" element
#
# Parameter
# 1. Store process request filename
#-----------------------------------------------------------------------
sub _modifyLastDocumentSP{
    #---- Fetch parameter
    #
    # 1. Store process request filename (string)
    #
    my $sSPFilenamePar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_modifyLastDocumentSP( $sSPFilenamePar )" ); }

    #---- Open store process request file
    my $iRetTmp = open( SPREQ, "+<" . $sSPFilenamePar );
    if ( $iRetTmp == 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Failed to open ( " . $sSPFilenamePar . " )" ); }

        return -1;
    }

    #---- Fetch content
    my @SPContentTmp = <SPREQ>;

    #---- Seek file position to beginning
    seek( SPREQ,    # File handle
          0,        # Offset
          0         # Whence, 0 for file start, 1 for current position & 2 for file end
        );

    #---- Transaction source id element's indent level
    my $sTSIdIndentTmp = "";

    #---- ProcessCount element's indent level
    my $sProcessCountIndentTmp = "";

    #---- Loop through content and add transactionend element
    for ( my $h = 0; $h < @SPContentTmp; $h++ ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "$h: " . @SPContentTmp[ $h ] ); }

        #---- Check for TransactionSourceID element and fetch it's indent level
        if ( @SPContentTmp[ $h ] =~ /(\s*)\<TransactionSourceID\>/ ){
            $sTSIdIndentTmp = $1;
        }

        #---- Add <TransactionEnd> element before <\/Transaction>
        if ( @SPContentTmp[ $h ] =~ /<\/Transaction>/ ){
            #---- Use actual indent level
            $sTELineTmp = $sTSIdIndentTmp . "<TransactionEnd>true<\/TransactionEnd>\n";
            print SPREQ ( $sTELineTmp );

            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Added TransactionEnd element: " . $sTELineTmp ); }
        }

        #---- Check for PrintSpoolInfo element and fetch it's indent level
        if ( @SPContentTmp[ $h ] =~ /(\s*)\<PrintSpoolInfo\>/ ){
            $sProcessCountIndentTmp = $1;
        }

        #---- Add <ProcessCount> element before <\/ImportProcess>
        if ( @SPContentTmp[ $h ] =~ /<\/ImportProcess>/ ){
            #---- Use actual indent level
            $sTELineTmp = $sProcessCountIndentTmp . "<ProcessCount>" . $iDocCount . "<\/ProcessCount>\n";
            print SPREQ ( $sTELineTmp );

            if ( $bLog == $TRUE ){ $theLogger->logMessage( "Added ProcessCount element: " . $sTELineTmp ); }
        }

        print SPREQ ( @SPContentTmp[ $h ] );
    }

    #---- Close store process request file
    close( SPREQ );

    return 1;
}
#-----------------------------------------------------------------------
# Open protocol file and write header
#-----------------------------------------------------------------------
sub _openProtocolAndWriteHeader{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_openProtocolAndWriteHeader()" ); }

    #---- Build protocol filename
    my $sProtocolFilenameTmp = $sOutputFilePath
                               . "2_"
                               . $sRequestId
                               . "_protocol_"
                               . $sStartTimeStamp
                               . ".txt";

    #---- Open protocol file
    my $iRetTmp = open( PROTOCOL, ">" . $sProtocolFilenameTmp );
    if ( $iRetTmp == 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Failed to open ( " . $sProtocolFilenameTmp . " )" ); }
        return -1;
    }
    $sProtocolFilename = $sProtocolFilenameTmp;

    #---- Write header
    # Running AFP2web Server Version 1.0 r30 [Compiled on Jun 15 2005 at 11:48:27] at 06:56:32 on 12 September 2007 using : -q processSpoolRequest 
    #
    # Process Spool Request Id: REQ_VERAB_001
    # Processing Spool File: VERAB.AFP, Resource File VERABR.res
    #
    my @MonthsNameTmp = (
                          "January",
                          "February",
                          "March",
                          "April",
                          "May",
                          "June",
                          "July",
                          "August",
                          "September",
                          "October",
                          "November",
                          "December"
                        );
    printf PROTOCOL ( "Running %s at %02d:%02d:%02d on %d %s %04d\n\n",
                      $sA2WVersion,
                      @StartTime[ 2 ],
                      @StartTime[ 1 ],
                      @StartTime[ 0 ],
                      @StartTime[ 3 ],
                      @MonthsNameTmp[ @StartTime[ 4 ] ],
                      ( @StartTime[ 5 ] + 1900 )
                    );
    printf PROTOCOL ( "Process Spool Request Id: %s\n",
                      $sRequestId
                    );
    printf PROTOCOL ( "Processing Spool File: %s, Resource File: %s\n\n",
                      $sSpoolFilename,
                      $sResourceFilename
                    );

    return 1;
}

#-----------------------------------------------------------------------
# Write protocol document entry
#
# E.g.
# [14:19:13] Processing Document Id : 1, at 0000000300, 4FABBE52D6850466F46D22F753BB4DB30709181419130008 : Ok
#
#
# Parameter(s)
# 1. Document id
# 2. Output filename
# 3. Status
#
#-----------------------------------------------------------------------
sub _writeProtocolDocumentEntry{
    #---- Fetch parameter(s)
    #
    # 1. Document id (int)
    # 2. Output filename (string)
    # 3. Status (int)
    #
    my $iDocIdPar = shift;
    my $sOutputFilenamePar = shift;
    my $iStatusPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_writeProtocolDocumentEntry( $iDocIdPar . ", " . $sOutputFilenamePar . ", " . $iStatusPar )" ); }

    #---- Build status string
    my $sStatusTmp = "Error";
    if ( $iStatusPar > 0 ){
        $sStatusTmp = "Ok";
    }

    #---- Write document entry
    # AFP2web Server Version 1.0 r30 [Compiled on Jun 15 2005 at 11:48:27] ended at 10:37:31 on 24 July 2008, Elapsed Time 00:00:00
    #
    # (sec,min,hour,mday,mon,year,wday,yday,isdst) = localtime( time );
    my @CurrentTimeTmp = localtime( time );
    printf PROTOCOL ( "[%02d:%02d:%02d] Processing Document Id : %d --> %s: %s\n",
                      @CurrentTimeTmp[ 2 ],
                      @CurrentTimeTmp[ 1 ],
                      @CurrentTimeTmp[ 0 ],
                      $iDocIdPar,
                      $sOutputFilenamePar,
                      $sStatusTmp
                    );
}

#-----------------------------------------------------------------------
# Write footer and close protocol
#-----------------------------------------------------------------------
sub _writeFooterAndCloseProtocol{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_writeFooterAndCloseProtocol()" ); }

    #---- Write footer
    # AFP2web Server Version 1.0 r30 [Compiled on Jun 15 2005 at 11:48:27] ended at 10:37:31 on 24 July 2008, Elapsed Time 00:00:00
    #
    my @MonthsNameTmp = (
                          "January",
                          "February",
                          "March",
                          "April",
                          "May",
                          "June",
                          "July",
                          "August",
                          "September",
                          "October",
                          "November",
                          "December"
                        );

    # (sec,min,hour,mday,mon,year,wday,yday,isdst) = localtime( time );
    my @EndTimeTmp = localtime( $tEnd );
    my $tElapsedTmp = $tEnd - $tStart;
    my @ElapsedTimeTmp = gmtime( $tElapsedTmp );

    printf PROTOCOL ( "\n%s ended at %02d:%02d:%02d on %d %s %04d, Elapsed Time %02d:%02d:%02d\n",
                      $sA2WVersion,
                      @EndTimeTmp[ 2 ],
                      @EndTimeTmp[ 1 ],
                      @EndTimeTmp[ 0 ],
                      @EndTimeTmp[ 3 ],
                      @MonthsNameTmp[ @EndTimeTmp[ 4 ] ],
                      ( @EndTimeTmp[ 5 ] + 1900 ),
                      @ElapsedTimeTmp[ 2 ],
                      @ElapsedTimeTmp[ 1 ],
                      @ElapsedTimeTmp[ 0 ]
                    );

    #---- Close protocol file
    close( PROTOCOL );
}

#-----------------------------------------------------------------------
# Write statistics file
#-----------------------------------------------------------------------
sub _writeStatisticsFile{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_writeStatisticsFile()" ); }

    #---- Build statistics filename
    my $sStatisticsFilenameTmp = $sOutputFilePath
                                 . "1_"
                                 . $sRequestId
                                 . "_stat_"
                                 . $sStartTimeStamp
                                 . ".txt";

    #---- Open statistics file
    my $iRetTmp = open( STAT, ">" . $sStatisticsFilenameTmp );
    if ( $iRetTmp == 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Failed to open ( " . $sStatisticsFilenameTmp . " )" ); }
        return -1;
    }
    $sStatisticFilename = $sStatisticsFilenameTmp;

    #---- Write AFP2web version
    # Version:                     AFP2web Server Version 1.0 r30 [Compiled on Jun 15 2005 at 11:48:27]
    #
    printf STAT ( "%-30s%s\n\n",
                  "Version:",
                  $sA2WVersion
                );

    #---- Write copyright
    # Copyright:                   Copyright (c) 1998-2019 Maas Holding GmbH (http://www.maas.de), All rights reserved.
    #                                     For conditions of use, license, and distribution, see LICENSE.txt.
    #                                     http://www.afp2web.de/ Support: support@maas.de
    #
    printf STAT ( "%-30s%s\n",
                  "Copyright:",
                  "Copyright (c) 1998-2019 Maas Holding GmbH (http://www.maas.de), All rights reserved."
                );
    printf STAT ( "%-30s%s\n",
                  "",
                  "For conditions of use, license, and distribution, see LICENSE.txt."
                );
    printf STAT ( "%-30s%s\n\n",
                  "",
                  "http://www.afp2web.de/ Support: support\@maas.de"
                );

    #---- Write start time
    # Started at:                  hh:mm:ss
    #
    printf STAT ( "%-30s%02d:%02d:%02d\n\n",
                  "Started at:",
                  @StartTime[ 2 ],
                  @StartTime[ 1 ],
                  @StartTime[ 0 ]
                );

    #---- Write end time
    # Ended at:                  hh:mm:ss
    #
    # (sec,min,hour,mday,mon,year,wday,yday,isdst) = localtime( time );
    my @EndTimeTmp = localtime( $tEnd );
    printf STAT ( "%-30s%02d:%02d:%02d\n\n",
                  "Ended at:",
                  @EndTimeTmp[ 2 ],
                  @EndTimeTmp[ 1 ],
                  @EndTimeTmp[ 0 ]
                );

    #---- Write elapsed time
    # Elapsed time:                  hh:mm:ss
    #
    # (sec,min,hour,mday,mon,year,wday,yday,isdst) = localtime( time );
    my $tElapsedTmp = $tEnd - $tStart;
    my @ElapsedTimeTmp = gmtime( $tElapsedTmp );
    printf STAT ( "%-30s%02d:%02d:%02d\n\n",
                  "Elapsed time:",
                  @ElapsedTimeTmp[ 2 ],
                  @ElapsedTimeTmp[ 1 ],
                  @ElapsedTimeTmp[ 0 ]
                );

    #---- Write list of parameters
    # List of Parameters:                  ???
    #
    printf STAT ( "%-30s%s\n\n",
                  "List of Parameters:",
                  $a2wConfigPar->getScriptArgs()
                );

    #---- Write process spool request info
    # Process Spool Request Info:  Request Id: REQ_VERAB_001
    #
    #                              Spool File Details:     Logical Location=HNRES Name=VERAB.AFP DocumentType=AFP 
    #                              Resource File Details:  Logical Location=HNRES Name=VERABR.res
    #                              DocumentData:           Location=HNOUT
    #                              StoreDocumentRequest:   Location=StoreProcess
    #
    printf STAT ( "%-30s%s\n\n",
                  "Process Spool Request Info:",
                  "Request Id: " . $sRequestId
                );
    printf STAT ( "%-30s%-24sName=%s",
                  "",
                  "Spool File Details:",
                  $sSpoolFilename
                );
    if ( $sResourceFilename ne "" ){
        printf STAT ( " ResourceGroup=%s\n",
                      $sResourceFilename
                    );
    }
    else {
        printf STAT ( "\n" );
    }

    if ( $sRGPFileLLName ne "" ){
        printf STAT ( "%-30s%-24sLogical Location=%s Name=%s\n",
                      "",
                      "Resource File Details:",
                      "",
                      ""
                    );
    }
    printf STAT ( "%-30s%-24sLocation=%s\n",
                  "",
                  "Document Data:",
                  $llDocumentData->{ 'name' }
                );
    printf STAT ( "%-30s%-24sLocation=%s\n\n",
                  "",
                  "StoreProcessRequest:",
                  $llStoreProcess->{ 'name' }
                );

    #---- Write location details
    # Location Details:            HNRES: Type=DIR DirectoryName=X:\Archiver3\SpoolIn PathSuffixPattern=RequestId/MD5Id FilenamePattern="%s%s0008",MD5ID,TIMESTAMP
    #                              HNRES: Type=DIR DirectoryName=X:\Archiver3\SpoolIn PathSuffixPattern=RequestId/MD5Id FilenamePattern="%s%s0008",MD5ID,TIMESTAMP
    #                              ProcessSpoolRequest: Type=DIR DirectoryName=..\perl\ProcessSpoolIn PathSuffixPattern=RequestId/MD5Id FilenamePattern="%s%s0008",MD5ID,TIMESTAMP
    #                              HNOUT: Type=DIR DirectoryName=X:\Archiver3 PathSuffixPattern=RequestId/ FilenamePattern="%s%s0008",MD5ID,TIMESTAMP
    #                              StoreProcess: Type=DIR DirectoryName=X:\Archiver3\StoreProcess PathSuffixPattern=RequestId/FileCount FilenamePattern="%s%s0008",MD5ID,TIMESTAMP FileCount=500
    #
    #printf STAT ( "%-30s%s\n",
    #              "Location details:",
    #              $sSpoolFileLLName . ": " . _logicalLocationToString( $llSpool )
    #            );
    #printf STAT ( "%-30s%s\n",
    #              "",
    #              $sRGPFileLLName . ": " . _logicalLocationToString( $llResource )
    #            );
    printf STAT ( "%-30s%s\n",
                  "Location details:",
                  $llDocumentData->{ 'name' } . ": " . _logicalLocationToString( $llDocumentData )
                );
    printf STAT ( "%-30s%s\n\n",
                  "",
                  $llStoreProcess->{ 'name' } . ": " . _logicalLocationToString( $llStoreProcess )
                );

    #---- Write document processed details
    # Document processed:          0 (0 Documents/min)
    #
    my $tElapsedInMinsTmp = $tElapsedTmp / 60;
    if ( $tElapsedInMinsTmp > 0 ){
        printf STAT ( "%-30s%d (%d Documents/min)\n\n",
                      "Document processed:",
                      $iDocCount,
                      ( $iDocCount / $tElapsedInMinsTmp )
                    );
    }
    else {
        printf STAT ( "%-30s%d (%d Documents/min)\n\n",
                      "Document processed:",
                      $iDocCount,
                      0
                    );
    }

    #---- Write total input pages processed details
    # Total Input Pages processed: 0 (0 Pages/min)
    #
    if ( $tElapsedInMinsTmp > 0 ){
        printf STAT ( "%-30s%d (%d Pages/min)\n\n",
                      "Total Input Pages processed:",
                      $iPageCount,
                      ( $iPageCount / $tElapsedInMinsTmp )
                    );
    }
    else {
        printf STAT ( "%-30s%d (%d Pages/min)\n\n",
                      "Total Input Pages processed:",
                      $iPageCount,
                      0
                    );

    }

    #---- Close statistics file
    close( STAT );

    return 1;
}

#-----------------------------------------------------------------------
# Fetch hash element using case-insensitive key
#
# Parameter(s)
# 1. Hash reference
# 2. Key
#
#-----------------------------------------------------------------------
sub _fetchHashValue{
    #---- Fetch parameter(s)
    #
    # 1. Hash reference
    # 2. Key (string)
    #
    my $hshrefDataTmp = shift;
    my $sKeyPar       = shift;

    if ( ref( $hshrefDataTmp ) ne "HASH"
         || $sKeyPar eq ""
       ){
       return undef;
    }

    #---- Prepare case-insensitive key
    my $sKeyTmp = lc( $sKeyPar );

    #---- Fetch keys list
    my @KeysListTmp = keys( %{ $hshrefDataTmp } );

    #---- Loop through hash elements and find required element
    my $ElementTmp = undef;
    for ( my $i = 0; $i < @KeysListTmp; $i++ ){
        if ( $sKeyTmp eq lc( @KeysListTmp[ $i ] ) ){
            $ElementTmp = $hshrefDataTmp->{ @KeysListTmp[ $i ] };
            last;    # break the loop
        }
    }

    return $ElementTmp;
}

__END__
