#-------------------------------------------------------------------------------
#  ci.smartfix_rescan.pm:
#
#  Module to process PDF input and to add rescan barcode page before each logical document
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -doc_cold -sp:ci.smartfix_rescan.pm -sa:rescan_args.json samples\insure.pdf
#
#  On Unix:    ./afp2web   -q -c -doc_cold -sp:ci.smartfix_rescan.pm -sa:rescan_args.json samples/insure.pdf
#
#  Author  : Fa. Maas
#
#  $V100   2016-11-04    Initial Release
#
#  $V101   2016-11-11    a. Extended to delete the script arguments JSON file
#                        b. Fixed minor bug in detecting missing pages, when process has only one page
#                        c. Fixed E086 PDF library error (Msg: Missing width information for character  (0x2F18D0), where width is available in between   (0x5F) and .)
#                        JiRa: OTS-971
#                        Done by: Panneer
#
#  $V102   2018-02-26    Extended to assert JSON attributes pages, reason, barcodeProcessId
#                        JiRa: OTS-1918
#                        Done by: Panneer
#
#  $V103   2018-05-02    Modified assertion of JSON attributes as given below
#                        1. If processes is empty or undefined, throw error (retaining existing flow)
#                        2. If processes[n]->pages is empty or undefined, throw error (retaining existing flow)
#                        3. If processes[n]->reason is empty or undefined, do not write reason on rescan page (since this attribute can be empty)
#                        4. If processes[n]->barcodeProcessId is empty or undefined, throw error (retaining existing flow and assertion of this attribute is not reported as wrong)
#                        JiRa: OTS-2151, TFS-160910
#                        Done by: Panneer
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
    unshift( @INC, $sScriptFilePathTmp . "/a2w" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/site/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/site/lib" );
}

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;

use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;

use a2w::core::bo::BOPool;       # V102 Change
use a2w::core::bo::Constants;    # V102 Change
use a2w::core::process::Logger;  # V102 Change

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sLogFilePath      = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGPATH );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    #---- Get global BO pool
    $theBOPoolLocal = a2w::core::bo::BOPool::getBOPool();

    #---- Get logger
    $theLoggerLocal = $theBOPoolLocal->getLogger();

    if ( $bLog == $TRUE ){
        #---- List of modules to be logged
        my $sLogModuleListTmp = ""
                                . "main"
                                ;

        #---- Register this module to log
        $theLoggerLocal->registerClasses( $sLogModuleListTmp );

        $bLog = $theLoggerLocal->isRegistered( "main" );

        #---- Open logger
        if ( $sLogModuleListTmp ne "" ){
            $theLoggerLocal->open( $sLogFilePath, "ci.smartfix" );
        }
    } 

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( "main", "initialize" );
        $theLoggerLocal->logMessage( "Running " . $sScriptProcTmp . "..." );
        $theLoggerLocal->logMessage( "Processing " . $sSpoolFilename );
        $theLoggerLocal->logMessage( "Args="
                                     . $sScriptArgs
                                     . ", OutputFilePath: " . $sOutputFilePath
                                   );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Load rescan arguments JSON
    my $iRcTmp = open( JSON, "<$sScriptArgsTmp" );
    if ( $iRcTmp == 0 ){
        return ( -1, "Unable to open rescan arguments JSON file $sScriptArgsTmp. reason=" . $! ); # V102 Change: Removed return code printing, since it is zero always
    }
    my $sRescanArgsTmp = do { local $/; <JSON> };
    close( JSON );

    #---- Decode rescan arguments
    $hrefRescanArgs = undef;
    eval{
        require JSON::Tiny;
        $hrefRescanArgs = JSON::Tiny::from_json( $sRescanArgsTmp );
    };
    if ( $@ ){
        return ( -2, "Unable to parse rescan arguments JSON file $sScriptArgsTmp. reason=" . $@ ); # V102 Change: Removed return code printing, since it is not from json parsing
    }
    elsif ( $hrefRescanArgs == undef ){
        return ( -2, "Unable to parse rescan arguments JSON file $sScriptArgsTmp." ); # V102 Change: Removed return code printing, since it is not from json parsing
    }
    if ( $bLog == $TRUE ){
        $theLoggerLocal->logMessage( "Rescan Arguments: $hrefRescanArgs" );
        my @arrKeysTmp = sort keys( %{ $hrefRescanArgs } );
        foreach my $k ( @arrKeysTmp ){
            if ( ref( $hrefRescanArgs->{ $k } ) eq "SCALAR" ){
                $theLoggerLocal->logMessage( "$k:>" . $hrefRescanArgs->{ $k } . "<" );
            }
            elsif ( ref( $hrefRescanArgs->{ $k } ) eq "ARRAY" ){
                my @arrTmp = @{ $hrefRescanArgs->{ $k } };
                $theLoggerLocal->logMessage( "$k:>arr(@arrTmp)<" );
            }
        }
    }

    $sSAJSONFile = $sScriptArgsTmp; # $V101a Change

    #---- Get list of processes
    @arrProcesses = @{ $hrefRescanArgs->{ 'processes' } };
    $iCurrentProcess = 0;
    $iPageCount = 0;

    # $V102 Begin
    #---- Validate rescan arguments
    $iFirstPageId = 0xFFFF;
    $iLastPageId = 0;

    # NOTE: _processAndValidateArguments will update $hrefRescanArgs
    my ( $iRetTmp, $sMsgTmp ) = _processAndValidateArguments( $hrefRescanArgs );
    if ( $iRetTmp < 0 ){ return ( -2, $sMsgTmp ); }
    if ( $bLog == $TRUE ){
        $theLoggerLocal->logMessage( "Lower page id: $iFirstPageId" );
        $theLoggerLocal->logMessage( "Upper page id: $iLastPageId" );
    }

    #---- Prepare pages list for first process
    $hrefActiveProcess = $arrProcesses[ $iCurrentProcess ];
    @arrPages = @{ $hrefActiveProcess->{ 'sorted_pages' } };
    %hshPages = map { $_ => $TRUE } @arrPages;
    # $V102 End

    #---- Rescan page constants
    $iPageWidth  = 598; # points
    $iPageHeight = 845; # points

    $sReasonTitle = "REASON FOR RESCAN :";
    $iRTXPos      = 37.8;  # points
    $iRTYPos      = $iPageHeight - 725.35; # points
    $fntText      = new a2w::Font( $a2w::FontConstants::TYPE_TYPE1, "helvetica" );
    $fntText->setHeight( 12 );
    $fntText->setCharacterSetName( "C0H20012" ); # $V101c Change

    $iReasonXPos  = 37.70;  # points
    $iReasonYPos  = $iPageHeight - 692; # points
    $MAX_REASON_LINE_LEN = 80;

    $iReadableStackIdXPos  = 35;  # points
    $iReadableStackIdYPos  = $iPageHeight - 266.95; # points

    $iStackIdXPos  = 201;  # points
    $iStackIdYPos  = $iPageHeight - 237; # points
    $fntCodabar    = new a2w::Font( $a2w::FontConstants::TYPE_TTF, "codabar" );
    $fntCodabar->setHeight( 49 );
    $fntBCHR       = new a2w::Font( $a2w::FontConstants::TYPE_TYPE1, "helvetica" );
    $fntBCHR->setHeight( 8 );
    $fntBCHR->setCharacterSetName( "C0H20008" ); # $V101c Change

    $iPgCntXPos    = 35;  # points
    $iPgCntYPos    = $iPageHeight - 167; # points

    $iProcIdXPos   = 201;  # points
    $iProcIdYPos   = $iPageHeight - 137; # points

    #---- AFP2web constants
    $A2W_PAGE_CONTENT_RESOLUTION = 240;

    #---- Set rescan watermark
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::WATERMARK, "RESCAN,helvetica,540,45,189,219,231" );

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( "main", "initializeDoc" );
        $theLoggerLocal->logMessage( "Id=" . $a2wDocumentPar->getId() );
    }

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){ $theLoggerLocal->logFunctionName( "main", "initializePage" ); }

    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){
    my $iPgIdTmp = $a2wPagePar->getParseId();
    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( "main", "afp2web" );
        $theLoggerLocal->logMessage( "Id=$iPgIdTmp" );
    }

    #---- Set default return value
    my $iRetTmp = $SKIP; # Skip page to Current Document

    #---- Assert whether page has to be skipped or not
    if ( %hshPages->{ $iPgIdTmp } == $TRUE ){
        $iPageCount++;
        $iRetTmp = $APPEND;
        %hshPages->{ $iPgIdTmp } = $FALSE; # V102 Change
    }

    #---- Get pages of current process
    if ( $iPgIdTmp == $arrPages[ 0 ] ){
        #---- Insert rescan page ----#
        my $a2wPageTmp = new a2w::Page( 4 ); # 4=IPS page
        if ( $a2wPageTmp == undef ){ return ( -3, "Unable to create recan page for process " . ( $iCurrentProcess + 1 ) ); }

        #---- Format page
        my $iResTmp = $a2wPagePar->getResolution();
        my $iPWTmp = ( $iPageWidth / 72 ) * $iResTmp;
        my $iPHTmp = ( $iPageHeight / 72 ) * $iResTmp;
        $a2wPageTmp->setResolution( $iResTmp );
        $a2wPageTmp->setWidth( $iPWTmp );
        $a2wPageTmp->setHeight( $iPHTmp );

        #---- Add contents to page ----#
        #---- Add reason title
        my $txtReasonTitleTmp = new a2w::Text();
        $txtReasonTitleTmp->setXPos( ( $iRTXPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
        $txtReasonTitleTmp->setYPos( ( $iRTYPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
        $txtReasonTitleTmp->setFont( $fntText );
        $txtReasonTitleTmp->setText( $sReasonTitle );
        $a2wPageTmp->addText( $txtReasonTitleTmp );

        #---- Add reason for rescan
        if ( $arrProcesses[ $iCurrentProcess ]->{ 'reason' } ne "" ){
            if ( length( $arrProcesses[ $iCurrentProcess ]->{ 'reason' } ) <= $MAX_REASON_LINE_LEN ){
                my $txtReasonTmp = new a2w::Text();
                $txtReasonTmp->setXPos( ( $iReasonXPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
                $txtReasonTmp->setYPos( ( $iReasonYPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
                $txtReasonTmp->setFont( $fntText );
                $txtReasonTmp->setText( $arrProcesses[ $iCurrentProcess ]->{ 'reason' } );
                $a2wPageTmp->addText( $txtReasonTmp );
            }
            else {
                my @arrReasonLinesTmp = $arrProcesses[ $iCurrentProcess ]->{ 'reason' } =~ /.{1,$MAX_REASON_LINE_LEN}/g;
                my $txtReasonTmp = undef;
                my $iYPosTmp = $iReasonYPos;
                if ( @arrReasonLinesTmp > 5 ){
                    @arrReasonLinesTmp = @arrReasonLinesTmp[ 0 .. 4 ];
                    @arrReasonLinesTmp[ 5 ] = "...";
                }

                foreach $sLineTmp ( @arrReasonLinesTmp ){
                    $txtReasonTmp = new a2w::Text();
                    $txtReasonTmp->setXPos( ( $iReasonXPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
                    $txtReasonTmp->setYPos( ( $iYPosTmp / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
                    $txtReasonTmp->setFont( $fntText );
                    $txtReasonTmp->setText( $sLineTmp );
                    $a2wPageTmp->addText( $txtReasonTmp );
                    $iYPosTmp += $fntText->getHeight() + 2;
                }
            }
        }

        #---- Add readable stack id
        if ( $hrefRescanArgs->{ 'stackId' } ne "" ){
            my $txtRSIdTmp = new a2w::Text();
            $txtRSIdTmp->setXPos( ( $iReadableStackIdXPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtRSIdTmp->setYPos( ( $iReadableStackIdYPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtRSIdTmp->setFont( $fntText );
            $txtRSIdTmp->setText( $hrefRescanArgs->{ 'stackId' } );
            $a2wPageTmp->addText( $txtRSIdTmp );
        }

        #---- Add stack id as barcode
        if ( $hrefRescanArgs->{ 'barcodeStackId' } ne "" ){
            my $txtSIdBCTmp = new a2w::Text();
            $txtSIdBCTmp->setXPos( ( $iStackIdXPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtSIdBCTmp->setYPos( ( $iStackIdYPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtSIdBCTmp->setFont( $fntCodabar );
            $txtSIdBCTmp->setText( "A" . $hrefRescanArgs->{ 'barcodeStackId' } . "B" );
            $a2wPageTmp->addText( $txtSIdBCTmp );

            #---- Add human readable text
            my $txtSIdHRTmp = new a2w::Text();
            my $iSIdHRXPosTmp = $iStackIdXPos + ( ( length( "a" . $hrefRescanArgs->{ 'barcodeStackId' } . "b" ) / 4 ) * ( $fntCodabar->getHeight() * 0.39 ) );
            my $iSIdHRYPosTmp = $iStackIdYPos + $fntBCHR->getHeight() - 2;
            $txtSIdHRTmp->setXPos( ( $iSIdHRXPosTmp / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtSIdHRTmp->setYPos( ( $iSIdHRYPosTmp / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtSIdHRTmp->setFont( $fntBCHR );
            $txtSIdHRTmp->setText( "a" . $hrefRescanArgs->{ 'barcodeStackId' } . "b" );
            $a2wPageTmp->addText( $txtSIdHRTmp );
        }

        #---- Add page count
        if ( @arrPages > 0 ){
            my $txtPgCntTmp = new a2w::Text();
            $txtPgCntTmp->setXPos( ( $iPgCntXPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtPgCntTmp->setYPos( ( $iPgCntYPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtPgCntTmp->setFont( $fntText );
            $txtPgCntTmp->setText( "" . @arrPages );
            $a2wPageTmp->addText( $txtPgCntTmp );
        }

        #---- Add process id as barcode
        if ( $arrProcesses[ $iCurrentProcess ]->{ 'barcodeProcessId' } ne "" ){
            my $txtPIdBCTmp = new a2w::Text();
            $txtPIdBCTmp->setXPos( ( $iProcIdXPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtPIdBCTmp->setYPos( ( $iProcIdYPos / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtPIdBCTmp->setFont( $fntCodabar );
            $txtPIdBCTmp->setText( "A" . $arrProcesses[ $iCurrentProcess ]->{ 'barcodeProcessId' } . "B" );
            $a2wPageTmp->addText( $txtPIdBCTmp );

            #---- Add human readable text
            my $txtPIdHRTmp = new a2w::Text();
            my $iPIdHRXPosTmp = $iProcIdXPos + ( ( length( "a" . $arrProcesses[ $iCurrentProcess ]->{ 'barcodeProcessId' } . "b" ) / 4 ) * ( $fntCodabar->getHeight() * 0.39 ) );
            my $iPIdHRYPosTmp = $iProcIdYPos + $fntBCHR->getHeight() - 2;
            $txtPIdHRTmp->setXPos( ( $iPIdHRXPosTmp / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtPIdHRTmp->setYPos( ( $iPIdHRYPosTmp / 72 ) * $A2W_PAGE_CONTENT_RESOLUTION );
            $txtPIdHRTmp->setFont( $fntBCHR );
            $txtPIdHRTmp->setText( "a" . $arrProcesses[ $iCurrentProcess ]->{ 'barcodeProcessId' } . "b" );
            $a2wPageTmp->addText( $txtPIdHRTmp );
        }

        #---- Add page to document
        $a2wDocumentPar->addPage( $a2wPageTmp, $iPageCount );
        if ( $bLog == $TRUE ){ $theLoggerLocal->logMessage( "Inserted rescan page at $iPageCount on output document" ); }
        $iPageCount++;
    }

    # $V101b Begin
    #elsif ( $iPgIdTmp == $arrPages[ $#arrPages ] ){
    if ( $iPgIdTmp == $arrPages[ $#arrPages ] ){
    # $V101b End
        # $V102 Begin
        $hrefActiveProcess->{ 'done' } = $TRUE;
        $iCurrentProcess++;
        if ( $iCurrentProcess < @arrProcesses ){
            #---- Prepare pages list for first process
            $hrefActiveProcess = $arrProcesses[ $iCurrentProcess ];
            @arrPages = @{ $hrefActiveProcess->{ 'sorted_pages' } };
            %hshPages = map { $_ => $TRUE } @arrPages;
        }
        # $V102 End
        else {
            @arrPages = ( 0 );
        }
    }

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){ $theLoggerLocal->logFunctionName( "main", "finalizePage" ); }

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){ $theLoggerLocal->logFunctionName( "main", "finalizeDoc" ); }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){ $theLoggerLocal->logFunctionName( "main", "finalize" ); }

    # $V101ab Begin
    my $iTransformationErrCodeTmp = $a2wKernelPar->getExitStatus();

    # Assert transformation is successful
    if ( $iTransformationErrCodeTmp >= 0 ){
        # V102 Begin
        #---- Determine the id of first missing page
        my $iFirstMissingPageIdTmp = -1;
        if ( $hrefActiveProcess->{ 'done' } == $FALSE ){
            my @arrPagesTmp = keys( %hshPages );
            @arrPagesTmp = grep { %hshPages->{ $_ } == $TRUE } @arrPagesTmp;
            @arrPagesTmp = sort @arrPagesTmp;
            $iFirstMissingPageId = $arrPagesTmp[ 0 ];
        }
        # V102 End

        #---- Assert whether processes were all done
        my $sProcessIdsTmp = "";
        my $sSeparatorTmp = "";
        foreach $p ( @arrProcesses ){
            if ( $p->{ 'done' } != $TRUE ){
                if ( $iFirstMissingPageIdTmp == -1 ){ $iFirstMissingPageIdTmp = $p->{ 'sorted_pages' }[ 0 ]; } # V102 Change
                $sProcessIdsTmp .= $sSeparatorTmp . $p->{ 'barcodeProcessId' };
                $sSeparatorTmp = ", ";
            }
        }

        # Assert missing pages of all process
        if ( $sProcessIdsTmp ne "" ){
            # Throw error
            #return ( -4, "Missing pages for process(es) $sProcessIdsTmp" ); # V102 Change
            return ( -4, "Pages $iFirstMissingPageIdTmp-$iLastPageId where missing for document(s) $sProcessIdsTmp" ); # V102 Change
        }

        #---- Delete script argument JSON file
        my $iRcTmp = unlink( $sSAJSONFile );
        if ( $iRcTmp <= 0 ){
            if ( $bLogTmp ){
                $theLoggerLocal->logMessage( "Unable to delete file " . $sSAJSONFile . " rc=" . $iRcTmp . " reason=" . $! );
            }

            #return ( -5, "Unable to delete file " . $sSAJSONFile . " rc=" . $iRcTmp . " reason=" . $! ); # V102 Change: Do not throw error, since transformation is successfull.
        }
    }
    # $V101ab End

    return 0;
}

# V102 Begin
#-----------------------------------------------------------------------
# Process and Validate arguments
#
# Process and Validate rescan JSON arguments
# Returns >=0 on valid arguments else <0, error message
#
#-----------------------------------------------------------------------
sub _processAndValidateArguments{

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( "main", "_processAndValidateArguments" );
    }

    #---- Get parameter
    #
    # 1. Rescan arguments (of type array reference)
    #
    my $hrefRescanArgsPar = shift;

    if ( $hrefRescanArgsPar->{ 'processes' } == undef ){ return ( -2, "Undefined 'processes' on ci.smartfix_rescan JSON script argument" ); }

    #---- Iterate through rescan processes (aka document) and validate them
	my @arrProcessesTmp = @{ $hrefRescanArgsPar->{ 'processes' } };
	if ( @arrProcessesTmp <= 0 ){ return ( -2, "Empty 'processes' on ci.smartfix_rescan JSON script argument" ); }

    my $iIdTmp = 0;
    my @arrMissingPGTmp = (); # list of process ids (whose pages argument is missing)
    my @arrMissingRSTmp = (); # list of process ids (whose reason argument is missing)
    my @arrMissingBCTmp = (); # list of process ids (whose barcodeProcessId argument is missing)

    my $iPgCntTmp = 0;
    my @arrDocsTmp = @{ $arefDocsPar };
    foreach my $p ( @arrProcessesTmp ){
        $iIdTmp++;

        #---- Assert pages, reason, barcodeProcessId values
        if ( !defined( $p->{ 'pages' } ) || length( $p->{ 'pages' } ) <= 0 ){ $arrMissingPGTmp[ @arrMissingPGTmp ] = $iIdTmp; next; }
        if ( !defined( $p->{ 'reason' } ) || length( $p->{ 'reason' } ) <= 0 ){
            $arrMissingRSTmp[ @arrMissingRSTmp ] = $iIdTmp;
            $p->{ 'reason' } = ""; # Force empty value to avoid undefined error when writing reason on page # $V103 Change
            #next; # $V103 Change
        }
        if ( !defined( $p->{ 'barcodeProcessId' } ) || length( $p->{ 'barcodeProcessId' } ) <= 0 ){ $arrMissingBCTmp[ @arrMissingBCTmp ] = $iIdTmp; next; }

        my @arrPagesTmp = split /,/, $p->{ 'pages' }; # convert string to array of page ids
        if ( @arrPagesTmp <= 0 ){ $arrMissingPGTmp[ @arrMissingPGTmp ] = $iIdTmp; next; }

        $p->{ 'pages' } = \@arrPagesTmp;
        $iPgCntTmp = @arrPagesTmp;
        $p->{ 'pagecount' } = $iPgCntTmp;

        my @arrSortedPagesTmp = sort { $a <=> $b } @arrPagesTmp;
        $p->{ 'sorted_pages' } = \@arrSortedPagesTmp;

        if ( $iFirstPageId > $arrSortedPagesTmp[ 0 ] ){ $iFirstPageId = $arrSortedPagesTmp[ 0 ]; }
        if ( $iLastPageId < $arrSortedPagesTmp[ $#arrSortedPagesTmp ] ){ $iLastPageId = $arrSortedPagesTmp[ $#arrSortedPagesTmp ]; }
    }

    #---- Assert arguments for mandatory fields
    my $iValidTmp = 0;
    my $sSeparatorTmp = "";
    my $sMsgTmp = "";
    if ( @arrMissingPGTmp > 0 ){
        $iValidTmp = -1;
        local $" = ',';
        $sMsgTmp .= "'pages' argument is missing or empty for document(s) @arrMissingPGTmp";
        $sSeparatorTmp = ", ";
    }
    # $V103 Begin
    # 3. If processes[n]->reason is empty or undefined, do not write reason on rescan page (since this attribute can be empty)
    #if ( @arrMissingRSTmp > 0 ){
    #    $iValidTmp = -1;
    #    local $" = ',';
    #    $sMsgTmp .= $sSeparatorTmp . "'reason' argument is missing or empty for document(s) @arrMissingRSTmp";
    #    $sSeparatorTmp = ", ";
    #}
    # $V103 End
    if ( @arrMissingBCTmp > 0 ){
        $iValidTmp = -1;
        local $" = ',';
        $sMsgTmp .= $sSeparatorTmp . "'barcodeProcessId' argument is missing or empty for document(s) @arrMissingBCTmp";
        $sSeparatorTmp = ", ";
    }
    if ( $iValidTmp < 0 ){ return ( $iValidTmp, $sMsgTmp ); }

    return $iValidTmp;
}
# V102 End

__END__
