#-------------------------------------------------------------------------------
#  ci.smartfix_thresholdDocs.pm:
#
#  Module to process 'n' PDF inputs and to threshold PDF pages if they are color
#
#  Call:
#
#  On Windows: afp2web.exe -q -c -doc_cold -sp:ci.smartfix_thresholdDocs.pm -bws:threshold:adaptive samples\insure.pdf samples\medform.pdf
#  On Unix   : ./afp2web   -q -c -doc_cold -sp:ci.smartfix_thresholdDocs.pm -bws:threshold:adaptive samples/insure.pdf samples/medform.pdf
#
#  Author  : Fa. Maas
#
#  $V100   2018-03-01    Initial Release
#
#  $V101   2018-06-05    Fixed minor bug (Can't call method "getBitsPerPixel" on an undefined value)
#                        in accessing image object info.
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
    unshift( @INC, $sScriptFilePathTmp . "/a2w" );
}

use a2w::Config;
use a2w::ConfigConstants;
use a2w::Document;
use a2w::DocumentConstants;
use a2w::Font;
use a2w::FontConstants;
use a2w::Image;
use a2w::ImageConstants;
use a2w::Kernel;
use a2w::Line;
use a2w::Page;
use a2w::PageConstants;
use a2w::Text;
use a2w::TypeConstants;
use a2w::Vector;

use a2w::core::bo::BOPool;
use a2w::core::bo::Constants;
use a2w::core::process::Logger;

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sLogFilePath      = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGPATH );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    #---- Get global BO pool
    $theBOPoolLocal = a2w::core::bo::BOPool::getBOPool();

    #---- Get logger
    $theLoggerLocal = $theBOPoolLocal->getLogger();

    if ( $bLog == $TRUE ){
        #---- List of modules to be logged
        my $sLogModuleListTmp =   "main"
                                #. "..."
                                ;

        #---- Register this module to log
        $theLoggerLocal->registerClasses( $sLogModuleListTmp );

        $bLog = $theLoggerLocal->isRegistered( "main" );

        #---- Open logger
        my $sLogPrefixTmp = "ci.smartfix_thresholdDocs";
        if ( $sSpoolFilename =~ /.*\/(.*)\..{3}$/ ){
            if ( $1 ne "" ){ $sLogPrefixTmp .= "_" . $1; }
        }
        if ( $sLogModuleListTmp ne "" ){ $theLoggerLocal->open( $sLogFilePath, $sLogPrefixTmp ); }

        $theLoggerLocal->logFunctionName( "main", "initialize" );
        $theLoggerLocal->logMessage( "Running " . $sScriptProcTmp . "..." );
        $theLoggerLocal->logMessage( "Processing " . $sSpoolFilename );
        $theLoggerLocal->logMessage( "Args=" . $sScriptArgs . ", OutputFilePath: " . $sOutputFilePath );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){ $theLoggerLocal->logFunctionName( "main", "initializeDoc(" . $a2wDocumentPar->getId() . ")" ); }

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){ $theLoggerLocal->logFunctionName( "main", "initializePage" ); }

    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    my $iPgIdTmp = $a2wPagePar->getParseId();
    if ( $bLog == $TRUE ){ $theLoggerLocal->logFunctionName( "main", "afp2web($iPgIdTmp)" ); }

    #---- Analyze page content and turn off threshold if page is B/W ----#
    if (    $a2wPagePar->getThreshold() == $TRUE
         && _isColorPage( $a2wPagePar ) == $FALSE # B/W Page, skip threshold
       ){
        $a2wPagePar->setThreshold( $FALSE );
        if ( $bLog == $TRUE ){ $theLoggerLocal->logMessage( "Threshold turned off for page " . $a2wPagePar->getParseId() ); }
    }

    return $APPEND; # Append page to document
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){ $theLoggerLocal->logFunctionName( "main", "finalizePage" ); }

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){ $theLoggerLocal->logFunctionName( "main", "finalizeDoc" ); }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){ $theLoggerLocal->logFunctionName( "main", "finalize" ); }

    return 0;
}

#-----------------------------------------------------------------------
# Analyze page content and determine whether page is color or not
#
# Returns TRUE if page is color else FALSE
#
#-----------------------------------------------------------------------
sub _isColorPage{
    my $pagePar = shift;

    if ( $bLog == $TRUE ){ $theLoggerLocal->logFunctionName( "main", "_isColorPage" ); }

    my $bColorPageTmp = $TRUE;
    #---- Analyze page contents ----#
    #
    # Rule 1:
    # If page has only one image or container and that image/container is of 1 bpp, then page is B/W
    #
    #NOTE: Overlay/PSEG are not checked since input is PDF always
    if (    $pagePar->getFirstLine() == undef
         && $pagePar->getFirstText() == undef
         && $pagePar->getFirstVector() == undef
         && $pagePar->getFirstImage() != undef # $V101 Change: Assert image object before accessing it's info
       ){
        my $a2wImageTmp = $pagePar->getFirstImage();
        if ( $a2wImageTmp->getBitsPerPixel() == 1 ){ $bColorPageTmp = $FALSE; }
    }

    if ( $bLog == $TRUE ){ $theLoggerLocal->logMessage( "Page " . $pagePar->getParseId() . " is " . ( ( $bColorPageTmp == $TRUE ) ? "color" : "b/w" ) ); }

    return $bColorPageTmp;
}
# V102 End

__END__
