#-------------------------------------------------------------------------------
#  ci.stack_metapage.pm:
#  Scripting Facility Module to add stack meta
#
#  Call:
#  On Windows:
#  afp2web.exe -q -c -pdf -doc_cold -sp:ci.stack_metapage.pm  -sa:<Stack Info JSON String> samples\afpsample.afp
#
#  On Unix:
#  ./afp2web   -q -c -pdf -doc_cold -sp:ci.stack_metapage.pm -sa:<Stack Info JSON String> samples/afpsample.afp
#
#  where
#  Stack Info JSON String must have following content
#	{
#		"XML Type": <XML Type like Export.XML|Import.XML>,
#		"info":{
#			"Import Date" : <Import Date>,
#			"StackId": <StackId>,
#			"ScannerID": <ScannerID>,
#			"Import Channel": <Import Channel> (Ex: ServiceMix2),
#			"Input Source": <Input Source>  (Ex: Fax, AVIBEmail),
#			"XML Filename": <Fully Qualified Filename>
#		}
#	}
#
#  Author  : Fa. Maas
#
#  V100   2017-03-01    Initial Release
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script file name
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script file name
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script file name: " . $0 . " Script file path: " . $sScriptFilePathTmp . "\n" );
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );

            #---- Add local perl lib path relative to script file parent path to module search path
            unshift( @INC, $sScriptFileParentPathTmp . "/a2w" );
            unshift( @INC, $sScriptFileParentPathTmp . "/perl/lib" );
            unshift( @INC, $sScriptFileParentPathTmp . "/perl/site/lib" );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );

    #---- Add local perl lib path relative to script file parent path to module search path
    unshift( @INC, $sScriptFilePathTmp . "/a2w" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/site/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/site/lib" );
}

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;

use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;
use a2w::TypeConstants;

use a2w::core::bo::BOPool;
use a2w::core::process::Logger;
use a2w::core::path::Utils;
use a2w::core::file::Utils;

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    $bDebugLog = $FALSE;
    my $sLogLevelTmp = lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL ));
    if (index( $sLogLevelTmp, "sf") >= 0){ $bLog = $TRUE; }
    if (index( $sLogLevelTmp, "debug") >= 0 ){ $bLog = $TRUE; $bDebugLog = $TRUE; }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $iOutputResolution = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::RESOLUTION );
    $sOutputFormat     = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFORMAT );
    $sLogFilePath      = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGPATH );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Get global BO pool
    $theBOPoolLocal = a2w::core::bo::BOPool::getBOPool();

    #---- Get logger
    $theLoggerLocal = $theBOPoolLocal->getLogger();

    if ( $bLog == $TRUE ){
        #---- List of modules to be logged
        my $sLogModuleListTmp = ""
                                . __PACKAGE__
                                ;

        #---- Register this module to log
        $theLoggerLocal->registerClasses( $sLogModuleListTmp );

        $bLog = $theLoggerLocal->isRegistered( __PACKAGE__ );

        #---- Open logger
        if ( $sLogModuleListTmp ne "" ){ $theLoggerLocal->open( $sLogFilePath, "ci.stack_metapage" ); }

        $theLoggerLocal->logFunctionName( __PACKAGE__, "initialize" );
        $theLoggerLocal->logMessage( "Running " . $sScriptProcTmp . "..." );
        $theLoggerLocal->logMessage( "Processing " . $sSpoolFilename );
        $theLoggerLocal->logMessage( "Args="
                                     . $sScriptArgsTmp
                                     . ", OutputFilePath: " . $sOutputFilePath
                                     . ", Resolution: " . $iOutputResolution
                                   );
    }
	$sArgsTmp = $sScriptArgsTmp;
	
    if ( $bDebugLog == $TRUE ){ $theLoggerLocal->logMessage( "JSON Content:>$sArgsTmp<" ); }

    #---- Decode arguments
    $hrefArgs = undef;
    eval{
        require JSON::Tiny;
        $hrefArgs = JSON::Tiny::from_json( $sArgsTmp );
    };
    if ( $@ ){
        return ( -2, "Unable to parse arguments JSON $sScriptArgsTmp. rc=$iRcTmp reason=" . $@ );
    }
    elsif ( $hrefArgs == undef ){
        return ( -2, "Unable to parse arguments JSON  $sScriptArgsTmp. rc=$iRcTmp" );
    }

	#---- Font Height in points
	$iFontHeight = 8;
	
    #---- Create text font
	$a2wFont = new a2w::Font( $a2w::FontConstants::TYPE_TYPE1, "Courier" );
    $a2wFont->setHeight( $iFontHeight );

    #---- Create header font
	$a2wHeaderFont = new a2w::Font( $a2w::FontConstants::TYPE_TYPE1, "Helvetica" );
    $a2wHeaderFont->setHeight( $iFontHeight * 2 );
	
	$TITLE_TEXT         = 'Dieses Dokument beinhaltet als Anhang die §TYPE§';
	
    #---- Meta keys
    @arrMetaKeys = (
          'Import Date'
        , 'Stack Id'
        , 'Scanner Id'
        , 'Import Channel'
        , 'Input Source'
        , 'XML Filename'
    );

    return 0;
}
#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "initializeDoc" );
        $theLoggerLocal->logMessage( "Id=" . $a2wDocumentPar->getId() );
    }

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "initializePage" );
    }

	#---- Page formatting constants
    $AFP_PTX_RESOLUTION = $a2wPagePar->getResolution();
	$PAGE_LEFT_MARGIN   = 15; # mm
	$PAGE_TOP_MARGIN    = 15; # mm
	$KEY_X_POS 			= int( ( $PAGE_LEFT_MARGIN / 25.4 ) * $AFP_PTX_RESOLUTION );
	$TEXT_Y_START       = int( ( $PAGE_TOP_MARGIN / 25.4 ) * $AFP_PTX_RESOLUTION );
	$LINE_INCREMENT     = int( ( $iFontHeight / 72 ) * $AFP_PTX_RESOLUTION );
	$VALUE_X_POS 		= $KEY_X_POS * 3;

	# Maximum characters allowed per line for value
	$VALUE_MAX_CHARS_PER_LINE = 75;

    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "afp2web" );
        $theLoggerLocal->logMessage( "Id=" . $a2wPagePar->getParseId() );
    }
	
	#---- Add title text
	my $sTitleTextTmp = $TITLE_TEXT;
	$sTitleTextTmp =~ s/§TYPE§/$hrefArgs->{ 'XML Type' }/;
	my ( $iRetTmp, $sMsgTmp ) = _addTitleToPage( $sTitleTextTmp );
	if ( $iRetTmp < 0 ){ return ( $iRetTmp, $sMsgTmp ); }

	#---- Add stack meta info
	( $iRetTmp, $sMsgTmp ) = _addStackMetaToPage();
	if ( $iRetTmp < 0 ){ return ( $iRetTmp, $sMsgTmp ); }
	
    return $APPEND;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "finalizePage" );
    }

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "finalizeDoc" );
    }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "finalize" );
    }

	return 0;
}

#-----------------------------------------------------------------------
# Add title text to page
#-----------------------------------------------------------------------
sub _addTitleToPage{
    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "_addTitleToPage" );
    }

    #---- Fetch parameter
    #
    # 1. Title text
    my $sTitleTextPar = shift;
    if ( $sTitleTextPar eq "" ){
        return ( -1, "Empty title text to add on page" );
    }

	my $a2wTextTmp = new a2w::Text();
	if ( $a2wTextTmp == undef ){ return ( -1, "Unable to allocate text object for text ($sTitleTextPar)" ); }

	#---- Fill in text details
	$a2wTextTmp->setXPos( $KEY_X_POS );     # Text X position
	$a2wTextTmp->setYPos( $TEXT_Y_START );  # Text Y position
	$a2wTextTmp->setText( $sTitleTextPar ); # Text value
	$a2wTextTmp->setFont( $a2wHeaderFont ); # set font

	#---- Add text to page
	$a2wPagePar->addText( $a2wTextTmp );
	$theLoggerLocal->logMessage( "txt=>$sTitleTextPar<@($KEY_X_POS, $TEXT_Y_START)" );
	$TEXT_Y_START += ( $LINE_INCREMENT * 2 );

    return 0;
}

#-----------------------------------------------------------------------
# Add stack meta info to page
#-----------------------------------------------------------------------
sub _addStackMetaToPage{
    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "_addStackMetaToPage" );
    }

	my @values = ();
	my $a2wTextTmp = undef;
	my $iYPosTmp = $TEXT_Y_START;

    foreach my $key ( @arrMetaKeys ){
        if ( $bLog == $TRUE ){ $theLoggerLocal->logMessage( $key . ': ' .  $hrefArgs->{ 'info' }{ $key } ); }

		#---- Add key as text
		$a2wTextTmp = new a2w::Text();
		if ( $a2wTextTmp == undef ){ return ( -1, "Unable to allocate text object for text ($key:)" ); }

		#---- Fill in text details
		$a2wTextTmp->setXPos( $KEY_X_POS );  # Text X position
		$a2wTextTmp->setYPos( $iYPosTmp );   # Text Y position
		$a2wTextTmp->setText( $key . ':' );    # Text value
		$a2wTextTmp->setFont( $a2wFont );    # set font
		
		#---- Add text to page
		$a2wPagePar->addText( $a2wTextTmp );
		$theLoggerLocal->logMessage( "txt=>$key<@($KEY_X_POS, $iYPosTmp)" );
		if ( length( $hrefArgs->{ 'info' }{ $key } ) <= 0 ){ $iYPosTmp += $LINE_INCREMENT; next; }

		#---- Add value as text
		@values = unpack( "(A" . $VALUE_MAX_CHARS_PER_LINE . ")*", $hrefArgs->{ 'info' }{ $key } ); # Split value into tokens of $VALUE_MAX_CHARS_PER_LINE  length

		# Some values might be longer, so split them on lines
		# XML Filename: /mnt/inputchannels/sm2/2015-04-22_96_000014_work/000000/c2179b87b1c6ae5135405ffd49ab722e15042216
		#
		foreach $value ( @values ){
			if ( length( $value ) <= 0 ){ next; }

			$a2wTextTmp = new a2w::Text();
			if ( $a2wTextTmp == undef ){ return ( -1, "Unable to allocate text object for text ($value)" ); }

			#---- Fill in text details
			$a2wTextTmp->setXPos( $VALUE_X_POS ); # Text X position
			$a2wTextTmp->setYPos( $iYPosTmp );    # Text Y position
			$a2wTextTmp->setText( $value );       # Text value
			$a2wTextTmp->setFont( $a2wFont );     # set font
		
			#---- Add text to page
			$a2wPagePar->addText( $a2wTextTmp );
			$theLoggerLocal->logMessage( "txt=>$value<@($VALUE_X_POS, $iYPosTmp)" );
			$iYPosTmp += $LINE_INCREMENT;
		}
	}

    return 0;
}

__END__
