#-------------------------------------------------------------------------------
#  processAttachments.pm:
#  Scripting Facility Module to process PDF letter attachments
#
#  Call:
#  On Windows:
#  afp2web.exe -q -c -afp -doc_cold -sp:processAttachments.pm -sa:processAttachments.json samples\afpsample.afp
#  afp2web.exe -q -c -pdf -doc_cold -sp:processAttachments.pm -sa:processAttachments.json samples\afpsample.afp
#
#  On Unix:
#  ./afp2web   -q -c -afp -doc_cold -sp:processAttachments.pm -sa:processAttachments.json samples/afpsample.afp
#  ./afp2web   -q -c -pdf -doc_cold -sp:processAttachments.pm -sa:processAttachments.json samples/afpsample.afp
#
#  where
#  processAttachments.json must have following content
#  {   I_ARCHIV: J|N
#    , AttId: <simple output file name>
#    , DocInfoList : {
#        <document file name>: {
#            'TypeId': <document type id>
#          , 'SeqNr': <sequence number>
#          , 'tifPath': <Path to TIFF pages>
#          , 'tifPattern': <Pattern to recognize TIFF pages on tifPath>
#         [, 'RePrintMsg': { ['BlackoutPages': <List of comma separated pages having black out annotations as string>] }]
#        }+
#    }
#  }
#  Example
#  {
#       "I_ARCHIV":"J"
#     , "AttId":"2017_01_12X01_02_03_000001.attachment"
#     , "DocInfoList": {
#            "mnt/attachments/XE00118/26455000/2017_01_12X01_02_03_000001/2017-01-11-13.32.45.798000-0.pdf": {
#                "tifPath":"mnt/attachments/XE00118/26455000/2017_01_12X01_02_03_000001"
#              , "tifPattern":"^2017-01-11-13.32.45.798000-0-[0-9]*.tif$"
#              , "TypeId": 9000
#              , "SeqNr": 1
#              , "RePrintMsg": { "BlackoutPages": "1,2,4" }
#      }
#    }
#  }
#
#  Author  : Fa. Maas
#
#  V100   2017-01-01    Initial Release
#
#  V101   2017-01-19    Bug:
#                       For Non Pendelattest cases, the scaling of TIFF image were smaller than expected
#
#                       Reason:
#                       During scaling, this module gets TIFF image size and assumes the size is in output
#                       resolution (i.e 240). But TIFF image could have different resolution (like 200) and
#                       in such case the scaling goes wrong and thus image gets smaller (when TIFF resolution
#                       is less than output resolution) or bigger (when TIFF resolution is greater than output
#                       resolution).
#
#                       Fix:
#                       When evaluating TIFF image size, ensure to evaluate it on output resolution.
#
#                       JiRa Id, Customer:
#                       OTS-1041, Hallesche (DMS20)
#
#                       Fixed by:
#                       Panneer
#
#  V102   2019-02-13    Extended PERL module search path list with "perl/site/lib"
#
#  V103   2019-04-01    a. OTS-2482/TFS-112591: Modified reprint message for black out pages as given below
#                       "Nachdruck aus dem Archiv. Nachdruck und Original stimmen berein (Teile des Dokuments"
#                       "wurden aus Datenschutz-Grnden unkenntlich gemacht). HALLESCHE Krankenversicherung"
#                       "auf Gegenseitigkeit 70166 Stuttgart"
#
#                       b. OTS-2482/TFS-112591: Skipped adding reprint message for external documents
#
#  V104   2019-10-24    OTS-2522/TFS-210847: Extended to fit external document TIFF page in A4 portrait
#                       output page with aspect ratio
#  V105   2019-12-10    OTS-2653: Fixed missing left margin for Non pendelattest to PDF transformation
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script file name
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script file name
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script file name: " . $0 . " Script file path: " . $sScriptFilePathTmp . "\n" );
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );

            #---- Add local perl lib path relative to script file parent path to module search path
            unshift( @INC, $sScriptFileParentPathTmp . "/a2w" );
            unshift( @INC, $sScriptFileParentPathTmp . "/../../perl/lib" );
            unshift( @INC, $sScriptFileParentPathTmp . "/../../perl/site/lib" ); # V102 Change
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );

    #---- Add local perl lib path relative to script file parent path to module search path
    unshift( @INC, $sScriptFilePathTmp . "/a2w" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/site/lib" ); # V102 Change
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/site/lib" ); # V102 Change
}

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;

use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;
use a2w::TypeConstants;

use a2w::core::bo::BOPool;
use a2w::core::process::Logger;
use a2w::core::path::Utils;
use a2w::core::file::Utils;

use Image::Info qw(image_info);

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    $bDebugLog = $FALSE;
    my $sLogLevelTmp = lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL ));
    if (index( $sLogLevelTmp, "sf") >= 0){ $bLog = $TRUE; }
    if (index( $sLogLevelTmp, "debug") >= 0 ){ $bLog = $TRUE; $bDebugLog = $TRUE; }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $iOutputResolution = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::RESOLUTION );
    $sOutputFormat     = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFORMAT );
    $sLogFilePath      = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGPATH );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Get global BO pool
    $theBOPoolLocal = a2w::core::bo::BOPool::getBOPool();

    #---- Get logger
    $theLoggerLocal = $theBOPoolLocal->getLogger();

    if ( $bLog == $TRUE ){
        #---- List of modules to be logged
        my $sLogModuleListTmp = ""
                                . __PACKAGE__
                                ;

        #---- Register this module to log
        $theLoggerLocal->registerClasses( $sLogModuleListTmp );

        $bLog = $theLoggerLocal->isRegistered( __PACKAGE__ );

        #---- Open logger
        if ( $sLogModuleListTmp ne "" ){ $theLoggerLocal->open( $sLogFilePath, "la_" ); }

        $theLoggerLocal->logFunctionName( __PACKAGE__, "initialize" );
        $theLoggerLocal->logMessage( "Running " . $sScriptProcTmp . "..." );
        $theLoggerLocal->logMessage( "Processing " . $sSpoolFilename );
        $theLoggerLocal->logMessage( "Args="
                                     . $sScriptArgsTmp
                                     . ", OutputFilePath: " . $sOutputFilePath
                                     . ", Resolution: " . $iOutputResolution
                                   );
    }

    #---- Load arguments JSON
    my $iRcTmp = open( JSON, "<$sScriptArgsTmp" );
    if ( $iRcTmp == 0 ){
        return ( -1, "Unable to open arguments JSON file $sScriptArgsTmp. rc=$iRcTmp reason=" . $! );
    }
    my $sArgsTmp = do { local $/; <JSON> };
    close( JSON );

    #if ( $bDebugLog == $TRUE ){ $theLoggerLocal->logMessage( "JSON Content:>$sArgsTmp<" ); }

    #---- Decode arguments
    $hrefArgs = undef;
    eval{
        require JSON::Tiny;
        $hrefArgs = JSON::Tiny::from_json( $sArgsTmp );
    };
    if ( $@ ){
        return ( -2, "Unable to parse arguments JSON file $sScriptArgsTmp. rc=$iRcTmp reason=" . $@ );
    }
    elsif ( $hrefArgs == undef ){
        return ( -2, "Unable to parse arguments JSON file $sScriptArgsTmp. rc=$iRcTmp" );
    }

    #---- Process variables
    @arrDocFilenames = sort { $hrefArgs->{ 'DocInfoList' }{ $a }{ 'SeqNr' } <=> $hrefArgs->{ 'DocInfoList' }{ $b }{ 'SeqNr' } } keys( %{ $hrefArgs->{ 'DocInfoList' } } );

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logMessage( "Process Arguments ($sScriptArgsTmp):" );
        $theLoggerLocal->logHashMessage( $hrefArgs, " " );
    }

    #---- Process variables
    @arrTIFFs = ();
    @arrTIFFSizes = ();
    $hrefTIFFsInfo = {}; # Key: TIFF file name (with out path) Values: W=<width> H=<height>, SeqNr=<document sequence number>, TypeId=<document type id>

    #---- Prepare pattern to filter output files on output path
    my @arrTFNsTmp      = ();
    my $sTIFFPathTmp    = "";
    my $sTIFFPathsTmp   = "";
    my $sTIFFPatternTmp = "";
    my $reTIFFFilterTmp = undef;

    my @arrTFSizesTmp   = ();
    my $hrefTFInfoTmp   = undef;

    my $cSeparatorTmp   = "";
    my $iTFPageIdTmp    = 0;

    foreach my $sDFNTmp ( @arrDocFilenames ){
        #---- Get TIFF path and pattern
        if ( $hrefArgs->{ 'DocInfoList' }{ $sDFNTmp }{ 'tifPath' } eq "" ){ return ( -1, "Undefined TIFF path for $sDFNTmp input" ); }
        $sTIFFPathTmp    = $hrefArgs->{ 'DocInfoList' }{ $sDFNTmp }{ 'tifPath' };
        if ( $hrefArgs->{ 'DocInfoList' }{ $sDFNTmp }{ 'tifPattern' } eq "" ){ return ( -1, "Undefined TIFF pattern for $sDFNTmp input" ); }
        $sTIFFPatternTmp = $hrefArgs->{ 'DocInfoList' }{ $sDFNTmp }{ 'tifPattern' };
        $sTIFFPatternTmp =~ s/\./\\./g; # Replace '.' with '\.' to escape it

        #---- Collect list of TIFF paths (to be specified as object container path list)
        $sTIFFPathsTmp .= $cSeparatorTmp . $sTIFFPathTmp;
        $cSeparatorTmp = ",";

        #---- Fetch tiff files list for current document
        $reTIFFFilterTmp = qr/$sTIFFPatternTmp/;
        @arrTFNsTmp = a2w::core::path::Utils::list( $sTIFFPathTmp, $TRUE, $reTIFFFilterTmp );
        if ( @arrTFNsTmp <= 0 ){ return ( -1, "No output file found for $sDFNTmp input, when searched using pattern ($sTIFFPatternTmp)" ); }
        @arrTFNsTmp = sort { my $s1 = $a; my $s2 = $b; $s1 =~s{(\d+)}{sprintf "%02d", $1}eg; $s2 =~s{(\d+)}{sprintf "%02d", $1}eg; $s1 cmp $s2; } @arrTFNsTmp;
        push( @arrTIFFs, @arrTFNsTmp );

        #---- Fetch tiff size files list for current document
        $iTFPageIdTmp = 0;
        if ( $sTIFFPathTmp !~ /[\\\/]$/ ){ $sTIFFPathTmp .= "/"; }
        foreach my $sTFNTmp ( @arrTFNsTmp ){
            $hrefTFInfoTmp = image_info( $sTIFFPathTmp . $sTFNTmp );
            if ( $hrefTFInfoTmp->{ 'width' } <= 0 || $hrefTFInfoTmp->{ 'height' } <= 0 ){
                return ( -1, "Could not determine size for TIFF page ($sTFNTmp) of $sDFNTmp input" );
            }

            $hrefTIFFsInfo->{ $sTFNTmp }{ 'W' } = int( ( $hrefTFInfoTmp->{ 'width' }  * $iOutputResolution ) / $hrefTFInfoTmp->{ 'XResolution' } ); # V101 Change
            $hrefTIFFsInfo->{ $sTFNTmp }{ 'H' } = int( ( $hrefTFInfoTmp->{ 'height' } * $iOutputResolution ) / $hrefTFInfoTmp->{ 'YResolution' } ); # V101 Change
            $hrefTIFFsInfo->{ $sTFNTmp }{ 'OrgXRes' } = $hrefTFInfoTmp->{ 'XResolution' }; # V104 Change
            $hrefTIFFsInfo->{ $sTFNTmp }{ 'OrgYRes' } = $hrefTFInfoTmp->{ 'YResolution' }; # V104 Change
            $hrefTIFFsInfo->{ $sTFNTmp }{ 'PageId' } = ++$iTFPageIdTmp;
            $hrefTIFFsInfo->{ $sTFNTmp }{ 'SeqNr' } = $hrefArgs->{ 'DocInfoList' }{ $sDFNTmp }{ 'SeqNr' };
            $hrefTIFFsInfo->{ $sTFNTmp }{ 'TypeId' } = $hrefArgs->{ 'DocInfoList' }{ $sDFNTmp }{ 'TypeId' };
        }
    }

    #---- Set attachment id as file name pattern and output file path as object container path
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::OBJECTCONTAINERPATH, $sTIFFPathsTmp ); # To search TIFFs in correct path
    $sOutputExt    = "." . lc( substr( $sOutputFormat, 0, 3 ) );

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logMessage( "ObjectContainerPath=>$sTIFFPathsTmp<" );
        $theLoggerLocal->logMessage( "OutputFormat=>$sOutputFormat<" );
        my $hrefInfoTmp = undef;
        foreach my $sTFNTmp ( @arrTIFFs ){
            $theLoggerLocal->logMessage( "$sTFNTmp=> {" );
            $hrefInfoTmp = $hrefTIFFsInfo->{ $sTFNTmp };
            $theLoggerLocal->logMessage( " W: " . $hrefInfoTmp->{ 'W' } );
            $theLoggerLocal->logMessage( " H: " . $hrefInfoTmp->{ 'H' } );
            $theLoggerLocal->logMessage( " XRes: " . $hrefInfoTmp->{ 'OrgXRes' } ); # V104 Change
            $theLoggerLocal->logMessage( " YRes: " . $hrefInfoTmp->{ 'OrgYRes' } ); # V104 Change
            $theLoggerLocal->logMessage( " PageId: " . $hrefInfoTmp->{ 'PageId' } );
            $theLoggerLocal->logMessage( " SeqNr: " . $hrefInfoTmp->{ 'SeqNr' } );
            $theLoggerLocal->logMessage( " TypeId: " . $hrefInfoTmp->{ 'TypeId' } );
            $theLoggerLocal->logMessage( "}" );
        }
    }

    $PAGE_WIDTH  = 8.27;  # inches, A4 size (8.27  11.69) width
    $PAGE_HEIGHT = 11.69; # inches, A4 size (8.27  11.69) height

    $AFP_PTX_RESOLUTION = 240;
    $PAGE_RESOLUTION    = $iOutputResolution;

    $AFP_LEFT_MARGIN    = 15; # mm
    $WHITE_BOX_WIDTH    = ( $AFP_LEFT_MARGIN / 25.4 ) * $AFP_PTX_RESOLUTION;
    $WHITE_BOX_HEIGHT   = $PAGE_HEIGHT * $AFP_PTX_RESOLUTION;

    #---- Create re-print stamp font
    $fntRePrintStamp = undef;
    if ( $hrefArgs->{ 'DocInfoList' }{ $arrDocFilenames[ $iSeqNrTmp ] }{ 'RePrintMsg' } != undef ){
        $fntRePrintStamp = new a2w::Font( $a2w::FontConstants::TYPE_TYPE1, "times" );
        $fntRePrintStamp->setHeight( 8 );
    }

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "initializeDoc" );
        $theLoggerLocal->logMessage( "Id=" . $a2wDocumentPar->getId() );
    }

    #---- Set output file name ----#
    my $sOFNTmp = $hrefArgs->{ 'AttId' } . $sOutputExt;
    $a2wDocumentPar->setOutputFilename( $sOFNTmp );

    #---- Process and Add TIFF pages ----#
    if ( lc( $sOutputFormat ) eq "afp" ){
        #---- Add I_ARCHIV TLE
        #---- Create new index
        my $a2wIndexTmp = new a2w::Index();

        #---- Set index info
        $a2wIndexTmp->setName( "I_ARCHIV" );
        $a2wIndexTmp->setValue( $hrefArgs->{ 'I_ARCHIV' } );

        #---- Add index to document
        $a2wDocumentPar->addIndex( $a2wIndexTmp );
        if ( $bLog == $TRUE ){
            $theLoggerLocal->logMessage( "Added I_ARCHIV=>" . $hrefArgs->{ 'I_ARCHIV' } . "< index to document" );
        }
    }

    my ( $iRetTmp, $sMsgTmp ) = _processAndAddTIFFPages( $a2wDocumentPar, \@arrTIFFs, $hrefTIFFsInfo );
    if ( $iRetTmp < 0 ){ return ( $iRetTmp, $sMsgTmp ); }

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "initializePage" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "afp2web" );
        $theLoggerLocal->logMessage( "Id=" . $a2wPagePar->getParseId() );
    }

    #---- Skip pages of dummy AFP
    if ( $ipFormat == $IF_AFP ){ return $SKIP; }

    return $APPEND;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "finalizePage" );
    }

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "finalizeDoc()" );
    }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "finalize" );
    }

	return 0;
}

#-----------------------------------------------------------------------
# Process and add TIFF images
#
# Iterates TIFF images and does the following
# - Erases OMR lines in case of Pendel Attest document
# - Scales image in case of Non Pendel Attest document
#
# Returns
# >=0 in case of success
# <0, message in case of error
#-----------------------------------------------------------------------
sub _processAndAddTIFFPages{

    #---- Get Parameter of _processAndAddTIFFPages
    # Parameters:
    # 1. Document handle
    # 2. Array reference of TIFF file names
    # 3. Hash reference of TIFFs info
    my $a2wDocPar = shift;
    my $arefTiffFNPar = shift;
    my $hrefTIFFsInfoPar = shift;

    if ( $bLog == $TRUE ){
        $theLoggerLocal->logFunctionName( __PACKAGE__, "_processAndAddTIFFPages( $a2wDocPar, $arefTiffFNPar, $hrefTIFFsInfoPar )" );
    }

    #---- Page attribute constants
    my $fPWTmp = $PAGE_WIDTH  * $PAGE_RESOLUTION;
    my $fPHTmp = $PAGE_HEIGHT * $PAGE_RESOLUTION;

    #---- Iterate through TIFF images ----#
    my @arrTIFFsTmp = @{ $arefTiffFNPar };
    my $iPageIdTmp = 0;
    my $a2wPageTmp = undef;
    my $objWhiteBoxTmp = undef;

    my $iLMarginTmp = 0;
    my $iTMarginTmp = 0;
    my $iImgWidthTmp = 0;
    my $iImgHeightTmp = 0;

    my $iDocTypeIdTmp = 0;
    my $iSeqNrTmp     = 0;
    my $iPgIdTmp      = 0;

    my $bAFPOutputTmp = ( lc( $sOutputFormat ) eq "afp" ) ? $TRUE : $FALSE; # V104 Change

    foreach my $sTIFFFilenameTmp ( @arrTIFFsTmp ){
        #---- Assert file name
        if ( $sTIFFFilenameTmp eq "" ){ next; }

        #---- Create page
        $iPageIdTmp++;
        $a2wPageTmp = new a2w::Page();
        if ( $a2wPageTmp == undef ){ return ( -2, "Unable to create page $iPageIdTmp for $sTIFFFilenameTmp" ); }

        #---- Format page
        $a2wPageTmp->setResolution( $PAGE_RESOLUTION );
        $a2wPageTmp->setWidth( $fPWTmp );
        $a2wPageTmp->setHeight( $fPHTmp );

        #---- Evaluate document type id and sequence number
        $iDocTypeIdTmp = $hrefTIFFsInfoPar->{ $sTIFFFilenameTmp }->{ 'TypeId' };
        $iSeqNrTmp     = $hrefTIFFsInfoPar->{ $sTIFFFilenameTmp }->{ 'SeqNr' };
        $iPgIdTmp      = $hrefTIFFsInfoPar->{ $sTIFFFilenameTmp }->{ 'PageId' };
        if ( $bLog == $TRUE ){
            $theLoggerLocal->logMessage( "" ); # force empty line
            $theLoggerLocal->logMessage( "Creating output page $iPageIdTmp: InputDoc:>$iSeqNrTmp< PageId:>$iPgIdTmp< DocTypeId=>$iDocTypeIdTmp<" );
        }

        #---- Fill in page content
        # V104 Begin
        #----     External Document      ----#
        if ( $iDocTypeIdTmp == 9000 ){ # V105 Change
            $iLMarginTmp = 0;
            $iTMarginTmp = 0;
            $iImgWidthTmp  = $hrefTIFFsInfoPar->{ $sTIFFFilenameTmp }->{ 'W' };
            $iImgHeightTmp = $hrefTIFFsInfoPar->{ $sTIFFFilenameTmp }->{ 'H' };
            if ( $iImgWidthTmp < $fPWTmp ){ $iLMarginTmp = ( $fPWTmp - $iImgWidthTmp ) / 2; }
            if ( $iImgHeightTmp < $fPHTmp ){ $iTMarginTmp = ( $fPHTmp - $iImgHeightTmp ) / 2; }

            my $iPWidthTmp  = $fPWTmp - ( $iLMarginTmp * 2 );
            my $iPHeightTmp = $fPHTmp - ( $iTMarginTmp * 2 );
            my $iWScaleTmp = $iPWidthTmp / $iImgWidthTmp;
            my $iHScaleTmp = $iPHeightTmp / $iImgHeightTmp;

            #---- Evaluate aspect ratio
            if ( $iWScaleTmp < 1.0 ){
                $iPHeightTmp = ( $iImgHeightTmp / $iImgWidthTmp ) * $iPWidthTmp;
                $iHScaleTmp = $iPHeightTmp / $iImgHeightTmp;
                $iTMarginTmp = ( $fPHTmp - $iPHeightTmp ) / 2;
            }
            elsif ( $iHScaleTmp < 1.0 ){
                $iPWidthTmp = ( $iImgWidthTmp / $iImgHeightTmp ) * $iPHeightTmp;
                $iWScaleTmp  = $iPWidthTmp / $iImgWidthTmp;
                $iLMarginTmp = ( $fPWTmp - $iPWidthTmp ) / 2;
            }

            #---- Add image ----#
            $iLMarginTmp = int( $iLMarginTmp );
            $iTMarginTmp = int( $iTMarginTmp );
            $iPWidthTmp  = int( $iPWidthTmp );
            $iPHeightTmp = int( $iPHeightTmp );
            $a2wPageTmp->addImage( # Filename
                                     $sTIFFFilenameTmp
                                   # X position
                                   , $iLMarginTmp
                                   # Y position
                                   , $iTMarginTmp
                                   # Width
                                   , $iPWidthTmp
                                   # Height
                                   , $iPHeightTmp
                                   # Rotation
                                   , 0
                                 );
            if ( $bLog == $TRUE ){ $theLoggerLocal->logMessage( "img>$sTIFFFilenameTmp<@($iLMarginTmp,$iTMarginTmp), W=$iPWidthTmp, H=$iPHeightTmp, R=0" ); }

            $a2wDocPar->addPage( $a2wPageTmp, $iPageIdTmp ); # add page to document
            next; # V103 Change: Skip adding reprint message for external documents
        }
        # V104 End
        #---- Pendel Attest Document ----#
        elsif ( $iDocTypeIdTmp == 225 ){
            #---- Add image ----#
            $a2wPageTmp->addImage( # Filename
                                     $sTIFFFilenameTmp
                                   # X position
                                   , 0
                                   # Y position
                                   , 0
                                   # Width
                                   , $fPWTmp
                                   # Height
                                   , $fPHTmp
                                   # Rotation
                                   , 0
                                 );
            if ( $bLog == $TRUE ){ $theLoggerLocal->logMessage( "img>$sTIFFFilenameTmp<@(0,0), W=$fPWTmp, H=$fPHTmp, R=0" ); }

            #---- Remove OMR lines for PendelAttest documents ----#
            # Render white box@(0,0) of size (15mm, Page height)
            $objWhiteBoxTmp = new a2w::Line();
            $objWhiteBoxTmp->setXPos( 0 );
            $objWhiteBoxTmp->setYPos( 0 );
            $objWhiteBoxTmp->setWidth( $WHITE_BOX_WIDTH );
            $objWhiteBoxTmp->setLength( $WHITE_BOX_HEIGHT );
            $objWhiteBoxTmp->setVertical( $TRUE );
            $objWhiteBoxTmp->setColor( 0xFFFFFF );    # White
            $a2wPageTmp->addLine( $objWhiteBoxTmp );
            if ( $bLog == $TRUE ){ $theLoggerLocal->logMessage( "box><@(0,0), W=$WHITE_BOX_WIDTH, H=$WHITE_BOX_HEIGHT, C=0xFFFFFF" ); }
        }
        # V105 Begin
        #---- Non Pendel Attest document ----#
        else {
            $iLMarginTmp = 0;
            $iTMarginTmp = 0;
            $iImgWidthTmp  = $hrefTIFFsInfoPar->{ $sTIFFFilenameTmp }->{ 'W' };
            $iImgHeightTmp = $hrefTIFFsInfoPar->{ $sTIFFFilenameTmp }->{ 'H' };
            if ( $iImgWidthTmp < $fPWTmp ){ $iLMarginTmp = ( $fPWTmp - $iImgWidthTmp ) / 2; }
            if ( $iImgHeightTmp < $fPHTmp ){ $iTMarginTmp = ( $fPHTmp - $iImgHeightTmp ) / 2; }

            my $iPWidthTmp  = $fPWTmp - ( $iLMarginTmp * 2 );
            my $iPHeightTmp = $fPHTmp - ( $iTMarginTmp * 2 );

            #---- For non pendel attest documents, adjust left margin and page width
			#     for both PDF and AFP outputs
            #if ( $bAFPOutputTmp ){
            #    $iLMarginTmp += ( $AFP_LEFT_MARGIN / 25.4 ) * $PAGE_RESOLUTION;
            #    $iPWidthTmp  -= ( $AFP_LEFT_MARGIN / 25.4 ) * $PAGE_RESOLUTION;
            #}
            $iLMarginTmp += ( $AFP_LEFT_MARGIN / 25.4 ) * $PAGE_RESOLUTION;
            $iPWidthTmp  -= ( $AFP_LEFT_MARGIN / 25.4 ) * $PAGE_RESOLUTION;

            my $iWScaleTmp = $iPWidthTmp / $iImgWidthTmp;
            my $iHScaleTmp = $iPHeightTmp / $iImgHeightTmp;

            #---- Evaluate aspect ratio
            if ( $iWScaleTmp < 1.0 ){
                $iPHeightTmp = ( $iImgHeightTmp / $iImgWidthTmp ) * $iPWidthTmp;
                $iHScaleTmp = $iPHeightTmp / $iImgHeightTmp;
                $iTMarginTmp = ( $fPHTmp - $iPHeightTmp ) / 2;
            }
            elsif ( $iHScaleTmp < 1.0 ){
                $iPWidthTmp = ( $iImgWidthTmp / $iImgHeightTmp ) * $iPHeightTmp;
                $iWScaleTmp  = $iPWidthTmp / $iImgWidthTmp;
                $iLMarginTmp = ( $fPWTmp - $iPWidthTmp ) / 2;
            }

            #---- Add image ----#
            $iLMarginTmp = int( $iLMarginTmp );
            $iTMarginTmp = int( $iTMarginTmp );
            $iPWidthTmp  = int( $iPWidthTmp );
            $iPHeightTmp = int( $iPHeightTmp );
            $a2wPageTmp->addImage( # Filename
                                     $sTIFFFilenameTmp
                                   # X position
                                   , $iLMarginTmp
                                   # Y position
                                   , $iTMarginTmp
                                   # Width
                                   , $iPWidthTmp
                                   # Height
                                   , $iPHeightTmp
                                   # Rotation
                                   , 0
                                 );
            if ( $bLog == $TRUE ){ $theLoggerLocal->logMessage( "img>$sTIFFFilenameTmp<@($iLMarginTmp,$iTMarginTmp), W=$iPWidthTmp, H=$iPHeightTmp, R=0" ); }
        }
        # V105 End

        #---- Process and add re-print stamp for non external doc
        my ($sDFNKeyTmp) = grep { $hrefArgs->{ 'DocInfoList' }{ $_ }{ 'SeqNr' } == $iSeqNrTmp } @arrDocFilenames;
        my $hrefDocInfoTmp = $hrefArgs->{ 'DocInfoList' }{ $sDFNKeyTmp };
        if ( $hrefDocInfoTmp->{ 'RePrintMsg' } != undef ){
            my $hrefRePrintMsgTmp = $hrefDocInfoTmp->{ 'RePrintMsg' };

            #---- Get blacked out pages list
            my $sBOPListTmp = $hrefRePrintMsgTmp->{ 'BlackoutPages' };
            my @arrBOPListTmp = sort { $a <=> $b } split( /,/, $sBOPListTmp ); # split list of page ids based on comma and sort them
            s{^\s+|\s+$}{}g foreach @arrBOPListTmp;                            # Remove leading/trailing spaces of each page id
            my %hshBOPListTmp = map { $_ => $TRUE } @arrBOPListTmp;            # Create an hash having structure { <Black out page id> => $TRUE }

            #---- Determine stamp text
            # Rule used to determine which reprint message should be added to page
            # When page has at least one black out annotation (determined by <script argument json>->DocInfoList->RePrintMsg->BlackoutPages-><Page id>)
            # then use the following reprint message
            #
            # "Nachdruck aus dem Archiv. Nachdruck und Original stimmen berein (Teile des Dokuments"
            # "wurden aus Datenschutz-Grnden unkenntlich gemacht). HALLESCHE Krankenversicherung"
            # "auf Gegenseitigkeit 70166 Stuttgart"
            #
            # else use the following reprint message (default)
            #
            # "Nachdruck aus dem elektronischen Archiv. Nachdruck und Original stimmen berein."
            # "HALLESCHE Krankenversicherung auf Gegenseitigkeit 70166 Stuttgart"
            #
            my @arrRePrintStampTextTmp = (   "Nachdruck aus dem elektronischen Archiv. Nachdruck und Original stimmen berein."
                                           , "HALLESCHE Krankenversicherung auf Gegenseitigkeit 70166 Stuttgart"
                                         );

            # If the pageId is in the list of pages having black out annotations then add following text to the page
            if ( %hshBOPListTmp->{ $iPgIdTmp } == $TRUE ){
                # V103 Change: Modified (replaced "Name und Adresse des Gutachters wurden geschwrzt" with "Teile des Dokuments wurden aus
                #              Datenschutz-Grnden unkenntlich gemacht") reprint message for black out pages
                @arrRePrintStampTextTmp = (   "Nachdruck aus dem Archiv. Nachdruck und Original stimmen berein (Teile des Dokuments"
                                            , "wurden aus Datenschutz-Grnden unkenntlich gemacht). HALLESCHE Krankenversicherung"
                                            , "auf Gegenseitigkeit 70166 Stuttgart"
                                          );
            }
            if ( $bDebugLog == $TRUE ){ $theLoggerLocal->logMessage( "BlackoutPages:>$sBOPListTmp< PageId:>$iPgIdTmp< RePrintStamp: " . ( (%hshBOPListTmp->{ $iPgIdTmp } == $TRUE) ? "Content blacked out" : "Same as Original" ) ); }

            my $iXPosTmp = ( # Points
                             250.0 / 72.0
                           ) * $AFP_PTX_RESOLUTION;
            my $iYPosTmp = ( # Points
                             17.0 / 72.0
                           ) * $AFP_PTX_RESOLUTION;
            my $iLineIncrTmp = ( ( $fntRePrintStamp->getHeight() + 2 ) / 72.0 ) * $AFP_PTX_RESOLUTION;
            my $txtRPStampTmp = undef;

            foreach my $sStampLineTmp ( @arrRePrintStampTextTmp ){
                $txtRPStampTmp = new a2w::Text();
                $txtRPStampTmp->setXPos( $iXPosTmp );
                $txtRPStampTmp->setYPos( $iYPosTmp );
                $txtRPStampTmp->setText( $sStampLineTmp );
                $txtRPStampTmp->setFont( $fntRePrintStamp );

                #---- Add text to page
                $a2wPageTmp->addText( $txtRPStampTmp );
                if ( $bLog == $TRUE ){ $theLoggerLocal->logMessage( "txt>$sStampLineTmp<@($iXPosTmp,$iYPosTmp), R=0" ); }

                $iYPosTmp += $iLineIncrTmp;
            }
        }

        #---- Add page to document
        $a2wDocPar->addPage( $a2wPageTmp, $iPageIdTmp );
    }

    return 0;
}

__END__
