#-------------------------------------------------------------------------------
#  a2w/core/dm/Database.pm
#
#  Perl module to hold page content as database for data mining
#
#  Author   : Panneer, AFP2web Team
#
#  $V100   2014-05-08    Initial Release
#
#  $V101   2015-07-20    Fixed minor bug in evaluating output file name when page
#                        output is turned on
#
#  $V102   2015-11-17    a. Extended with page reference attribute for database
#                        b. Extended with "object sequence id" attribute for objects
#
#  $V103   2018-01-10    Extended with container objects processing (AFP-456, AFP-623)
#
#  $V104   2018-08-09    - AFP-735: Extended to handle tagging for reusable objects (with different tag info
#                          at each presentation of same object)
#                        - AFP-734: Extended to write tagging info at document level instead of page level
#
#  $V105   2018-09-04    Extended with API to get objects of a region
#
#  $V106   2018-10-03    AFP-743: Extended with composite block
#
#  $V110   2018-10-26    a. AFP-756: Preprocessed page content to have lines with Y tolerance
#                        b. AFP-756: Extended with data mining APIs to find and fetch eyecatcher/value
#
#-------------------------------------------------------------------------------
package a2w::core::dm::Database;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    my $this = {
          'ObjectCount'    => 0        # Total Object Count
        , 'MaxXPos'        => 0        # Maximum X position value
        , 'MinXPos'        => 0        # Minimum X position value
        , 'MaxYPos'        => 0        # Maximum Y position value
        , 'MinYPos'        => 0        # Minimum Y position value
        , 'PageID'         => 0        # Page identifier from which objects were filled
        , 'PageWidth'      => 0        # Page width
        , 'PageHeight'     => 0        # Page height
        , 'PageRes'        => 0        # Page resolution
        , 'PageBackground' => ""       # Page background filename
        , 'hshObj'         => {}       # Hash of Object references
        , 'hshObjInfo'     => {}       # Hash of object specific info
        , 'hshXPos'        => {}       # X position hash
        , 'hshYPos'        => {}       # Y position hash
        , 'hshAngle'       => {}       # Angle hash
        , 'ResTable'       => {}       # Resource table
        , 'PageFilename'   => ''       # Page output file name    # V101 Change
        , 'PageReference'  => undef    # Reference to core page   # V102 Change
        , 'ResId'          => 0        # Active resource id       # V104 Change
        , 'PageObjects'    => undef    # Array reference of page objects sorted (top to bottom and right to left) # V107 Change
        , 'Lines'          => undef    # Array reference of lines # V107 Change
    };

    bless( $this, $class );

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){
    #    $theLogger->logFunctionName( __PACKAGE__, "new()" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Page id
    #
    if ( @_ ){
        $this->{ 'PageID' } = shift;
    }
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------
sub setPageID{
    my $this = shift;
    $this->{ 'PageID' } = shift;
}

sub setPageWidth{
    my $this = shift;
    $this->{ 'PageWidth' } = shift;
}

sub setPageHeight{
    my $this = shift;
    $this->{ 'PageHeight' } = shift;
}

sub setPageRes{
    my $this = shift;
    $this->{ 'PageRes' } = shift;
}

sub setPageBackground{
    my $this = shift;
    $this->{ 'PageBackground' } = shift;
}

# $V101 Begin
sub setPageFilename{
    my $this = shift;
    $this->{ 'PageFilename' } = shift;
}
# $V101 End

# $V102 Begin
sub setPageReference{
    my $this = shift;
    $this->{ 'PageReference' } = shift;
}
# $V102 End

# $V104 Begin
sub setResId{
    my $this = shift;
    $this->{ 'ResId' } = shift;
}
# $V104 End

# V107 Begin
sub setPageObjects{
    my $this = shift;
    $this->{ 'PageObjects' } = shift;
}

sub setLines{
    my $this = shift;
    $this->{ 'Lines' } = shift;
}
# V107 End

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------
sub getObjectCount{
    my $this = shift;
    return $this->{ 'ObjectCount' };
}

sub getMaxXPos{
    my $this = shift;
    return $this->{ 'MaxXPos' };
}

sub getMinXPos{
    my $this = shift;
    return $this->{ 'MinXPos' };
}

sub getMaxYPos{
    my $this = shift;
    return $this->{ 'MaxYPos' };
}

sub getMinYPos{
    my $this = shift;
    return $this->{ 'MinYPos' };
}

sub getPageID{
    my $this = shift;
    return $this->{ 'PageID' };
}

sub getPageWidth{
    my $this = shift;
    return $this->{ 'PageWidth' };
}

sub getPageHeight{
    my $this = shift;
    return $this->{ 'PageHeight' };
}

sub getPageRes{
    my $this = shift;
    return $this->{ 'PageRes' };
}

sub getPageBackground{
    my $this = shift;
    return $this->{ 'PageBackground' };
}

sub getHashObj{
    my $this = shift;
    return \%{ $this->{ 'hshObj' } };
}

sub getHashObjInfo{
    my $this = shift;
    return \%{ $this->{ 'hshObjInfo' } };
}

sub getHashXPos{
    my $this = shift;
    return \%{ $this->{ 'hshXPos' } };
}

sub getHashYPos{
    my $this = shift;
    return \%{ $this->{ 'hshYPos' } };
}

sub getHashAngle{
    my $this = shift;
    return \%{ $this->{ 'hshAngle' } };
}

sub getResourceTable{
    my $this = shift;
    return \%{ $this->{ 'ResTable' } };
}

# $V101 Begin
sub getPageFilename{
    my $this = shift;
    return $this->{ 'PageFilename' };
}
# $V101 End

# $V102 Begin
sub getPageReference{
    my $this = shift;
    return $this->{ 'PageReference' };
}
# $V102 End

# $V104 Begin
sub getResId{
    my $this = shift;
    return $this->{ 'ResId' };
}
# $V104 End

# V107 Begin
sub getPageObjects{
    my $this = shift;
    return $this->{ 'PageObjects' };
}

sub getLines{
    my $this = shift;
    return $this->{ 'Lines' };
}
# V107 End

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Add object
#-----------------------------------------------------------------------
sub addObject{
    my $this = shift;

    #---- Get parameters
    #
    # 1. Object Type (Text, Line so on)
    # 2. X Position
    # 3. Y Position
    # 4. Angle
    # 5. Color
    # 6. Object Reference
    my (   $ObjectTypePar
         , $iXPosPar
         , $iYPosPar
         , $iAnglePar
         , $iColorPar
         , $a2wObjectPar
    ) = @_;

    if ( $bLog == $TRUE ){
        my $sObjTypeTmp = "";
        my $sObjInfoStringTmp = "";

        if ( $ObjectTypePar == $a2w::core::dm::Constants::OT_TEXT ){
            $sObjTypeTmp = "Text";
            $sObjInfoStringTmp = " Length=" . $a2wObjectPar->getTextLen()
                               . " Text=>"  . $a2wObjectPar->getText() . "<";
        }
        elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_LINE ){
            $sObjTypeTmp = "Line";
            $sObjInfoStringTmp = " Width="  . $a2wObjectPar->getWidth()
                               . " Length=" . $a2wObjectPar->getLength();
        }
        elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_VECTOR ){
            $sObjTypeTmp = "Vector";
            $sObjInfoStringTmp = " Width="  . $a2wObjectPar->getWidth()
                               . " Height=" . $a2wObjectPar->getHeight();
        }
        elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_IMAGE ){
            $sObjTypeTmp = "Image";
            $sObjInfoStringTmp =   " Name="  . $a2wObjectPar->getName() # $V103 Change
                                 . " Width="  . $a2wObjectPar->getWidth()
                                 . " Height=" . $a2wObjectPar->getHeight();
        }
        # $V103 Begin
        elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_CONTAINER ){
            $sObjTypeTmp = "Container";
            $sObjInfoStringTmp =   " Name="  . $a2wObjectPar->getName()
                                 . " ObjectType="  . $a2wObjectPar->getObjectTypeName()
                                 . " Width="  . $a2wObjectPar->getWidth()
                                 . " Height=" . $a2wObjectPar->getHeight();
        }
        # $V103 End
        $theLogger->logFunctionName( __PACKAGE__, "addObject: " . $sObjTypeTmp . " @(" . $iXPosPar . ", " . $iYPosPar . ") Rot=" . $iAnglePar . $sObjInfoStringTmp );
    }

    #---- Create object of database
    $this->{ 'ObjectCount' }++;         #---- Increment object count

    #---- Set maximum & minimum X/Y positions
    #---- Initialize, if it is first time
    if ( $this->{ 'ObjectCount' } == 1 ){
        $this->{ 'MaxXPos' } = $iXPosPar;
        $this->{ 'MinXPos' } = $iXPosPar;
        $this->{ 'MaxYPos' } = $iYPosPar;
        $this->{ 'MinYPos' } = $iYPosPar;
    }
    else {
        if ( $this->{ 'MaxXPos' } < $iXPosPar ){
            $this->{ 'MaxXPos' } = $iXPosPar;
        }

        if ( $this->{ 'MinXPos' } > $iXPosPar ){
            $this->{ 'MinXPos' } = $iXPosPar;
        }

        if ( $this->{ 'MaxYPos' } < $iYPosPar ){
            $this->{ 'MaxYPos' } = $iYPosPar;
        }

        if ( $this->{ 'MinYPos' } > $iYPosPar ){
            $this->{ 'MinYPos' } = $iYPosPar;
        }
    }

    my $iObjSequenceIDTmp = $this->{ 'ObjectCount' };

    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Object Sequence ID:>" . $iObjSequenceIDTmp . "<" );
    }

    #---- Set object reference on objects list
    $this->{ 'hshObj' }{ $iObjSequenceIDTmp }{ 'OBJECT' } = $a2wObjectPar;

    #---- Get object reference
    my $hrefObjectTmp = \%{ $this->{ 'hshObj' }{ $iObjSequenceIDTmp } };

    #---- Set attribute references on object database
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE } = $ObjectTypePar;

    # $V102 Begin
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJSEQID } = $iObjSequenceIDTmp; # Object sequence id
    # $V102 End

    # V104 Begin
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_ID } = $this->getResId();
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_ID } <<= 16;
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_ID } |= $a2wObjectPar->getSequenceId();
    # V104 End

    #---- Refer page (POM) on object
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_PAGEREF } = $this; # V106 Change

    if ( $ObjectTypePar == $a2w::core::dm::Constants::OT_TEXT ){
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO } = {
              $a2w::core::dm::Constants::OI_TEXT_VALUE  => $a2wObjectPar->getText()     # value
            , $a2w::core::dm::Constants::OI_TEXT_LENGTH => $a2wObjectPar->getTextLen()  # length
        };

        #---- Update resource table with text applied font resource info ----#
        #---- Fetch text font
        my $a2wFontTmp = $a2wObjectPar->getFont();
        if ( $a2wFontTmp != undef ){ # V106 Change
            #---- Update font table, if font is added to it already
            my $iFontLocalIdTmp = $this->{ 'PageID' } . $a2wObjectPar->getMappedFontLocalId();
            $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTID } = $iFontLocalIdTmp;
            $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO }{ $a2w::core::dm::Constants::OI_TEXT_FONTHT } = $a2wFontTmp->getHeight();

            #---- Update resource table with text used font
            if ( not defined( $this->{ 'ResTable' }{ $iFontLocalIdTmp } ) ){
                $this->updateResourceTable( # Unique resource id
                                               $iFontLocalIdTmp
                                            # Resource
                                            , $a2wFontTmp
                                            # Resource type
                                            , $a2w::core::dm::Constants::RT_FONT
                                            # Object reference
                                            , $a2wObjectPar
                                          );
            }
        } # V106 Change
    }
    elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_LINE ){
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO } = {
              $a2w::core::dm::Constants::OI_LINE_WIDTH  => $a2wObjectPar->getWidth()    # width
            , $a2w::core::dm::Constants::OI_LINE_LENGTH => $a2wObjectPar->getLength()   # length
        };
    }
    elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_VECTOR ){
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO } = {
              $a2w::core::dm::Constants::OI_VECTOR_WIDTH  => $a2wObjectPar->getWidth()  # width
            , $a2w::core::dm::Constants::OI_VECTOR_HEIGHT => $a2wObjectPar->getHeight() # height
        };
    }
    elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_IMAGE ){
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO } = {
              $a2w::core::dm::Constants::OI_IMAGE_WIDTH  => $a2wObjectPar->getWidth()          # width
            , $a2w::core::dm::Constants::OI_IMAGE_HEIGHT => $a2wObjectPar->getHeight()         # height
            , $a2w::core::dm::Constants::OI_IMAGE_NAME => $a2wObjectPar->getName()             # name
            , $a2w::core::dm::Constants::OI_IMAGE_RESOLUTION => $a2wObjectPar->getResolution() # resolution
        };
    }
    # $V103 Begin
    elsif ( $ObjectTypePar == $a2w::core::dm::Constants::OT_CONTAINER ){
        $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO } = {
              $a2w::core::dm::Constants::OI_CONTAINER_WIDTH  => $a2wObjectPar->getWidth()              # width
            , $a2w::core::dm::Constants::OI_CONTAINER_HEIGHT => $a2wObjectPar->getHeight()             # height
            , $a2w::core::dm::Constants::OI_CONTAINER_NAME => $a2wObjectPar->getName()                 # name
            , $a2w::core::dm::Constants::OI_CONTAINER_RESOLUTION => $a2wObjectPar->getResolution()     # resolution
            , $a2w::core::dm::Constants::OI_CONTAINER_OBJECTTYPE => $a2wObjectPar->getObjectTypeName() # object type
        };
    }
    # $V103 End
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_XPOS       } = $iXPosPar;
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_YPOS       } = $iYPosPar;
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_ANGLE      } = ( $iAnglePar == 360 ) ? 0 : $iAnglePar;
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_COLOR      } = $iColorPar;
    $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_RESOLUTION } = $this->{ 'PageRes' };

    #---- Add attributes
    $this->{ 'hshObjInfo' }{ $ObjectTypePar }{ $iObjSequenceIDTmp } = $hrefObjectTmp->{ $a2w::core::dm::Constants::AT_OBJINFO };
    $this->{ 'hshXPos' }{ $iXPosPar }{ $iObjSequenceIDTmp }         = $hrefObjectTmp;
    $this->{ 'hshYPos' }{ $iYPosPar }{ $iObjSequenceIDTmp }         = $hrefObjectTmp;
    $this->{ 'hshAngle' }{ $iAnglePar }{ $iObjSequenceIDTmp }       = $hrefObjectTmp;

    return $iObjSequenceIDTmp; # V106 Change
}

# V105 Begin
# Removed old data mining APIs
# - getObjects
# - getFirstResultSet
# - getNextResultSet
# - getResultSet
# V105 End

#-----------------------------------------------------------------------
# Update resource table
#-----------------------------------------------------------------------
sub updateResourceTable{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "updateResourceTable()" );
    }

    #---- Fetch parameters
    #
    # 1. Unique resource id
    # 2. Resource reference
    # 3. Resource type
    # 4. Object reference
    #
    my $iUniqueResIdPar = shift;
    my $a2wResourcePar  = shift;
    my $iResTypePar     = shift;
    my $a2wObjectPar    = shift;

    #---- Get resource table reference
    my $hrefResTableTmp = $this->{ 'ResTable' };

    #---- Update resource table with given resource info
    $hrefResTableTmp->{ $iUniqueResIdPar } = {
          'NAME'      => $a2wResourcePar->getName()
        , 'TYPE'      => $iResTypePar
    };

    if ( $iResTypePar == $a2w::core::dm::Constants::RT_FONT ){
        $hrefResTableTmp->{ $iUniqueResIdPar }{ 'FONT' } = {
              'BOLD'      => $a2wResourcePar->isBold()
            , 'ITALIC'    => $a2wResourcePar->isItalic()
            , 'HEIGHT'    => $a2wResourcePar->getHeight()
            , 'FAMILY'    => $a2wResourcePar->getTypefaceName()
        };
    }
    elsif ( $iResTypePar == $a2w::core::dm::Constants::RT_PSEG ){
        $hrefResTableTmp->{ $iUniqueResIdPar }{ 'PSEG' } = {
              'XPOS'      => $a2wResourcePar->getIncludedXPosition()
            , 'YPOS'      => $a2wResourcePar->getIncludedYPosition()
        };
    }
    elsif ( $iResTypePar == $a2w::core::dm::Constants::RT_OVERLAY ){
        $hrefResTableTmp->{ $iUniqueResIdPar }{ 'OVERLAY' } = {
              'XPOS'      => $a2wResourcePar->getIncludedXPosition()
            , 'YPOS'      => $a2wResourcePar->getIncludedYPosition()
            , 'WIDTH'     => $a2wResourcePar->getWidth()
            , 'HEIGHT'    => $a2wResourcePar->getHeight()
        };
    }

    if ( $bLog == $TRUE ){
        if ( $iResTypePar == $a2w::core::dm::Constants::RT_FONT ){
            $theLogger->logMessage(   "Resource Details:"
                                    . " Name=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'NAME' }
                                    . " Type=Font"
                                    . " Bold=" . ( ( $hrefResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'BOLD' } == $TRUE ) ? "Y" : "N" )
                                    . " Italic=" . ( ( $hrefResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'ITALIC' } == $TRUE ) ? "Y" : "N" )
                                    . " Height=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'HEIGHT' }
                                    . " Family=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'FONT' }{ 'FAMILY' }
                                  );
        }
        elsif ( $iResTypePar == $a2w::core::dm::Constants::RT_PSEG ){
            $theLogger->logMessage(   "Resource Details:"
                                    . " Name=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'NAME' }
                                    . " Type=Page Segment"
                                    . " XPos=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'PSEG' }{ 'XPOS' }
                                    . " YPos=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'PSEG' }{ 'YPOS' }
                                  );
        }
        elsif ( $iResTypePar == $a2w::core::dm::Constants::RT_OVERLAY ){
            $theLogger->logMessage(   "Resource Details:"
                                    . " Name=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'NAME' }
                                    . " Type=Overlay"
                                    . " XPos=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'OVERLAY' }{ 'XPOS' }
                                    . " YPos=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'OVERLAY' }{ 'YPOS' }
                                    . " Width=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'OVERLAY' }{ 'WIDTH' }
                                    . " Height=" . $hrefResTableTmp->{ $iUniqueResIdPar }{ 'OVERLAY' }{ 'HEIGHT' }
                                  );
        }
    }

    return 0;
}

# V105 Begin
#-----------------------------------------------------------------------
# Get region objects
#
# Collect and return objects that fall in given region
#
# Parameter
# Region    Hash Reference    Object having following structure
#                             {
#                               X: Starting X position of region on page
#                               Y: Starting Y position of region on page
#                               W: Width of region
#                               H: Height of region
#                             }
#
# Returns list of objects as array reference else undef
#
#-----------------------------------------------------------------------
sub getRegionObjects{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "getRegionObjects()" );
    }

    #---- Fetch parameters
    #
    # 1. Region
    #
    my $hrefRegionPar = shift;

    #---- Assert objects
    if (    $hrefRegionPar->{ 'X' } < 0
         || $hrefRegionPar->{ 'Y' } < 0
         || $hrefRegionPar->{ 'W' } <= 0
         || $hrefRegionPar->{ 'H' } <= 0
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Invalid region values (X=" . $hrefRegionPar->{ 'X' } . ", Y=" . $hrefRegionPar->{ 'Y' } . ", W=" . $hrefRegionPar->{ 'W' } . ", H=" . $hrefRegionPar->{ 'H' } . ") to get objects, returning undef" ); }
        return undef;
    }

    #---- Get page objects
    my $hrefYPosTmp = $this->getHashYPos();
    my $hrefObjectsTmp = $this->getHashObj();

    #---- Sort Y positions
    my @arrSortedYPositionsTmp = sort { $a <=> $b } keys( %{ $hrefYPosTmp } );

    #---- Filter region specific Y positions
    my $startY = $hrefRegionPar->{ 'Y' };
    my $endY   = $startY + $hrefRegionPar->{ 'H' };
    my @arrRegionYPositionsTmp = grep { $_ >= $startY && $_ <= $endY } @arrSortedYPositionsTmp;
    if ( @arrSortedYPositionsTmp <= 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "No objects fall in region vertical range (StartY=" . $startY . ", EndY=" . $endY . "), returning undef" ); }
        return undef;
    }
    
    #---- Iterate through sorted Y positions
    my $iRegionObjsCountTmp = 0;
    my @arrRegionObjsTmp = ();
    my $hrefObjectTmp = undef;
    my $startX = $hrefRegionPar->{ 'X' };
    my $endX   = $startY + $hrefRegionPar->{ 'W' };
    foreach my $fYPosTmp ( @arrRegionYPositionsTmp ){
        #---- Sort X positions (i.e, sort multiple objects based on X positions that occur on same Y position aka line)
        my @arrSortedXPositionsTmp = sort { $hrefYPosTmp->{ $fYPosTmp }{ $a }{ $a2w::core::dm::Constants::AT_XPOS } <=> $hrefYPosTmp->{ $fYPosTmp }{ $b }{ $a2w::core::dm::Constants::AT_XPOS } } keys( %{ $hrefYPosTmp->{ $fYPosTmp } } );

        #---- Filter region specific Y positions
        my @arrRegionXPositionsTmp = grep { $_ >= $startX && $_ <= $endX } @arrSortedXPositionsTmp;
        if ( @arrRegionXPositionsTmp <= 0 ){ next; }

        #---- Iterate through sorted X positions
        foreach $x ( @arrRegionXPositionsTmp ){
            #---- Get object reference
            $hrefObjectTmp = $hrefYPosTmp->{ $fYPosTmp }{ $x };

            #---- Add object to return list
            @arrRegionObjsTmp[ $iRegionObjsCountTmp++ ] = {
                  'A2WOBJ' => $hrefObjectsTmp->{ $x }{ 'OBJECT' }
                , 'POMOBJ' => $hrefObjectTmp
            };
        }
    }

    if ( @arrRegionObjsTmp <= 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "No objects fall in region horizontal range (StartX=" . $startX . ", EndX=" . $endX . "), returning undef" ); }
        return undef;
    }

    return \@arrRegionObjsTmp;
}
# V105 End

# V106 Begin
#-----------------------------------------------------------------------
# Get POM object
#
# Returns POM object in case of success or undef in case of error
#-----------------------------------------------------------------------
sub getPOMObject{
    my $this = shift;

    #---- Get parameters
    #
    # 1. Object sequence id
    my $objSequenceIdPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getPOMObject(" . $objSequenceIdPar . ")" ); }

    #---- Get POM object
    my $hrefObjectTmp = $this->{ 'hshObj' }{ $objSequenceIdPar };

    return $hrefObjectTmp;
}
# V106 End

# V107 Begin
#-----------------------------------------------------------------------
# Preprocess page content
#
# Sorts page content in top to bottom and left to right reading order
# with minor tolerance in Y position
#
# Updates database with array of lines where in each line will again be
# an array of objects sorted based on X position
#
# Returns
# >=0 in case of success
# <0 in case of error
#-----------------------------------------------------------------------
sub preProcess{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "preProcess()" ); }

    #---- Process objects by Y position (top to bottom) ----#
    my $hrefObjectsTmp = $this->getHashObj();

    #---- Sort Y positions
    my $hrefYPosTmp    = $this->getHashYPos();
    my @arrSortedYPositionsTmp = sort { $a <=> $b } keys( %{ $hrefYPosTmp } );
    if ( @arrSortedYPositionsTmp <= 0 ){ return -1; }

    #---- Dummy Y position added to collect last line of page in single loop
    my $iToleranceTmp  = $a2w::core::dm::Constants::CONT_LINE_TOLERANCE * $this->getPageRes();
    push( @arrSortedYPositionsTmp, ( $arrSortedYPositionsTmp[ $#arrSortedYPositionsTmp ] + $iToleranceTmp + 1 ) );

    #---- Iterate through sorted Y positions
    my $fYTmp          = 0;
    my $fCurYTmp       = $arrSortedYPositionsTmp[ 0 ];
    my $arefLineTmp    = [];
    my @arrLinesTmp    = ();
    my @arrPgObjsTmp   = ();
    my @arrYOIdsTmp    = ();
    for ( my $i = 0; $i < @arrSortedYPositionsTmp; $i++ ){
        $fYTmp = $arrSortedYPositionsTmp[ $i ]; # fetch current Y position

        #---- Group objects having minor Y tolerance
        my @arrLineTmp = @{ $arefLineTmp };
        if ( $fYTmp > ( $fCurYTmp + $iToleranceTmp ) && @arrLineTmp > 0 ){
            #---- Process current line
            # a. sort objects by X position
			@arrLineTmp = sort { $a->{ $a2w::core::dm::Constants::AT_XPOS } <=> $b->{ $a2w::core::dm::Constants::AT_XPOS } } @arrLineTmp;

            # b. Update object with processing values
            #    - Set adjusted Y position as current Y position
            #    - Set page id
			foreach ( @arrLineTmp ){
			    $_->{ $a2w::core::dm::Constants::AT_ADJ_YPOS } = $fCurYTmp;
			    $_->{ $a2w::core::dm::Constants::AT_PAGE_ID } = $this->{ 'PageID' };
			}

            # c. Store line in array of lines
            @arrLinesTmp[ ( $#arrLinesTmp + 1 ) ] = \@arrLineTmp;
            foreach my $o ( @arrLineTmp ){ push( @arrPgObjsTmp, { 'A2WOBJ' => $hrefObjectsTmp->{ $o->{ $a2w::core::dm::Constants::AT_OBJSEQID } }{ 'OBJECT' }, 'POMOBJ' => $o } ); }

            #---- Start new line
            $fCurYTmp = $fYTmp;
            $arefLineTmp = [];
        } # if ( $fYTmp > ( $fCurYTmp + $iToleranceTmp ) )

        #---- Collect objects that fall in current Y position
        @arrYOIdsTmp = keys( %{ $hrefYPosTmp->{ $fYTmp } } );
        foreach my $oid ( @arrYOIdsTmp ){ $arefLineTmp->[ ( $#{ $arefLineTmp } + 1 ) ] = $hrefYPosTmp->{ $fYTmp }{ $oid }; }
    } # for ( my $i = 0; $i < @arrSortedYPositionsTmp; $i++ )

    #---- Process collected lines
    if ( @arrLinesTmp > 0 ){
        #---- Store collected lines
        $this->setPageObjects( \@arrPgObjsTmp );
        $this->setLines( \@arrLinesTmp );

        #---- Mark page first and last objects for page begin/end marking
        my @arrFirstLineTmp = @{ $arrLinesTmp[ 0 ] };
        my @arrLastLineTmp  = @{ $arrLinesTmp[ $#arrLinesTmp ] };

        $arrFirstLineTmp[ 0 ]->{ $a2w::core::dm::Constants::AT_PAGEFIRST } = $TRUE;
        $arrLastLineTmp[ $#arrLastLineTmp ]->{ $a2w::core::dm::Constants::AT_PAGELAST } = $TRUE;

        #---- Log lines
        #if ( $bLog == $TRUE ){
        #    $theLogger->logMessage( 'Page ' . $this->{ 'PageID' } . ' lines: ' . @arrLinesTmp );
        #    for ( my $l = 0; $l < @arrLinesTmp; $l++ ){
        #        $theLogger->logMessage( "Line $l:" );
        #        $theLogger->logHashMessage( { 'Line' => $arrLinesTmp[ $l ] } );
        #    }
        #}
    }

    return 0;
}

#-----------------------------------------------------------------------
# findEyecatcher
#
# Find eyecatcher on page content
#
# Parameters:
# options     options is an object, it should be defined as follow
#             "options":{
#               'xRange':        {'from':0,'to':100000}, // where xRange, from and to are optional
#               'yRange':        {'from':0,'to':100000}, // where yRange, from and to are optional
#               'reEC':          <regular expression>,   // eyecatcher regexpr
#               'start':         OPTIONAL: specify from which array index the search should start from, default is 0
#               'end':           OPTIONAL: specify up to which array index the search should go, default is maximum of content array
#               'objectType':    Type of eyecatcher object (text, line, vector, image, container). Default is text
#             }
#                  
# Prototypes:
#  ecIndexName = findEyecatcher(content, options)   where options is as defined above
# 
# Example:
#  ecIndexName = findEyecatcher(content,{'reEC' => qr/^Datum.*$/i})
#-----------------------------------------------------------------------
sub findEyecatcher{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "findEyecatcher()" ); }

    #---- Get page objects
    my $arefObjsTmp = $this->getPageObjects();
    my @arrObjsTmp = @{ $arefObjsTmp };
    if ( @arrObjsTmp <= 0 ){ return undef; }

    return a2w::core::dm::MiningUtils::findEyecatcher( $arefObjsTmp, @_ );
}

#-----------------------------------------------------------------------
# findEyecatcherValue
#
# Find eyecatcher value on page content
#
# Parameters:
# pageid      Page id. Used to get page content for searching eyecatcher/value.
# options     options is an object, it should be defined as follow
#             "options":{
#               'xRange':        {'from':0,'to':100000}, // where xRange, from and to are optional
#               'yRange':        {'from':0,'to':100000}, // where yRange, from and to are optional
#               'reEC':          <regular expression>,   // eyecatcher regexpr
#               'reECValue':     <regular expression>,   // eyecatcher value regexpr
#               'start':         OPTIONAL: specify from which array index the search should start from, default is 0
#               'end':           OPTIONAL: specify up to which array index the search should go, default is maximum of content array
#               'objectType':    Type of eyecatcher object (text, line, vector, image, container). Default is text
#               'direction':     Search direction for the eyecatcher value (right, left, top and bottom). Default is right
#               'xTolerance':    OPTIONAL: Specify adjustment in X range to search value. Default is 1
#               'yTolerance':    OPTIONAL: Specify adjustment in Y range to search value. Default is 1
#             }
#                  
# Prototypes:
#  ecIndexValue = findEyecatcherValue(content, options)   where options is as defined above
# 
# Example:
#  ecIndexValue = findEyecatcherValue(content,{'reEC' => qr/^Datum.*$/i, 'reECValue' => qr/^(\d{2}.\d{2}.\d{4})$/i})
#-----------------------------------------------------------------------
sub findEyecatcherValue{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "findEyecatcherValue()" ); }

    #---- Get page objects
    my $arefObjsTmp = $this->getPageObjects();
    my @arrObjsTmp = @{ $arefObjsTmp };
    if ( @arrObjsTmp <= 0 ){ return undef; }

    return a2w::core::dm::MiningUtils::findEyecatcherValue( $arefObjsTmp, @_ );
}
# V107 End

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

