#-------------------------------------------------------------------------------
#  a2w::core::visitor::JSONVisitor.pm:
#  PERL module to generate JSON output
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2018-08-31    Initial Release
#
#  $V101   2019-01-28    a. OXS-8853: Extended to store output info on block
#
#-------------------------------------------------------------------------------
package a2w::core::visitor::JSONVisitor;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::Font;
use a2w::Text;
use a2w::Line;
use a2w::Image;
use a2w::Vector;

use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::dm::Anchor;
use a2w::core::dm::Block;
use a2w::core::visitor::Visitor;

use JSON::Tiny;

#-----------------------------------------------------------------------
# Inherit from base class
#-----------------------------------------------------------------------
our @ISA = qw( a2w::core::visitor::Visitor );

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Instantiate from base class to inherit base class attributes
    my $this = a2w::core::visitor::Visitor->new( @_ );

    bless( $this, $class );

    #---- Initialize constants ----#

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "new()" ); }
    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutator
#-----------------------------------------------------------------------
sub setUnitBase{
    my $this = shift;

    $this->{ 'UnitBase' } = shift;
    if ( lc( $this->{ 'UnitBase' } ) eq "pixel" ){
        $this->{ 'UnitBase' } = "px";
    }
    elsif ( lc( $this->{ 'UnitBase' } ) eq "mm" ){
        $this->{ 'UnitBase' } = "mm";
    }
    else {
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Invalid unit base (" . $this->{ 'UnitBase' } . ") using default px" ); }
        $this->{ 'UnitBase' } = "px";
    }
}

#-----------------------------------------------------------------------
# Accessor
#-----------------------------------------------------------------------
sub getUnitBase{
    my $this = shift;

    return $this->{ 'UnitBase' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Initialize
#-----------------------------------------------------------------------
sub initialize{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "initialize(@_)" ); }

    $this->{ 'PageOutput' } = $FALSE; # Turn off to get document level output
    $this->{ 'SkipOutput' } = $FALSE;  # Turn off to generate JSON output

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::initialize( @_ );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    # pageDump structure:
    #
    # $this->{ 'pageDump' } = { 
    #     'filename'    => '<fully qualified filename of the original PDF>', 
    #     'filesize'    => <original PDF file size>,
    #     'date'        => '<current timestamp, format: YYYYMMDD hh:mm:ss>',
    #     'pagesCount'  => <original PDF pages count>,
    #     'pages'       => [ # array of pages. A page contains an id and an array of blocks
    #       { 
    #           'id' => 1, 
    #           'blocks' => [] 
    #       }
    #     ]
    #
    # A block contains a type, a size, an array of indexes and an array of objects:
    #
    # block = {
    #     'type' => 'header', # header|shipment|bordero|footer
    #     'size' => { 
    #         'x'     => 0,
    #         'y'     => 10,
    #         'width' => 1234,
    #         'height'=> 5678
    #     },
    #     'indexes' => [
    #         {'NAME' => 'CNTEXT', 'VALUE' => 'ROLLKA'},
    #         {...}
    #     ],
    #     'objs' => [ # a2w objects: text, line & vector
    #         {...}
    #     ]
    # }
    #
    my $sSpoolFilenameTmp = $this->getSpoolFilename();
    my $iSpoolFileSizeTmp = -s $sSpoolFilenameTmp;
    my ($s,$m,$h,$D,$M,$Y,$WD,$YD,$isDST) = localtime( time );
    $this->{ 'pageDump' } = {
          'filename'    => $sSpoolFilenameTmp
        , 'filesize'    => $iSpoolFileSizeTmp
        #, 'date'        => sprintf ( "%04d%02d%02d %02d:%02d:%02d", ( $Y + 1900 ), ( $M + 1 ), $D, $h, $m, $s )
        , 'date'        => sprintf ( "%04d%02d%02d", ( $Y + 1900 ), ( $M + 1 ), $D )
        , 'pagesCount'  => 0 # <Pages count>
        , 'pages'       => [] # array of pages. A page contains an id and an array of blocks
    };

    return 0;
}

#-----------------------------------------------------------------------
# Finalize
#-----------------------------------------------------------------------
sub finalize{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "finalize()" ); }

    #---- Fetch parameters
    #
    # 1. Additional content for body end (optional)
    # 2. Additional content for footer (optional)
    #
    my $sBodyAddContentPar   = "";
    my $sAddFooterContentPar = "";
    if ( @_ > 0 ){
        $sBodyAddContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddFooterContentPar = shift;
    }

    # Convert the pageDump object to JSON
    my $jsonObjTmp = JSON::Tiny::encode_json( $this->{ 'pageDump' } );

    #---- Write to the disk
    if ( $this->{ 'bFileOpen' } == $TRUE ){
        my $out = $this->{ 'fHandle' };
        printf $out ( $jsonObjTmp );
    }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::finalize( @_ );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# Initialize Page
#-----------------------------------------------------------------------
sub initializePage{
    my $this = shift;

    #---- Fetch parameters
    #
    # 1. Database
    # 2. Title
    # 3. Additional content for header (optional)
    # 4. Additional content for before body (optional)
    #
    my $dbPagePar = shift;
    my $sTitlePar = shift;
    my $sAddHeaderContentPar = "";
    my $sAddBodyContentPar   = "";
    if ( @_ > 0 ){
        $sAddHeaderContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddBodyContentPar = shift;
    }
    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "initializePage(" . $dbPagePar->getPageID() . ")" ); }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::initializePage( $dbPagePar, $sAddHeaderContentPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    # Increase pages count
    my $pc = $this->{ 'pageDump' }->{ 'pagesCount' } + 1;
    $this->{ 'pageDump' }->{ 'pagesCount' } = $pc;

    # Create a new page
    my $pageTmp = {   
        'id' => $dbPagePar->getPageID(),    # page id
        'w' => $dbPagePar->getPageWidth(),  # page width
        'h' => $dbPagePar->getPageHeight(), # page height
        'res' => $dbPagePar->getPageRes(),  # page resolution
        'blocks' => undef                   # array of blocks
    };

    # Add the new page to the array of pages
    push @{ $this->{ 'pageDump' }->{ 'pages' } }, $pageTmp;

    return 0;
}

#-----------------------------------------------------------------------
# Finalize page
#-----------------------------------------------------------------------
sub finalizePage{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "finalizePage()" ); }

    #---- Fetch parameters
    #
    # 1. Additional content for body end (optional)
    # 2. Additional content for footer (optional)
    #
    my $sBodyAddContentPar   = "";
    my $sAddFooterContentPar = "";
    if ( @_ > 0 ){
        $sBodyAddContentPar = shift;
    }
    if ( @_ > 0 ){
        $sAddFooterContentPar = shift;
    }
    #if ( $bLog == $TRUE ){ $theLogger->logMessage( "Additional Content: Body=>" . $sBodyAddContentPar . "< Footer=>" . $sAddFooterContentPar . "<" ); }

    # ... processing statements ...

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::finalizePage( $sAddFooterContentPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# Get File Extension
#
# Returns output file extension based on visitor type. Base returns empty
# string and concrete visitors MUST return appropriate extension
#
#-----------------------------------------------------------------------
sub getFileExtension{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "getFileExtension()" ); }

    return "";
}

#-----------------------------------------------------------------------
# Update resource table
#-----------------------------------------------------------------------
sub updateResourceTable{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "updateResourceTable()" ); }

    #---- Fetch parameters
    #
    # 1. Unique resource id
    # 2. Resource reference
    # 3. Object hash (having both kernel object and pom object)
    #
    my $iUniqueResIdPar = shift;
    my $a2wResourcePar  = shift;
    my $hrefObjectPar   = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::updateResourceTable( $iUniqueResIdPar, $a2wResourcePar, $hrefObjectPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# isWritable
#
# Asserts whether given object is writeable by visitor
#
# Returns
# TRUE is visitor can write the object
# FALSE is visitor can not write the object
#
#-----------------------------------------------------------------------
sub isWritable{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "isWritable()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Object hash (having both kernel object and pom object)
    #
    my $hrefObjPar = shift;

    #---- Assert object
    my $a2wObjTmp = $hrefObjPar->{ 'A2WOBJ' };
    my $pomObjTmp = $hrefObjPar->{ 'POMOBJ' };
    my $bRetTmp   = $TRUE;

    my $iObjTypeTmp = $pomObjTmp->{ $a2w::core::dm::Constants::AT_OBJTYPE };
    unless (    $iObjTypeTmp == $a2w::core::dm::Constants::OT_TEXT
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_LINE
             || $iObjTypeTmp == $a2w::core::dm::Constants::OT_VECTOR
           ){
        $bRetTmp = $FALSE;
    }

    return $bRetTmp;
}

#---- Block related writing methods ----#
#-----------------------------------------------------------------------
# Begin block
#-----------------------------------------------------------------------
sub beginBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "beginBlock()" ); }

    #---- Fetch parameter(s)
    #
    # 1. Block
    # 2. Skip flag
    #
    my $blkToBeginPar = shift;
    my $bSkipPar = shift;

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::beginBlock( $blkToBeginPar, $bSkipPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }
    if ( $bSkipPar == $TRUE ){ return 0; }

    #---- Skip page default blocks (that are not configured)
    my $sBlkIdTmp   = $blkToBeginPar->getId();
    my $reDefPfxTmp = qr/$a2w::core::dm::Constants::BLK_PAGE_DEF_PREFIX/;
    if ( $sBlkIdTmp =~ $reDefPfxTmp ){ $this->{ 'SkipMode' } = $TRUE; return 0; }

    return 0;
}

#-----------------------------------------------------------------------
# End block
#-----------------------------------------------------------------------
sub endBlock{
    my $this = shift;

    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "endBlock()" ); }

    #---- Fetch parameter(s)
    #
    # 2. Skip flag
    #
    my $blkToEndPar = shift;
    my $bSkipPar = shift;

    #---- End skipping page default blocks (that are not configured)
    my $sBlkIdTmp   = $blkToEndPar->getId();
    my $reDefPfxTmp = qr/$a2w::core::dm::Constants::BLK_PAGE_DEF_PREFIX/;
    if ( $sBlkIdTmp =~ $reDefPfxTmp && $this->{ 'SkipMode' } == $TRUE ){ $this->{ 'SkipMode' } = $FALSE; return 0; }

    #---- Invoke base class implementation
    my $iRetTmp = $this->SUPER::endBlock( $blkToEndPar, $bSkipPar );

    return $iRetTmp;
}

#---- RAW block related writing methods ----#
#-----------------------------------------------------------------------
# Write raw objects
#-----------------------------------------------------------------------
sub writeRawObjects{
    my $this = shift;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( __PACKAGE__, "writeRawObjects()" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Array of objects
    #
    my $arefObjectsPar = shift;

    #---- Fetch objects list
    my @arrObjsTmp = @{ $arefObjectsPar };
    if ( @arrObjsTmp <= 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "No object to write, skipped writing raw objects" ); }
        return 0;
    }
    #if ( $bLog == $TRUE ){ $this->_logObjects( $arefObjectsPar ); }

    #---- Get page handle
    my $pgHandleTmp = $this->{ 'ActivePage' };

    #---- Get active block
    my $blkRawTmp = $pgHandleTmp->{ 'ActiveBlock' };

    #---- Build a JSON structure for block objects and indexes
    my @arBlocksTmp = $this->_buildJSONBlock( $arefObjectsPar );

    # Add the block to the page
    my $sBlockIdTmp = $blkRawTmp->getId();
    my $startX      = $blkRawTmp->getStartX();
    my $startY      = $blkRawTmp->getStartY();
    my $endX        = $blkRawTmp->getEndX();
    my $endY        = $blkRawTmp->getEndY();
    my $blockTmp = {
        'name' => $sBlockIdTmp,          # add block name (this is the name defined in the block definition file; we should use header|shipment|bordero|footer)
        'size' => { 
            'x'     => 0,
            'y'     => $startY,
            'width' => $pgHandleTmp->{ 'POM' }->getPageWidth(),
            'height'=> ( $endY - $startY )
        },
        'objs'    => \@arBlocksTmp              # add block objs
    };

    #---- Check whether block ended on current page or not
    if (    $sBlockIdTmp eq "shipment"
         || $sBlockIdTmp eq "bordero"
       ){
        # Block ends with below given vector object
        # Vector@(480, ....) Width=22560 Height=0
        #
        #---- Create a search anchor for block end
        my $hrefBlockEndDefTmp = {
            'Constraints' => {
                'COND1' => {
                    'OBJ_TYPE'  => 'VECTOR',
                    'CONDITION' => 'ATTR_XPOS GE 470'
                }, 
                'COND2' => {
                    'OBJ_TYPE'  => 'VECTOR',
                    'CONDITION' => 'ATTR_XPOS LE 490'
                }, 
                'COND3' => {
                    'OBJ_TYPE'  => 'VECTOR',
                    'CONDITION' => 'ATTR_WIDTH GE 22500'
                }, 
                'COND4' => {
                    'OBJ_TYPE'  => 'VECTOR',
                    'CONDITION' => 'ATTR_WIDTH LE 22600'
                }
            }
        };
        my $ancBlockEndTmp = $blkRawTmp->createAnchor( $hrefBlockEndDefTmp );

        # Search within last 10 block objects for block end object
        # If exists, then block ended on this page
        my $iObjCountTmp = @arrObjsTmp;
        my $iTailObjsCountTmp = ( $iObjCountTmp > 10 ) ? 10 : ( $iObjCountTmp - 1 );
        my @arrTailObjsTmp = @arrObjsTmp[ -$iTailObjsCountTmp .. -1 ];
        my $bBlockEndedTmp = $FALSE;
        foreach my $to ( @arrTailObjsTmp ){
            $bBlockEndedTmp = $ancBlockEndTmp->doesConstraintsMatch( $to->{ 'A2WOBJ' }, $to->{ 'POMOBJ' } );
            if ( $bBlockEndedTmp == $TRUE ){ last; } # break the loop
        }
        if ( $bBlockEndedTmp == $TRUE ){
            $blockTmp->{ 'multipage' } = 'false';
        }
        else {
            $blockTmp->{ 'multipage' } = 'true';
        }
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Block: multipage:>" . $blockTmp->{ 'multipage' } . "<" ); }
    }

    #---- Add indexes to the block
    my $arefIndexesTmp = $blkRawTmp->getIndexes();
    my @arrIndexesTmp = @{ $arefIndexesTmp };
    if ( @arrIndexesTmp > 0 ){ $blockTmp->{ 'indexes' } = $arefIndexesTmp; }
 
    # Add the page blocks to the pageDump object
    my $pageIdTmp = $this->{ 'pageDump' }->{ 'pagesCount' } -1; # array index starts from 0
    my $pageTmp   = $this->{ 'pageDump' }->{ 'pages' }[$pageIdTmp];    
    push @{ $pageTmp->{ 'blocks' } }, $blockTmp;

    #---- Set output context on block
    $blkRawTmp->setOutputContext( $blockTmp ); # V101 Change

    return 0;
}

#-----------------------------------------------------------------------
# Build json structure from block
#
# Parameters:
# Array of block objects
#
# Returns JSON object in following format
#
# block = {
#     'type' => 'header', # header|shipment|bordero|footer
#     'size' => { 
#         'x'     => 0,
#         'y'     => 10,
#         'width' => 1234,
#         'height'=> 5678
#     },
#     'objs' => [ # a2w objects: text, line & vector
#         {...}
#     ]
# }
#
#-----------------------------------------------------------------------
sub _buildJSONBlock{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Array of objects hash (having both kernel object and pom object)
    #
    my $arefObjsPar = shift;
    my @arrObjsTmp  = @{ $arefObjsPar };

    #---- Iterate objects and log them
    my $sObjTypeTmp = "";
    my $a2wObjTmp = undef;
    my $pomObjTmp = undef;
    my @arBlocksTmp; # array of blocks
    my $xpos = -1;
    my $objTmp; # the object to be pushed on the block array
    my $a2wFontTmp = undef; # a2w font object

    foreach my $elem ( @arrObjsTmp ){
        #---- Get object details
        $a2wObjTmp   = $elem->{ 'A2WOBJ' };
        $pomObjTmp   = $elem->{ 'POMOBJ' };
        $sObjTypeTmp = $a2wObjTmp->_getType();

        $objTmp = {}; # reset
        $objTmp->{'type'}   = $sObjTypeTmp;
        $objTmp->{'x'}      = $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS };
        $objTmp->{'y'}      = $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS };
        if ( $sObjTypeTmp eq "text" ){
            # Reference: see a2w/core/ext/Text.pm
            # 'ebcdictext'     => 'getEBCDICText'
            # 'unicodetext'    => 'getUnicodeText'
            # 'unicodetextlen' => 'getUnicodeTextLen'
            # 'underline'      => 'isUnderline'
            # 'overstrike'     => 'isOverstrike'
            #                  => getFont
            #                  => getMappedFontLocalId
            $objTmp->{'text'}   = $a2wObjTmp->getText();
            $objTmp->{'len'}    = $a2wObjTmp->getTextLen();
            $objTmp->{'angle'}  = $a2wObjTmp->getAngle();
            $objTmp->{'color'}  = $a2wObjTmp->_getAttribute('color');
            $objTmp->{'fontsize'} = $a2wObjTmp->getFontSize() / 10;

            $a2wFontTmp = $a2wObjTmp->getFont();
            if ( $a2wFontTmp != undef ){
                $objTmp->{'font'}   = {
                    'typeface'  => $a2wFontTmp->getTypefaceName(),
                    #'size'      => $a2wFontTmp->getHeight(), # not used  since the value may not be always correct
                    #'width'     => $a2wFontTmp->getWidth(),  # not used since it always returns -1
                    'encoding'  => $a2wFontTmp->getEncoding(),
                    'bold'      => ( $a2wFontTmp->isBold() == $TRUE ) ? "true" : "false",
                    'italic'    => ( $a2wFontTmp->isItalic() == $TRUE ) ? "true" : "false"
                };
            };

            #$objTmp->{'flid'}   = $a2wObjTmp->getMappedFontLocalId(); # not used since it's a a2w internal property
        }
        elsif ( $sObjTypeTmp eq "line" ){
            # Reference: see a2w/core/ext/Line.pm
            # 'negative'   => 'isNegative'
            $objTmp->{'width'}      = $a2wObjTmp->getWidth();
            $objTmp->{'len'}        = $a2wObjTmp->getLength();
            $objTmp->{'color'}      = $a2wObjTmp->_getAttribute('color');
            $objTmp->{'horizontal'} = $a2wObjTmp->isHorizontal();
            $objTmp->{'vertical'}   = $a2wObjTmp->isVertical();
        }
        elsif ( $sObjTypeTmp eq "vector" ){
            # Reference: see a2w/core/ext/Vector.pm
            # 'angle'             => 'getAngle'
            # 'uniquecolorscount' => 'getUniqueColorsCount'
            $objTmp->{'width'}  = $a2wObjTmp->getWidth();
            $objTmp->{'height'} = $a2wObjTmp->getHeight();
            $objTmp->{'color'}  = $a2wObjTmp->_getAttribute('color');
        }
        elsif ( $sObjTypeTmp eq "image" ){
            # Reference: see a2w/core/ext/Image.pm
            # 'angle'              => 'getAngle'
            # 'name'               => 'getName'
            # 'data'               => 'getData'
            # 'datalen'            => 'getDataLength'
            # 'compression'        => 'getCompression'
            # 'bitsperpixel'       => 'getBitsPerPixel'
            # 'samplesperpixel'    => 'getSamplesPerPixel'
            # 'width'              => 'getWidth'
            # 'height'             => 'getHeight'
            # 'presentationwidth'  => 'getPresentationWidth'
            # 'presentationheight' => 'getPresentationHeight'
            # 'uniquecolorscount'  => 'getUniqueColorsCount'
            $objTmp->{'name'}  = $a2wObjTmp->getName();
            $objTmp->{'width'}  = $a2wObjTmp->getWidth();
            $objTmp->{'height'} = $a2wObjTmp->getHeight();
            $objTmp->{'bpp'} = $a2wObjTmp->getBitsPerPixel();
        }
        elsif ( $sObjTypeTmp eq "container" ){
            # Reference: see a2w/core/ext/Container.pm
            $objTmp->{'name'}  = $a2wObjTmp->getName();
            $objTmp->{'width'}  = $a2wObjTmp->getWidth();
            $objTmp->{'height'} = $a2wObjTmp->getHeight();
        }
       
        push @arBlocksTmp, $objTmp;

    }
    return @arBlocksTmp;
}

#-----------------------------------------------------------------------
# Log objects
#-----------------------------------------------------------------------
sub _logObjects{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Array of objects hash (having both kernel object and pom object)
    #
    my $arefObjsPar = shift;
    my @arrObjsTmp  = @{ $arefObjsPar };

    #---- Iterate objects and log them
    my $sObjTypeTmp = "";
    my $a2wObjTmp = undef;
    my $pomObjTmp = undef;
    foreach my $elem ( @arrObjsTmp ){
        #---- Get object details
        $a2wObjTmp   = $elem->{ 'A2WOBJ' };
        $pomObjTmp   = $elem->{ 'POMOBJ' };
		$sObjTypeTmp = $a2wObjTmp->_getType();

        #---- Log object
        if ( $sObjTypeTmp eq "text" ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( sprintf( $sIndentTmp . "text  @(%d,%d)>%s< ISeqId=%d", $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getText(), $a2wObjTmp->getSequenceId() ) ); }
        }
        elsif ( $sObjTypeTmp eq "image" ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( sprintf( $sIndentTmp . "image @(%d,%d)>%s< W=%d, H=%d, ISeqId=%d", $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getName(), $a2wObjTmp->getWidth(), $a2wObjTmp->getHeight(), $a2wObjTmp->getSequenceId() ) ); }
        }
        elsif ( $sObjTypeTmp eq "container" ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( sprintf( $sIndentTmp . "cont. @(%d,%d)>%s< W=%d, H=%d, ISeqId=%d", $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getName(), $a2wObjTmp->getWidth(), $a2wObjTmp->getHeight(), $a2wObjTmp->getSequenceId() ) ); }
        }
        elsif ( $sObjTypeTmp eq "line" ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( sprintf( $sIndentTmp . "line  @(%d,%d)>< W=%d, L=%d, ISeqId=%d", $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getLength(), $a2wObjTmp->getSequenceId() ) ); }
        }
        elsif ( $sObjTypeTmp eq "vector" ){
            if ( $bLog == $TRUE ){ $theLogger->logMessage( sprintf( $sIndentTmp . "vector@(%d,%d)>< W=%d, H=%d, ISeqId=%d", $pomObjTmp->{ $a2w::core::dm::Constants::AT_XPOS }, $pomObjTmp->{ $a2w::core::dm::Constants::AT_YPOS }, $a2wObjTmp->getWidth(), $a2wObjTmp->getHeight(), $a2wObjTmp->getSequenceId() ) ); }
        }
    }
}

#-----------------------------------------------------------------------
# Log indexes
#-----------------------------------------------------------------------
sub _logIndexes{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Array of indexes
    #
    my $arefIdxsPar = shift;
    my @arrIdxsTmp  = @{ $arefIdxsPar };

    #---- Iterate objects and log them
    my $sNTmp = "";
    my $sVTmp = "";
    foreach my $idx ( @arrIdxsTmp ){
        #---- Get object details
        $sNTmp = $idx->{ 'NAME' };
        if ( $sNTmp eq "" ){ next; }
        $sVTmp = $idx->{ 'VALUE' };

        #---- Log index
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "index @" . $sNTmp .  "=>" . $sVTmp ."<" ); }
    }
}

#-----------------------------------------------------------------------
# Print to output dump
#-----------------------------------------------------------------------
sub _printMessage{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Message to be printed
    #
    my $sMsgPar = shift;

    if ( $bLog == $TRUE ){ $theLogger->logMessage( $sMsgPar ); }

    #---- Get output file handle
    my $handleFileTmp = $this->{ 'fHandle' };
    if ( $handleFileTmp == undef ){ return; } # do nothing, output is configured to be skipped (refer 'SkipOutput' flag of visitor)

    #---- Print message to output
    printf $handleFileTmp ( $sMsgPar . "\n" );
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
