#-------------------------------------------------------------------------------
#  cartagenote.pm:
#
#  Module to process generate rollkarte.json from original PDF
#
#  Call:
#  On Windows : afp2web.exe -q -c -doc_cold -sp:cartagenote.pm samples\original.pdf
#  On Unix    : ./afp2web   -q -c -doc_cold -sp:cartagenote.pm samples/original.pdf
#
#  Author    : Fa. Maas (AFP2web Team)
#  Copyright : (C) 2018-2019 by Maas Holding GmbH
#
#  V102    2019-12-18    OXS-10109: Extended to support multiple "Bordero" blocks
#  V101    2019-01-28    Fixed minor bug in determining block end on current page
#  V100    2018-08-31    Initial Release
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
    unshift( @INC, $sScriptFilePathTmp . "/a2w" );

    #---- Add perl path to module search path
    unshift( @INC, $sScriptFilePathTmp . "/perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/site/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/site/lib" );

}

use a2w::Config;
use a2w::ContentBlock;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;

use a2w::ConfigConstants;
use a2w::ContentBlockConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;

use a2w::core::dm::MiningEngine;     # Data mining module
use a2w::core::dm::Anchor;           # Anchor
use a2w::core::log::Logger;          # Log engine
use a2w::core::visitor::JSONVisitor;

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sDocumentTitle    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::TITLE );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sLogPath          = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGPATH );
    $sTempFilePath     = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::TEMPPATH );
    $iOutputResolution = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::RESOLUTION );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    #---- Set AutoSplit to true
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::AUTOSPLIT, "on" );

    #---- Add temp path to module search path
    unshift( @INC, $sTempFilePath );

    #---- Instantiate Logger
    $theLogger = undef;
    $bDebugLog = $FALSE;
    if ( $bLog == $TRUE ){
        #---- Set log on log engine
        $theLogger = a2w::core::log::Logger->getSingleton();
        $theLogger->setStartTime( $sStartTime );

        #---- Register modules that has to be logged
        $theLogger->registerClasses(   "a2w::Main"
                                     . ",a2w::core::dm::MiningEngine"
                                     . ",a2w::core::visitor::JSONVisitor"
                                   );

        #---- Open log file
        my $sLogFilenameTmp = $sSpoolFilename;
        $sLogFilenameTmp =~ s/^.*[\\\/]//;
        $sLogFilenameTmp .= ".cartagenote.log";
        $theLogger->setFilename( $sLogFilenameTmp );
        if ( $theLogger->open( $sLogPath ) == $FALSE ){
            return ( -1, "[ERROR] Unable to open log file (" . $sLogPath . $sLogFilenameTmp . ")" );
        }
        $bLog = $theLogger->isRegistered( "a2w::Main" );

        if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sfdbg") >= 0 ){
            $theLogger->setLevel( $a2w::core::log::Logger::LEVEL_DEBUG );
            $bDebugLog = $TRUE;
        }

        $theLogger->logFunctionName( "a2w::Main", "initialize()" );
        $theLogger->logMessage( "Running $sScriptProcTmp..." );
        $theLogger->logMessage( "initialize(): Processing " . $sSpoolFilename );
        $theLogger->logMessage( "initialize(): Args: $sScriptArgsTmp, OutputFilePath: $sOutputFilePath" );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Create mining engine
    $dmMiningEngine = new a2w::core::dm::MiningEngine();

    #---- Fill info on mining engine
    $dmMiningEngine->registerClassesForLog();
    $dmMiningEngine->setSpoolFilename( $sSpoolFilename );
    $dmMiningEngine->setOutputPath( $sOutputFilePath );
    $dmMiningEngine->setOutputRes( $iOutputResolution );

    #---- Load config
    @arrBlocks = ();
    $hrefGlobalBlocks = undef; # Script argument specified block definition
    $hrefSpoolBlocks = undef;  # Block definition specified in NOPs before first document (spool level)
    $hrefBlocks = undef;       # Block definition specified in NOPs within named page group (document level)
    if ( $sScriptArgsTmp ne "" && $sScriptArgsTmp =~ /^.*\:\:.*$/ ){
        my $sErrMsgTmp = "";
        ( $hrefGlobalBlocks, $sErrMsgTmp ) = _loadBlockDefinition( $sScriptArgsTmp );
        if ( lc( ref( $hrefGlobalBlocks ) ) ne "hash" && $hrefGlobalBlocks < 0 ){ return ( $hrefGlobalBlocks, $sErrMsgTmp ); }
        $hrefBlocks = $hrefGlobalBlocks;
    }
    else {
        if ( $bLog == $TRUE ){ if ( $sScriptArgsTmp ne "" ){ $theLogger->logMessage( "Invalid script argument: $sScriptArgsTmp" ); } }
    }

    #---- Detected parser rules count
    $sCurrentBlockType = "";
    $iParserRulesCount = 0;

    #---- Overwrite line gap constant
    $a2w::core::dm::Constants::CONT_LINE_TOLERANCE = 0.0125; # V101 Change

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "initializeDoc()" );
        $theLogger->logMessage( "Name=" . $a2wDocumentPar->getName() . " Id=" . $iDocumentId );
    }

    #---- Initialize document
    $dmMiningEngine->initializeDoc(   'a2w::core::visitor::JSONVisitor'
                                    , $a2wDocumentPar
                                    , $sDocumentTitle
                                    , $FALSE # Page output is false always for JSON visitor
                                  );

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "initializePage()" ); }

    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    my $iPgIdTmp = $a2wPagePar->getParseId(); # V101 Change

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "afp2web()" );
        $theLogger->logMessage( "PageId=" . $iPgIdTmp ); # V101 Change
    }

    #---- Set default return value
    my $iRetTmp = $SKIP; # append page to Current Document

    #---- Process page
    my $iPPRetTmp = $dmMiningEngine->processPage( $a2wPagePar, $TRUE );
    if ( $iPPRetTmp < 0 ){ return ( -1, "Unable to process page " . $a2wPagePar->getParseId() . " content" ); }

    #---- Write page
    $dmMiningEngine->writePage( $iPgIdTmp ); # V101 Change

    # V101 Begin
    # If last block of page is not multipage, then do not add start anchor on following page
    my $arefPageBlocksTmp = $dmMiningEngine->getPageWrittenBlocks( $iPgIdTmp );
    if ( $arefPageBlocksTmp != undef ){
        my @arrPageBlocksTmp = @{ $arefPageBlocksTmp };

        #---- Filter raw blocks alone
        @arrPageBlocksTmp = grep { $_->getContentDefType() eq "raw" } @arrPageBlocksTmp;
        if ( @arrPageBlocksTmp > 0 ){
            my $blkLastTmp = $arrPageBlocksTmp[ $#arrPageBlocksTmp ];
            if ( $blkLastTmp != undef ){
                #---- Get block output context
                #---- (refer a2w\core\visitor\JSONVisitor->writeRawObjects function for structure of output context object)
                my $ctxtOutTmp = $blkLastTmp->getOutputContext();
                if ( $ctxtOutTmp->{ 'multipage' } eq 'false' ){
                    $sCurrentBlockType = ""; # Reset current block type
                }
            }
        }
    }
    # V101 End

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizePage()" ); }

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizeDoc()" ); }

    #---- Dump the page contents
    if ( $bDebugLog == $TRUE ){
        $dmMiningEngine->dumpDOM( $sOutputFilePath, $TRUE );
    }

    #---- Finalize document
    my $iRetTmp = $dmMiningEngine->finalizeDoc();
    return $iRetTmp;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalize()" ); }

    #---- Set end time and close log file
    if ( $bLog == $TRUE ){        
        $theLogger->setEndTime( time() );    # set end time
        $theLogger->close();                # close log file
    }

    #---- Cleanup
    $theLogger      = undef;
    $dmMiningEngine = undef;

    return 0;
}

#-----------------------------------------------------------------------
# Load block definition
#
# Loads given block definition module and returns the hash structure
# Returns <0 and message in case of error
#
#-----------------------------------------------------------------------
sub _loadBlockDefinition{

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_loadBlockDefinition()" ); }

    #---- Get parameter
    #
    # 1. Module name
    # 2. Module filename (optional)
    #
    my $sBlkDefModPar = shift;
    my $sBlkDefFilePar = "";
    if ( @_ > 0 ){ $sBlkDefFilePar = shift; }

    #---- Assert parameter
    if ( $sBlkDefFilePar eq "" && $sBlkDefModPar eq "" ){ return ( -1, "Empty block definition module name" ); } # Return if name is empty
    if ( $sBlkDefFilePar eq "" && $sBlkDefModPar !~ /^.*\:\:.*$/ ){ return ( -1, "Invalid block definition module name ($sBlkDefModPar)" ); } # Return if name is not like module name

    #---- Parse and load block definition module
    if ( $sBlkDefFilePar eq "" ){ # $V102 Change
        eval "require $sBlkDefModPar";
        if ( $@ ){ return ( -2, "Unable to parse block definition module $sBlkDefModPar. reason=" . $@ ); }
    }
    else {
        require $sBlkDefFilePar;
    }
    my $hrefBlocksTmp = \%{ $sBlkDefModPar . '::Blocks' }; # Fetch defined blocks list
    my @arrRulesTmp = @{ $sBlkDefModPar . '::ParserRules' }; # Fetch defined parser rules
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "$sBlkDefModPar Defined Parser Rules:>" . @arrRulesTmp . "<" ); }

    #---- Load all blocks
    $dmMiningEngine->loadConfig( \%{ $sBlkDefModPar . '::Blocks' } );

    #---- Fetch all config module defined blocks
    @arrBlocks = sort keys( %{ $hrefBlocksTmp } );
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "$sBlkDefModPar Defined Blocks:>@arrBlocks<" ); }

    #---- Iterate through blocks list and add them as document block
    my $blkCurrentTmp = undef;
    foreach my $sBlkNameTmp ( @arrBlocks ){
        #---- Get block
        $blkCurrentDefTmp = $hrefBlocksTmp->{ $sBlkNameTmp };
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Name:>" . $sBlkNameTmp . "< Id:>" . $blkCurrentDefTmp->{ 'Id' } . "< Info:>" . $blkCurrentDefTmp->{ 'Info' } . "<" ); }

        #---- Add block to document
        $dmMiningEngine->addDocumentBlock( $blkCurrentDefTmp->{ 'Id' }, $blkCurrentDefTmp->{ 'Info' } );
    }

    #---- Iterate through array of rules and add them to mining engine
    foreach my $pr ( @arrRulesTmp ){
        $dmMiningEngine->addParserRule( $pr, \&_preProcessParserObject );
    }

    return $hrefBlocksTmp;
}

#-----------------------------------------------------------------------
# Preprocess object callback implementation
#
# prototype:
# <Array of objects> <function name>( a2w::core::dm::Anchor, a2w::Page, a2w::<Container|Image|Line|Text|Vector>Object )
#
# Callback must preprocess the object and return array of objects to add on the page
#
# NOTE: Object passed in must also be included on returned array of objects if it
#       has to be added on page otherwise the object will be skipped
#
#-----------------------------------------------------------------------
sub _preProcessParserObject{
    #---- Get parameter
    #
    # 1. Anchor
    # 2. Page reference
    # 3. Object
    #
    my $ancMatchPar   = shift;
    my $a2wPageRefPar = shift;
    my $a2wObjectPar  = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "_preProcessParserObject()" ); }

    #---- Assert parameters
    if ( $ancMatchPar == undef || $a2wPageRefPar == undef || $a2wObjectPar == undef ){ return [ $a2wObjectPar ]; }
    $iParserRulesCount++;

    #---- Evaluate line gap
    my $iLineGapTmp = int( $a2w::core::dm::Constants::CONT_LINE_TOLERANCE * $a2wPageRefPar->getResolution() ) + 1; # V101 Change

    #---- Process specific to matched rule
    my @arrObjsTmp = ( $a2wObjectPar );
    my $sRuleIdTmp = lc( $ancMatchPar->getId() );
    my $iOffAdj4ShipmentAnchorsTmp = 495;  #  ~51 pixels@300 dpi # V101 Change
    my $iOffAdj4BorderoAnchorsTmp  = 432;  #  ~45 pixels@300 dpi
    my $iOffAdj4FooterAnchorsTmp   = 1728; # ~180 pixels@300 dpi
    my $iStartAnchorXPosTmp = 450;
    my $iEndAnchorXPosTmp = 23050;
    if ( $bLog == $TRUE ){ $theLogger->logMessage( "Rule type:>$sRuleIdTmp<" ); }
    if ( $sRuleIdTmp eq "shipment_start" ){
        my $iObjYPosTmp = $a2wObjectPar->getYPos() - $iOffAdj4ShipmentAnchorsTmp;
        my $iLineWidthTmp  = 2; # Default is header block
        my $iLineLengthTmp = 2; # Default is header block
        if ( $sCurrentBlockType eq "shipment" ){
            $iLineWidthTmp  = 4;
            $iLineLengthTmp = 4;
        }

        $sCurrentBlockType = "shipment";

        #---- Create start/end anchors for shipment block and add them to page
        # end anchor: Line, Position: X position = 23050, Y Position = <Object's Y> - <$iOffAdj4ShipmentAnchorsTmp>, Length = 4, Width = 4
        # start anchor: Line, Position: X position = 450, Y Position = <Object's Y> - <$iOffAdj4ShipmentAnchorsTmp> + 10, Length = 4, Width = 4

        #---- First create and add end anchor for previous block
        my $iIdxTmp = 1;
        my $objEndAncTmp = new a2w::Line();
        $objEndAncTmp->setXPos( $iEndAnchorXPosTmp );
        $objEndAncTmp->setYPos( $iObjYPosTmp );
        $objEndAncTmp->setWidth( $iLineWidthTmp );
        $objEndAncTmp->setLength( $iLineLengthTmp );
        $objEndAncTmp->remove();                  # Don't present it on output
        $a2wPageRefPar->addLine( $objEndAncTmp ); # Add it to page, so it gets deleted later in core
        @arrObjsTmp[ $iIdxTmp++ ] = $objEndAncTmp;

        #---- Create and add start anchor
        $iLineWidthTmp  = 4;
        $iLineLengthTmp = 4;
        if ( $iObjYPosTmp < 820 ){ $iObjYPosTmp = 820; } # Ensure minimum start Y position
        else { $iObjYPosTmp += $iLineGapTmp; } # V101 Change
        my $objStartAncTmp = new a2w::Line();
        $objStartAncTmp->setXPos( $iStartAnchorXPosTmp );
        $objStartAncTmp->setYPos( $iObjYPosTmp );
        $objStartAncTmp->setWidth( $iLineWidthTmp );
        $objStartAncTmp->setLength( $iLineLengthTmp );
        $objStartAncTmp->remove();                  # Don't present it on output
        $a2wPageRefPar->addLine( $objStartAncTmp ); # Add it to page, so it gets deleted later in core
        @arrObjsTmp[ $iIdxTmp ] = $objStartAncTmp;
    }
    elsif ( $sRuleIdTmp eq "bordero_start" ){
        my $iObjYPosTmp = $a2wObjectPar->getYPos() - $iOffAdj4BorderoAnchorsTmp; # V101 change
        my $iLineWidthTmp  = 2; # Default is header block
        my $iLineLengthTmp = 2; # Default is header block
        if ( $sCurrentBlockType eq "shipment" ){
            $iLineWidthTmp  = 4;
            $iLineLengthTmp = 4;
        }
        # V102 Begin
		# Multiple bordero blocks might occur too
        elsif ( $sCurrentBlockType eq "bordero" ){
            $iLineWidthTmp  = 6;
            $iLineLengthTmp = 6;
        }
        # V102 End

        $sCurrentBlockType = "bordero";

        #---- Create start/end anchors for bordero block and add them to page
        # end anchor: Line, Position: X position = 23050, Y Position = <Object's Y> - <$iOffAdj4BorderoAnchorsTmp>, Length = 6, Width = 6
        # start anchor: Line, Position: X position = 450, Y Position = <Object's Y> - <$iOffAdj4BorderoAnchorsTmp> + 10, Length = 6, Width = 6

        #---- First create and add end anchor for previous block
        my $iIdxTmp = 1;
        my $objEndAncTmp = new a2w::Line();
        $objEndAncTmp->setXPos( $iEndAnchorXPosTmp );
        $objEndAncTmp->setYPos( $iObjYPosTmp );
        $objEndAncTmp->setWidth( $iLineWidthTmp );
        $objEndAncTmp->setLength( $iLineLengthTmp );
        $objEndAncTmp->remove();                  # Don't present it on output
        $a2wPageRefPar->addLine( $objEndAncTmp ); # Add it to page, so it gets deleted later in core
        @arrObjsTmp[ $iIdxTmp++ ] = $objEndAncTmp;

        #---- Create and add start anchor
        $iLineWidthTmp  = 6;
        $iLineLengthTmp = 6;
        my $objStartAncTmp = new a2w::Line();
        $objStartAncTmp->setXPos( $iStartAnchorXPosTmp );
        $objStartAncTmp->setYPos( $iObjYPosTmp + $iLineGapTmp ); # V101 Change
        $objStartAncTmp->setWidth( $iLineWidthTmp );
        $objStartAncTmp->setLength( $iLineLengthTmp );
        $objStartAncTmp->remove();                  # Don't present it on output
        $a2wPageRefPar->addLine( $objStartAncTmp ); # Add it to page, so it gets deleted later in core
        @arrObjsTmp[ $iIdxTmp ] = $objStartAncTmp;
    }
    elsif ( $sRuleIdTmp eq "footer_start" ){
        my $iObjYPosTmp = $a2wObjectPar->getYPos() - $iOffAdj4FooterAnchorsTmp; # V101 change
        my $iLineWidthTmp  = 2; # Default is header block
        my $iLineLengthTmp = 2; # Default is header block
        if ( $sCurrentBlockType eq "shipment" ){
            $iLineWidthTmp  = 4;
            $iLineLengthTmp = 4;
        }
        elsif ( $sCurrentBlockType eq "bordero" ){
            $iLineWidthTmp  = 6;
            $iLineLengthTmp = 6;
        }

        $sCurrentBlockType = "footer";

        #---- Create start/end anchors for bordero block and add them to page
        # end anchor: Line, Position: X position = 23050, Y Position = <Object's Y> - <$iOffAdj4<Shipment|Bordero>AnchorsTmp>, Length = 8, Width = 8
        # start anchor: Line, Position: X position = 450, Y Position = <Object's Y> - <$iOffAdj4<Shipment|Bordero>AnchorsTmp> + 10, Length = 8, Width = 8

        #---- First create and add end anchor for previous block
        my $iIdxTmp = 1;
        my $objEndAncTmp = new a2w::Line();
        $objEndAncTmp->setXPos( $iEndAnchorXPosTmp );
        $objEndAncTmp->setYPos( $iObjYPosTmp );
        $objEndAncTmp->setWidth( $iLineWidthTmp );
        $objEndAncTmp->setLength( $iLineLengthTmp );
        $objEndAncTmp->remove();                  # Don't present it on output
        $a2wPageRefPar->addLine( $objEndAncTmp ); # Add it to page, so it gets deleted later in core
        @arrObjsTmp[ $iIdxTmp++ ] = $objEndAncTmp;

        #---- Create and add start anchor
        $iLineWidthTmp  = 8;
        $iLineLengthTmp = 8;
        my $objStartAncTmp = new a2w::Line();
        $objStartAncTmp->setXPos( $iStartAnchorXPosTmp );
        $objStartAncTmp->setYPos( $iObjYPosTmp + $iLineGapTmp ); # V101 Change
        $objStartAncTmp->setWidth( $iLineWidthTmp );
        $objStartAncTmp->setLength( $iLineLengthTmp );
        $objStartAncTmp->remove();                  # Don't present it on output
        $a2wPageRefPar->addLine( $objStartAncTmp ); # Add it to page, so it gets deleted later in core
        @arrObjsTmp[ $iIdxTmp ] = $objStartAncTmp;
    }
    elsif ( $sRuleIdTmp eq "page_start" ){
        # V101 Begin
        # start anchor: Line, Position: X position = 450, Y Position = <Object's Y> + 10, Length = 2, Width = 2
        my $iObjXPosTmp = $iStartAnchorXPosTmp;
        my $iObjYPosTmp = 0;
        my $iLineWidthTmp  = 2; # Default is header block
        my $iLineLengthTmp = 2; # Default is header block

        # Add line based on page
        my $bAddAnchorTmp = $TRUE;
        if ( $a2wPageRefPar->getParseId() == 1 ){ # First page
            #---- Create start anchor for header or shipment or bordero block. Add it to page
            if ( $sCurrentBlockType eq "shipment" ){
                $iLineWidthTmp  = 4;
                $iLineLengthTmp = 4;
            }
            elsif ( $sCurrentBlockType eq "bordero" ){
                $iLineWidthTmp  = 6;
                $iLineLengthTmp = 6;
            }
            elsif ( $sCurrentBlockType eq "footer" ){
                $iLineWidthTmp  = 8;
                $iLineLengthTmp = 8;
            }
            else { # header block
                $iObjXPosTmp = 0; # Just reset to top of page
                $iObjYPosTmp = 0; # Just reset to top of page
            }
        }
        else { # Following page
            #---- Create start anchor for shipment or bordero or footet block. Add it to page
            if ( $sCurrentBlockType eq "shipment" ){
                $iLineWidthTmp  = 4;
                $iLineLengthTmp = 4;
            }
            elsif ( $sCurrentBlockType eq "bordero" ){
                $iLineWidthTmp  = 6;
                $iLineLengthTmp = 6;
            }
            elsif ( $sCurrentBlockType eq "footer" ){
                $iLineWidthTmp  = 8;
                $iLineLengthTmp = 8;
            }
            else {
                #---- shipment or bordero block ended on previous page, So do not add any anchor
                $bAddAnchorTmp = $FALSE;
            }
        }

        #---- Create and add start anchor
        if ( $bAddAnchorTmp == $TRUE ){
            my $objStartAncTmp = new a2w::Line();
            $objStartAncTmp->setXPos( $iObjXPosTmp );
            $objStartAncTmp->setYPos( $iObjYPosTmp );
            $objStartAncTmp->setWidth( $iLineWidthTmp );
            $objStartAncTmp->setLength( $iLineLengthTmp );
            $objStartAncTmp->remove();                  # Don't present it on output
            $a2wPageRefPar->addLine( $objStartAncTmp ); # Add it to page, so it gets deleted later in core
            @arrObjsTmp[ 1 ] = $objStartAncTmp;
		}
        # V101 End
    }
    elsif ( $sRuleIdTmp eq "page_end" ){
        #---- Create end anchor for shipment or bordero or footer block. Add it to page
        # end anchor: Line, Position: X position = 23050, Y Position = <Page height>, Length = 4, Width = 4
        my $iObjYPosTmp = $a2wPageRefPar->getHeight();
        my $iLineWidthTmp  = 4; # Default is shipment block
        my $iLineLengthTmp = 4; # Default is shipment block
        if ( $sCurrentBlockType eq "bordero" ){
            $iLineWidthTmp  = 6;
            $iLineLengthTmp = 6;
        }
        elsif ( $sCurrentBlockType eq "footer" ){
            $iLineWidthTmp  = 8;
            $iLineLengthTmp = 8;
        }

        #---- Create and add end anchor
        my $objEndAncTmp = new a2w::Line();
        $objEndAncTmp->setXPos( $iEndAnchorXPosTmp );
        $objEndAncTmp->setYPos( $iObjYPosTmp );
        $objEndAncTmp->setWidth( $iLineWidthTmp );
        $objEndAncTmp->setLength( $iLineLengthTmp );
        $objEndAncTmp->remove();                  # Don't present it on output
        $a2wPageRefPar->addLine( $objEndAncTmp ); # Add it to page, so it gets deleted later in core
        @arrObjsTmp[ 0 ] = $objEndAncTmp;         # Do not include page end object, since it is already added to page
    }

    return \@arrObjsTmp;
}

__END__
