#-------------------------------------------------------------------------------
#  pdfAttachments.pm:
#
#  Scripting Facility to process 'Attachments' INI option and create a report
#  page with following content
#  1. Page header
#     "Attachments" text
#  2. Page content
#     Iterate through the Attachments and add them as row as given below
#     <description>
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -doc_cold -sp:pdfAttachments.pm -atts:[{filename:<attachment file name>,desc:<attachment description>}] samples\afpsample.afp
#
#  On Unix:    ./afp2web   -q -c -doc_cold -sp:pdfAttachments.pm -atts:[{filename:<attachment file name>,desc:<attachment description>}] samples/afpsample.afp
#
#  Author  : Fa. Maas
#
#  $V100   2019-07-24    AFP-877: Initial Release
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );

    #---- Add local perl lib path relative to script file parent path to module search path
    unshift( @INC, $sScriptFilePathTmp . "/perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/perl/site/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../../perl/site/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/a2w" );
}

use a2w::Config;
use a2w::ContentBlock;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;

use a2w::ConfigConstants;
use a2w::ContentBlockConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;

use a2w::core::log::Logger; # Log engine

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    $sLogFilePath      = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGPATH );
    $sPDFAttachments   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::ATTACHMENTS );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    #---- Instantiate Logger
    $theLogger = undef;
    $bDebugLog = $FALSE;
    if ( $bLog == $TRUE ){
        #---- Set log on log engine
        $theLogger = a2w::core::log::Logger->getSingleton();
        $theLogger->setStartTime( $sStartTime );

        #---- Register modules that has to be logged
        $theLogger->registerClasses( "a2w::Main" );

        #---- Open log file
        my $sLogFilenameTmp = $sSpoolFilename;
        $sLogFilenameTmp =~ s/^.*[\\\/]//;
        $sLogFilenameTmp .= ".pdfAttachments.log";
        $theLogger->setFilename( $sLogFilenameTmp );
        if ( $theLogger->open( $sLogFilePath ) == $FALSE ){
            return ( -1, "[ERROR] Unable to open log file (" . $sLogFilePath . $sLogFilenameTmp . ")" );
        }
        $bLog = $theLogger->isRegistered( "a2w::Main" );

        if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sfdbg") >= 0 ){
            $theLogger->setLevel( $a2w::core::log::Logger::LEVEL_DEBUG );
            $bDebugLog = $TRUE;
        }

        $theLogger->logFunctionName( "a2w::Main", "initialize()" );
        $theLogger->logMessage( "Running $sScriptProcTmp..." );
        $theLogger->logMessage( "initialize(): Processing " . $sSpoolFilename );
    }

    #---- Preprocess attachments value
    # Double quote filename field name
    if (    $sPDFAttachments !~ /"filename/
         && $sPDFAttachments !~ /filename"/
       ){
        $sPDFAttachments =~ s/filename/"filename"/g;
    }

    # Double quote desc field name
    if (    $sPDFAttachments !~ /"desc/
         && $sPDFAttachments !~ /desc"/
       ){
        $sPDFAttachments =~ s/\bdesc\b/"desc"/g;
    }

    # Escape '\' character
    $sPDFAttachments =~ s/\\/\//g;

    #---- Process PDF attachments json
    $arefAttachments = undef;
    eval{
        require JSON::Tiny;
        $arefAttachments = JSON::Tiny::from_json( $sPDFAttachments );
    };
    if ( $@ ){
        return ( -2, "Unable to parse PDF Attachments option value ($sPDFAttachments), reason=" . $@ );
    }
    elsif ( $arefAttachments == undef ){
        return ( -2, "Unable to parse PDF Attachments option value ($sPDFAttachments), reason=" . $@ );
    }
    if ( $bLog == $TRUE ){ $theLogger->logHashMessage( { 'PDFAttachments' => $arefAttachments } ); }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

	#---- Font Height in points
	$iFontHeight = 8;
	
    #---- Create text font
	$a2wFont = new a2w::Font( $a2w::FontConstants::TYPE_TYPE1, "Courier" );
    $a2wFont->setHeight( $iFontHeight );

    #---- Create header font
	$a2wHeaderFont = new a2w::Font( $a2w::FontConstants::TYPE_TYPE1, "Helvetica" );
    $a2wHeaderFont->setHeight( $iFontHeight * 2 );

    # Page header text
    $PAGE_HEADER_TEXT = 'Attachments';

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "initializeDoc()" );
        $theLogger->logMessage( "Name=" . $a2wDocumentPar->getName() . " Id=" . $iDocumentId );
    }

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "initializePage()" ); }

	#---- Page formatting constants
    $AFP_PTX_RESOLUTION = $a2wPagePar->getResolution();
	$PAGE_LEFT_MARGIN   = 15; # mm
	$PAGE_TOP_MARGIN    = 15; # mm
	$KEY_X_POS 			= int( ( $PAGE_LEFT_MARGIN / 25.4 ) * $AFP_PTX_RESOLUTION );
	$TEXT_Y_START       = int( ( $PAGE_TOP_MARGIN / 25.4 ) * $AFP_PTX_RESOLUTION );
	$LINE_INCREMENT     = int( ( $iFontHeight / 72 ) * $AFP_PTX_RESOLUTION );
	$VALUE_X_POS 		= $KEY_X_POS * 3;

	# Maximum characters allowed per line for value
	$VALUE_MAX_CHARS_PER_LINE = 75;

    return 0;
}

#-----------------------------------------------------------------------
# afp2web
#
# Page processing method
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "afp2web()" );
        $theLogger->logMessage( "PageId=" . $a2wPagePar->getParseId() );
    }

	#---- Add page header text
	my ( $iRetTmp, $sMsgTmp ) = _addPageHeader( $PAGE_HEADER_TEXT );
	if ( $iRetTmp < 0 ){ return ( $iRetTmp, $sMsgTmp ); }

	#---- Add page content texts
	( $iRetTmp, $sMsgTmp ) = _addPageContent( $arefAttachments );
	if ( $iRetTmp < 0 ){ return ( $iRetTmp, $sMsgTmp ); }

    return $APPEND; # default: Append page
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizePage()" ); }

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizeDoc()" ); }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalize()" ); }

    return 0;
}

#-----------------------------------------------------------------------
# Add page header text
#-----------------------------------------------------------------------
sub _addPageHeader{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_addPageHeader" ); }

    #---- Fetch parameter
    #
    # 1. Page header text
    my $sHeaderTextPar = shift;
    if ( $sHeaderTextPar eq "" ){ return ( -1, "Empty page header text to add on page" ); }

	my $a2wTextTmp = new a2w::Text();
	if ( $a2wTextTmp == undef ){ return ( -1, "Unable to allocate text object for page header text ($sHeaderTextPar)" ); }

	#---- Fill in text details
	$a2wTextTmp->setXPos( $KEY_X_POS );      # Text X position
	$a2wTextTmp->setYPos( $TEXT_Y_START );   # Text Y position
	$a2wTextTmp->setText( $sHeaderTextPar ); # Text value
	$a2wTextTmp->setFont( $a2wHeaderFont );  # set font

	#---- Add text to page
	$a2wPagePar->addText( $a2wTextTmp );
	if ( $bLog == $TRUE ){ $theLogger->logMessage( "txt=>$sHeaderTextPar<@($KEY_X_POS, $TEXT_Y_START)" ); }
	$TEXT_Y_START += ( $LINE_INCREMENT * 2 );

    return 0;
}

#-----------------------------------------------------------------------
# Add page content
#-----------------------------------------------------------------------
sub _addPageContent{
    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_addPageContent" ); }

    #---- Fetch parameter
    #
    # 1. PDF attachments array reference
    my $arefAttachmentsPar = shift;
	if ( $arefAttachmentsPar == undef ){ return ( -1, "No attachments to list as page content" ); }

	my @arrAttachmentsTmp = @{ $arefAttachmentsPar };
	my $sSFNTmp = undef;
	my @values = ();
	my $iValLenTmp = 0;
	my $a2wTextTmp = undef;
	my $iYPosTmp = $TEXT_Y_START;

    foreach my $att ( @arrAttachmentsTmp ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( 'Attachment: filename:>' . $att->{ 'filename' } . '< desc:>' . $att->{ 'desc' } . '<' ); }

		#---- Add simple filename as text
        $sSFNTmp = $att->{ 'filename' };
		if ( $sSFNTmp =~ /\/.*\/(.*)$/ ){ $sSFNTmp = $1; }
		$a2wTextTmp = new a2w::Text();
		if ( $a2wTextTmp == undef ){ return ( -1, "Unable to allocate text object for text (" . $sSFNTmp . ")" ); }

		#---- Fill in text details
		$a2wTextTmp->setXPos( $KEY_X_POS );      # text X position
		$a2wTextTmp->setYPos( $iYPosTmp );       # text Y position
		$a2wTextTmp->setText( $sSFNTmp . ':' );  # text value
		$a2wTextTmp->setFont( $a2wFont );        # set font
		$a2wPagePar->addText( $a2wTextTmp );     # add text to page
		if ( $bLog == $TRUE ){ $theLogger->logMessage( "txt=>" . $sSFNTmp . ":<@($KEY_X_POS, $iYPosTmp)" ); }

		#---- Add description as text
		@values = unpack( "(a" . $VALUE_MAX_CHARS_PER_LINE . ")*", $att->{ 'desc' } ); # Split value into tokens of $VALUE_MAX_CHARS_PER_LINE  length
		$iValLenTmp = @values;

        #---- Add '-' for lines which got word split in 2 lines
        for ( my $v = 0; $v < ( $iValLenTmp - 1 ); $v++ ){
		    if ( $bLog == $TRUE ){ $theLogger->logMessage( "$v=>" . $values[ $v ] . "<" ); }
		    if ( $values[ $v ] !~ /\s$/ && $values[ $v + 1 ] !~ /^\s/ ){ $values[ $v ] .= '-'; }
		}

		# Some values might be longer, so split them on lines
		foreach $value ( @values ){
			if ( length( $value ) <= 0 ){ next; }

			$a2wTextTmp = new a2w::Text();
			if ( $a2wTextTmp == undef ){ return ( -1, "Unable to allocate text object for text ($value)" ); }

			#---- Fill in text details
			$a2wTextTmp->setXPos( $VALUE_X_POS ); # Text X position
			$a2wTextTmp->setYPos( $iYPosTmp );    # Text Y position
			$a2wTextTmp->setText( $value );       # Text value
			$a2wTextTmp->setFont( $a2wFont );     # set font
		
			#---- Add text to page
			$a2wPagePar->addText( $a2wTextTmp );
			if ( $bLog == $TRUE ){ $theLogger->logMessage( "txt=>$value<@($VALUE_X_POS, $iYPosTmp)" ); }
			$iYPosTmp += $LINE_INCREMENT;
		}
	}

    return 0;
}

__END__
