#-------------------------------------------------------------------------------
#  eyecatcher_mmd.pm:
#  Scripting Facility Module to demonstrate MMD document content and to collect
#  indexes
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -mmdin -doc_cold -sp:sfsamples\eyecatcher_mmd.pm -sft:ASA_EBCDIC_FIXED_133 -fd:f1a2w -pd:p1a2w samples\mmdsample.mmd
#
#  On Unix   :    ./afp2web   -q -c -mmdin -doc_cold -sp:sfsamples/eyecatcher_mmd.pm -sft:ASA_EBCDIC_FIXED_133 -fd:f1a2w -pd:p1a2w samples/mmdsample.mmd
#
#  Author  : Fa. OXSEED
#  Date    : 2010-07-27
#  Version : 1.0.0
#
#  $V100   2010-07-27    Initial Release
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );

    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;
use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sSpoolFileType    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SPOOLFILETYPE );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    if ( $bLog == $TRUE ){
        printf STDERR ( "Running $sScriptProcTmp...\n" );
        printf STDERR ( "initialize(): Processing $sSpoolFilename\n" );
        printf STDERR ( "initialize(): Args: $sScriptArgsTmp, IndexFilePath: $sIndexFilePath, OutputFilePath: $sOutputFilePath\n" );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Initialize Page Id
    $iPageId = 0;

    #---- Create Current Index List
    @arrIndexList = ();

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializeDoc(): DocId " . $a2wDocumentPar->getId() . "\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "afp2web(): PageId " . $a2wPagePar->getParseId() . "\n" );
    }    

    #---- Set default return value
    my $iRetTmp = $APPEND; # default: append page

    #---- Identify new document ----#
    # Rule:
    # "001 von" in following text
    #
    # @(181,385)>                  AAAAAAAAAAAAA                                            001 von 001                   <
    #
    #---- Fetch first text
    my $a2wTextTmp = $a2wPagePar->getFirstText();

    #---- Loop through all texts
    while ( $a2wTextTmp != 0 ){
        #---- Assert whether document eyecatcher exists
        if ( $a2wTextTmp->getXPos() >= 176
             && $a2wTextTmp->getXPos() <= 186
             && $a2wTextTmp->getYPos() >= 380
             && $a2wTextTmp->getYPos() <= 390 	
             && ((substr ($a2wTextTmp->getText(), 75, 3) eq "001" )) # Still identify substring "001" from the "001 von 001" string
            ){
                #---- Reset Page Id
                $iPageId = 0;

                $iRetTmp = $NEWDOC;
                if ( $bLog == $TRUE ){
                    printf STDERR "Found New Document\n";
                }
                last; # leave while loop
        }

        #---- Fetch next Text Object
        $a2wTextTmp = $a2wPagePar->getNextText();
    }

    #---- Increment Page Id
    $iPageId++;

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizeDoc()\n" );
    }

    #---- Process and fetch document indexes ----#
    my $a2wFirstPageTmp = $a2wDocumentPar->getFirstPage();

    #---- Process index fields
    my $hshIndexesTmp = _processAndFetchIndexFields( $a2wFirstPageTmp );

    #---- Assert processed index value
    if ( $hshIndexesTmp == undef ){
        return ( -999, "Unable to fetch indexes for document " . $a2wDocumentPar->getId() );
    }

    #---- Write index to file ----#
    #---- Build Index Filename
    my $sSimpleFilenameTmp = $a2wDocumentPar->getSimpleFilename();    # get document simple filename
    if ( $sSimpleFilenameTmp eq "" ){
        return ( -997, "Invalid Simple Filename(" . $sSimpleFilenameTmp . ") for document " . $a2wDocumentPar->getId() );
    }

    my $IndexFilenameTmp = $sIndexFilePath . $sSimpleFilenameTmp . ".idx"; # build Index Filename
    if ( $bLog == $TRUE ){
        printf STDERR ( "Writing Index File ==> $IndexFilenameTmp\n" );
    }

    #---- Open Index file
    my $bFileOpenSuccessTmp = open( fIndexFile, ">$IndexFilenameTmp" );
    if ( !$bFileOpenSuccessTmp ){
        return ( -998, "Unable to open index file $IndexFilenameTmp, rc=" . $bFileOpenSuccessTmp . "msg=" . $! );
    }

    #---- Fetch index name list
    my @arrIndexNameListTmp = sort keys( %{ $hshIndexesTmp } );
    for ( my $i = 0; $i < @arrIndexNameListTmp; $i++ ){
        print fIndexFile ( $arrIndexNameListTmp[ $i ] . "=" . $hshIndexesTmp->{ $arrIndexNameListTmp[ $i ] } . "\n" );
    }
    close( fIndexFile );

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalize()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Process index fields
#
# Parameter(s)
# 1. Page (of type a2w::Page)
#
# Returns
# - undef in case of error
#
# - Hash reference having indexes in case of success, hash elements would
#   be as given below
#   %hshCurrentDocIndexes = {
#         'I_BRANCH'      => ""         # Branch
#       , 'I_BRANCH_CODE' => ""         # Branch code
#       , 'I_AC_NAME'     => ""         # Account holder name
#       , 'I_ST_DATE'     => ""         # Statement date
#   };
#
# Remarks
# Processes given index fields line, splits indexes and returns the same
#
#-----------------------------------------------------------------------
sub _processAndFetchIndexFields(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "_processAndFetchIndexFields()\n" );
    }

    #---- Get parameter(s)
    #
    # Page (of type a2w::Page)
    #
    my $a2wFirstPagePar = shift;

    #---- Return value
    my $hshIndexFieldsTmp = undef;

    #---- Assert parameter(s)
    if ( $a2wFirstPagePar == 0 ){
        return $hshIndexFieldsTmp;
    }

    #---- Define the list of Eyecatcher positions we should look for
    #
    #  @(181,265)>                  Stuttgart                                                                              <
    #  @(181,295)>                  70173-70794                                                                            <
    #  @(181,325)>                  Geoffrey R Stephens                                      13.03.2009                    <
    #
    my @arrEyeCatcherIndexListTmp = ( ( 181,265 ),       # Branch (Filiale)
                                      ( 181,295 ),       # Bank code number (Bankleitzahl)
                                      ( 181,325 )        # Account holder name & Date of statement
                                    );

    #---- Define temp variables
    my $arrECILLenTmp   = @arrEyeCatcherIndexListTmp / 2;
    my $sTextTmp        = "";
    my $iTextXPosTmp    = 0;
    my $iTextYPosTmp    = 0;
    my $iArrPosTmp      = 0;
    my $iIdxTextXTmp    = 0;
    my $iIdxTextYTmp    = 0;
    my $sIndexValueTmp  = "";

    #---- Get the Index list count
    my $iIndexPtrTmp = @arrIndexList;

    #---- Fetch first text
    my $a2wTextTmp = $a2wFirstPagePar->getFirstText();

    #---- Loop thru all the Text Objects
    while ( $a2wTextTmp != 0 ){
        #---- Fetch text info
        $sTextTmp     = $a2wTextTmp->getText();
        $iTextXPosTmp = $a2wTextTmp->getXPos();
        $iTextYPosTmp = $a2wTextTmp->getYPos();

        #---- Search all the defined Eyecatchers
        for (my $i = 0; $i < $arrECILLenTmp; $i++ ){
            $iArrPosTmp = $i * 2;
            $iIdxTextXTmp = $arrEyeCatcherIndexListTmp[ $iArrPosTmp ];
            $iIdxTextYTmp = $arrEyeCatcherIndexListTmp[ $iArrPosTmp + 1 ];

            #---- If true ==> we found one of the defined Eyecatchers
            if (    $iTextXPosTmp >= ( $iIdxTextXTmp - 10 )
                 && $iTextXPosTmp <= ( $iIdxTextXTmp + 10 )
                 && $iTextYPosTmp >= ( $iIdxTextYTmp - 10 )
                 && $iTextYPosTmp <= ( $iIdxTextYTmp + 10 )
               ){
                #---- Fetch indexes
                if ( $i == 0 ){
                    $sTextTmp =~ s/^\s+//g;    # strip leading spaces
                    $sTextTmp =~ s/\s+$//g;    # strip trailing spaces

                    if ( $hshIndexFieldsTmp == undef ){
                        $hshIndexFieldsTmp = {};
                    }
                    $hshIndexFieldsTmp->{ 'I_BRANCH' } = $sTextTmp;
                } elsif ( $i == 1 ){
                    $sTextTmp =~ s/^\s+//g;    # strip leading spaces
                    $sTextTmp =~ s/\s+$//g;    # strip trailing spaces

                    if ( $hshIndexFieldsTmp == undef ){
                        $hshIndexFieldsTmp = {};
                    }
                    $hshIndexFieldsTmp->{ 'I_BRANCH_CODE' } = $sTextTmp;
                } elsif ( $i == 2 ){
                    $sTextTmp =~ s/^\s+//g;    # strip leading spaces
                    $sTextTmp =~ s/\s+$//g;    # strip trailing spaces

                    if ( $sTextTmp =~ /(.{30}).{27}(.{10})/ ){
                        if ( $hshIndexFieldsTmp == undef ){
                            $hshIndexFieldsTmp = {};
                        }
                        $hshIndexFieldsTmp->{ 'I_AC_NAME' } = $1;
                        $hshIndexFieldsTmp->{ 'I_ST_DATE' } = $2;
                        
                        $hshIndexFieldsTmp->{ 'I_AC_NAME' } =~ s/\s+$//g;    # strip trailing spaces
                    }
                }
            } 
        }            
        #---- Fetch next text
        $a2wTextTmp = $a2wFirstPagePar->getNextText();
    }

    if ( $hshIndexFieldsTmp != undef
         && $bLog == $TRUE
       ){
        printf STDERR ( "Index:"
                        . " Branch=" . $hshIndexFieldsTmp->{ 'I_BRANCH' }
                        . " BranchCode=" . $hshIndexFieldsTmp->{ 'I_BRANCH_CODE' }
                        . " AccountName=" . $hshIndexFieldsTmp->{ 'I_AC_NAME' }
                        . " StatementDate=" . $hshIndexFieldsTmp->{ 'I_ST_DATE' }
                        . "\n"
                      );
    }

    return $hshIndexFieldsTmp;
}

__END__
