#-------------------------------------------------------------------------------
#  any2html.pm:
#  Module to build HTML output
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -doc_cold -sp:sfsamples\any2html.pm samples\insure.afp
#
#  On Unix:    ./afp2web   -q -c -doc_cold -sp:sfsamples/any2html.pm samples/insure.afp
#
#  Author  : Fa. OXSEED
#  Date    : 2010-11-30
#  Version : 1.0.3
#
#  $V100   2009-10-23    Initial Release
#
#  $V101   2010-08-16    a. Extended to utilize
#                           "a2w::ConfigConstants"
#                           "a2w::DocumentConstants"
#                           "a2w::PageConstants"
#                           "a2w::FontConstants"
#                           packages
#                        b. Optimized constant declarations
#                        c. Modified PERL "@INC" extending to consider the
#                           Scripting Facility samples grouping under "sfsamples"
#                           directory
#                        d. Extended to support MMD/LPD inputs
#                        e. Extended to handle text with trailing/leading spaces
#
#  $V102   2010-11-29    Modified HTML generation to add CSS font mapping script block
#                        within "BODY" element instead of "HEAD" element
#
#  $V103   2010-11-30    Extended to generate HTML output in a string before writing in to
#                        a output file
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );

    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    # $V101 Begin
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }
    # $V101 End
    
    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}

use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;
# $V101 Begin
use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;
# $V101 End

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );   # $V101 Change
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );    # $V101 Change
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );         # $V101 Change
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );    # $V101 Change
    $sSpoolFileType    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SPOOLFILETYPE );     # $V101 Change
    $sSpoolType        = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INPUTFORMAT );       # $V101 Change
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    if ( $bLog == $TRUE ){
        printf STDERR ( "Running $sScriptProcTmp...\n" );
        printf STDERR ( "initialize(): Processing $sSpoolFilename\n" );
        printf STDERR ( "initialize(): Args: $sScriptArgsTmp, IndexFilePath: $sIndexFilePath, OutputFilePath: $sOutputFilePath\n" );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Configure output for HTML ----#
    $HTML_RESOLUTION = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::RESOLUTION );
    
    $sOutputFormat = "png";
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::RESOLUTION, $HTML_RESOLUTION );
    $a2wConfigPar->setAttribute( $a2w::ConfigConstants::OUTPUTFORMAT, $sOutputFormat );
    $a2wConfigPar->setAttribute( "PKEY", "UseGD" );
    
    #---- Font size scale factor
    $FONT_RESOLUTION = 72;
    $fFontScaleFactor = $HTML_RESOLUTION / $FONT_RESOLUTION;

    # $V101 Begin
    #---- Page formatting constants
    $iPageWidth  = 842.0;
    $iPageHeight = 595.0;

    $iLeftMargin  = 15.0;
    $iTopMargin   = 0.0;

    $a2wDefaultFont = undef;
    $sScriptUnitBase = "pixel";

    $bTRCExist = $FALSE;    # Flag indicating whether records have TRC or not

    $iLineSpacing = 9;

    #---- Initialize current page index fields line
    $iIndexFieldsLineNr = 64;
    $sIndexFieldsLine   = "";

    if ( lc( $sSpoolType ) eq "lpd" ){
        #---- Create default font
        $a2wDefaultFont = new a2w::Font( $a2w::FontConstants::TYPE_TYPE1, "Courier" );
        $a2wDefaultFont->setHeight( $iLineSpacing );

        #---- Set script unit base as Pixels    
        $a2wConfigPar->setAttribute( $a2w::ConfigConstants::SCRIPTUNITBASE, $sScriptUnitBase );
    }    
    # $V101 End

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializeDoc(): DocId " . $a2wDocumentPar->getId() . "\n" );
    }

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializePage()\n" );
    }

    #---- Page font table
    #
    # $hrefFontTable = {
    #    <Local ID> = <HTML Font definition>
    # }
    #
    $hrefPageFontTable = {};

    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    #---- Get spool page id
    my $iSpoolPageIdTmp = $a2wPagePar->getParseId();

    if ( $bLog == $TRUE ){
        printf STDERR ( "afp2web(): PageId " . $iSpoolPageIdTmp . "\n" );
    }

    #---- Set default return value
    my $iRetTmp = $APPEND; # append page to Current Document

    # $V101 Begin
    #---- Format page, for LPD input ----#
    if ( lc( $sSpoolType ) eq "lpd" ){
        my ( $iRcTmp, $sMsgTmp ) = _formatPage( $a2wPagePar );
        if ( $iRcTmp < 0 ){
            return ( $iRcTmp, $sMsgTmp );
        }
    }
    # $V101 End

    #---- Set required page output filename
    my $sSimplePageFilenameTmp = sprintf( "%s_%04d", $a2wDocumentPar->getName(), $a2wPagePar->getParseId() );
    my $sPageFilenameTmp = $sSimplePageFilenameTmp . "." . $sOutputFormat;
    $a2wPagePar->setOutputFilename( $sPageFilenameTmp );

    # $V103 Begin
    #---- Build page content as HTML
    my $sPageHTMLBufferTmp = _buildPageHTML( $a2wPagePar, $sPageFilenameTmp );
    # $V103 End

    #---- Open HTML file
    my $sHTMLFilenameTmp = $sOutputFilePath . $sSimplePageFilenameTmp . ".html";
    my $iOpenStatusTmp = open( HTMLFILE, ">$sHTMLFilenameTmp" );
    if ( $iOpenStatusTmp <= 0 ){
        printf STDERR ( "%s\n",
                        "Error! Unable to open " . $sHTMLFilenameTmp . ", reason: " . $!
                      );
        return -1;
    }

    #---- Write page html content ----#
    printf HTMLFILE ( $sPageHTMLBufferTmp );

    #---- Close HTML file
    close( HTMLFILE );

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizePage()\n" );
    }

    #---- Clear page font table
    $hrefPageFontTable = undef;

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizeDoc()\n" );
    }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalize()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Build HTML header meta info
#
#-----------------------------------------------------------------------
sub _buildHeaderMetaInfo{

    #if ( $bLog == $TRUE ){
    #    printf STDERR ( "_buildHeaderMetaInfo()\n" );
    #}

    my $sMetaInfoTmp = '<META http-equiv="Content-Type" content="text/html; charset=Windows-1252">';

    return $sMetaInfoTmp;
}

# $V103 Begin
#-----------------------------------------------------------------------
# Build page HTML
#
#-----------------------------------------------------------------------
sub _buildPageHTML{

    #if ( $bLog == $TRUE ){
    #    printf STDERR ( "_buildPageHTML()\n" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Page (of type a2w::Page)
    # 2. Page html filename (of type string)
    #
    my $a2wPageHandlePar = shift;
    my $sPageFilenamePar = shift;

    #---- Fetch page size
    my $iPageWidthTmp  = $a2wPageHandlePar->getWidth();
    my $iPageHeightTmp = $a2wPageHandlePar->getHeight();

    #---- EValute scale factor
    my $fScaleFactorTmp = $HTML_RESOLUTION / $a2wPageHandlePar->getResolution();

    #---- Modify page width and height to html resolution
    $iPageWidthTmp  *= $fScaleFactorTmp;
    $iPageHeightTmp *= $fScaleFactorTmp;

    #---- Round the values
    $iPageWidthTmp  = int( $iPageWidthTmp  + .5 * ( $iPageWidthTmp  <=> 0 ) );
    $iPageHeightTmp = int( $iPageHeightTmp + .5 * ( $iPageHeightTmp <=> 0 ) );

    #---- Build Header meta info
    if ( $sHeaderMetaInfo eq "" ){
        $sHeaderMetaInfo = _buildHeaderMetaInfo();
    }

    #---- Get spool page id
    my $iSpoolPageIdTmp = $a2wPageHandlePar->getParseId();

    #---- Build HTML header ----#
    my $sHeaderTmp =   "<HEAD>"
                     . "\n" . "<TITLE>Page " . $iSpoolPageIdTmp . "</TITLE>"
                     . "\n" . $sHeaderMetaInfo
                     . "\n" . "<DIV style=\"position:relative;width:" . $iPageWidthTmp . ";height:" . $iPageHeightTmp . ";\">"
                     . "\n" . "</HEAD>"    # $V102 Change
                     ;

    #---- Collect and remove text content
    my $sPageContentTmp = _collectAndRemoveTextContent( $a2wPageHandlePar, $fScaleFactorTmp );

    #---- Write page font table list
    my $sFontTableTmp = "";
    my %hshFontTableTmp = %{ $hrefPageFontTable };
    my @arrFontTableListTmp = sort keys( %hshFontTableTmp );
    if ( @arrFontTableListTmp > 0 ){
        #---- Begin font table style
        $sFontTableTmp =   '<STYLE type="text/css">'
                         . "\n" . '<!--';

        #---- Write font table entry
        for ( my $i = 0; $i < @arrFontTableListTmp; $i++ ){
            $sFontTableTmp .= "\n" . $hrefPageFontTable->{ @arrFontTableListTmp[ $i ] };
        }

        #---- End font table style
        $sFontTableTmp .=   "\n" . '-->'
                          . "\n" . '</STYLE>';
    }

    #---- Build HTML body ----#
    my $sBodyTmp =   '<BODY bgcolor="#FFFFFF">'
                   . "\n" . $sFontTableTmp
                   . "\n" . '<IMG width="' . $iPageWidthTmp . '" height="' . $iPageHeightTmp . '" src="' . $sPageFilenamePar . '" alt="background image">'
                   . $sPageContentTmp
                   . "\n" . '</BODY>'
                   ;

    #---- Write HTML footer
    printf HTMLFILE ( "\n</HTML>" );

    #---- Build page HTML ----#
    my $sPageHTMLTmp =   '<HTML>'
                       . "\n" . $sHeaderTmp
                       . "\n" . $sBodyTmp
                       . "\n" . '</HTML>'
                       ;

    return $sPageHTMLTmp;
}
# $V103 End

#-----------------------------------------------------------------------
# Fetch equivalent HTML font name
#
# Search for text font equivalent HTML font name and returns it
#
#-----------------------------------------------------------------------
sub _fetchHTMLFont{

    #if ( $bLog == $TRUE ){
    #    printf STDERR ( "_fetchHTMLFont()\n" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Text font name (of type string)
    #
    my $sTextFontNamePar = shift;

    my $sHTMLFontNameTmp = $sTextFontNamePar;
    if ( $bLog == $TRUE ){
        printf STDERR ( "_fetchHTMLFont(), Font=" . $sHTMLFontNameTmp . "\n" );
    }

    return $sHTMLFontNameTmp;
}

#-----------------------------------------------------------------------
# Is Font Bold
#
# Checks whether given font has BOLD weight and returns true if yes
#
#-----------------------------------------------------------------------
sub _isFontBold{

    #if ( $bLog == $TRUE ){
    #    printf STDERR ( "_isFontBold()\n" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Font (of type a2w::Font)
    #
    my $a2wFontPar = shift;

    my $bBoldTmp = $a2wFontPar->isBold();

    return $bBoldTmp;
}

#-----------------------------------------------------------------------
# Is Font Italic
#
# Checks whether given font has Italic slant and returns true if yes
#
#-----------------------------------------------------------------------
sub _isFontItalic{

    #if ( $bLog == $TRUE ){
    #    printf STDERR ( "_isFontItalic()\n" );
    #}

    #---- Fetch parameter(s)
    #
    # 1. Font (of type a2w::Font)
    #
    my $a2wFontPar = shift;

    my $bItalicTmp = $a2wFontPar->isItalic();

    return $bItalicTmp;
}

#-----------------------------------------------------------------------
# Collect and remove text content from page
#-----------------------------------------------------------------------
sub _collectAndRemoveTextContent{

    if ( $bLog == $TRUE ){
        printf STDERR ( "_collectAndRemoveTextContent()\n" );
    }

    #---- Fetch parameter(s)
    #
    # 1. Page (of type a2w::Page)
    # 2. Scale factor (of type float)
    #
    my $a2wCurrentPagePar = shift;
    my $fScaleFactorPar   = shift;

    #---- Loop through page text content and remove all
    my $a2wTextTmp = $a2wCurrentPagePar->getFirstText();

    my $fXPosTmp = 0.0;
    my $fYPosTmp = 0.0;

    my $a2wFontTmp = undef;
    my $iFontLocalIdTmp = -1;

    my $sFontStyleNameTmp = "";
    my $sTextContentTmp = "";

    my $iFontHeightTmp = 0;
    my $bBoldFontTmp = $FALSE;
    my $bItalicFontTmp = $FALSE;

    while ( $a2wTextTmp != 0 ){
        #---- Fetch text position
        $fXPosTmp = $a2wTextTmp->getXPos() * $fScaleFactorPar;
        $fYPosTmp = $a2wTextTmp->getYPos() * $fScaleFactorPar;
        
        #---- Round text position
        $fXPosTmp = int( $fXPosTmp + .5 * ( $fXPosTmp <=> 0 ) );
        $fYPosTmp = int( $fYPosTmp + .5 * ( $fYPosTmp <=> 0 ) );
        
        #---- Fetch text font
        $a2wFontTmp = $a2wTextTmp->getFont();

        #---- Fetch font local id
        $iFontLocalIdTmp = $a2wTextTmp->getMappedFontLocalId();
        $sFontStyleNameTmp = "ft" . sprintf( "%03d", $iFontLocalIdTmp );

        #---- Evaluate font attributes
        $iFontHeightTmp = int($a2wFontTmp->getHeight() * $fFontScaleFactor);
        $bBoldFontTmp = _isFontBold( $a2wFontTmp );
        $bItalicFontTmp = _isFontItalic( $a2wFontTmp );
        
        # $V101 Begin
        # Check for leading/trailing spaces
        my $bPreFormattedTmp = $FALSE;
        my $sTextTmp = $a2wTextTmp->getText();
        if (    $sTextTmp =~ /^\s{1,}/
             || $sTextTmp =~ /\s(1,)$/
             || $sTextTmp =~ /\s{2,}/
           ){
           $bPreFormattedTmp = $TRUE;
        }
        # $V101 End
        
        #---- Check whether font style is already created or not
        #     If not created already, create it
        if ( $hrefPageFontTable->{ $iFontLocalIdTmp } ){    # Already created
            # Do nothing
        }
        else {                                          # Not created already, create NOW
            #---- Build font class style
            $hrefPageFontTable->{ $iFontLocalIdTmp } = "."
                                                       . $sFontStyleNameTmp
                                                       . '{'
                                                       . 'font-size:' . $iFontHeightTmp . 'px;'
                                                       . 'font-family:' . _fetchHTMLFont( $a2wFontTmp->getName() ) . ';'
                                                       . '}'
                                                       ;
        }
        #---- Add text to page content
        $sTextContentTmp .= "\n"
                            . '<DIV style="position:absolute;'
                            . 'left:' . $fXPosTmp . ';'
                            . 'top:'  . ( $fYPosTmp - $iFontHeightTmp ) . ';'
                            . '"'
                            . '>'
                            . '<SPAN' . ( ( $bPreFormattedTmp == $TRUE ) ? '' : ' class="' . $sFontStyleNameTmp . '"' )
                            . ' style="color:#' . sprintf( "%06X", $a2wTextTmp->getColor() ) . ';"'
                            . '>'
                            . ( ( $bBoldFontTmp == $TRUE ) ? '<B>' : '' )
                            . ( ( $bItalicFontTmp == $TRUE ) ? '<I>' : '' )
                            # $V101 Begin
                            . ( ( $bPreFormattedTmp == $TRUE ) ? '<PRE class="' . $sFontStyleNameTmp . '">' : '' )
                            . $sTextTmp
                            . ( ( $bPreFormattedTmp == $TRUE ) ? '</PRE>' : '' )
                            # $V101 End
                            . ( ( $bItalicFontTmp == $TRUE ) ? '</I>' : '' )
                            . ( ( $bBoldFontTmp == $TRUE ) ? '</B>' : '' )
                            . '</SPAN>'
                            . '</DIV>'
                            ;
        
        #---- Remove text from presentation
        $a2wTextTmp->remove();

        #---- Fetch next text
        $a2wTextTmp = $a2wCurrentPagePar->getNextText();
    }

    return $sTextContentTmp;
}

# $V101 Begin
#-----------------------------------------------------------------------
# Format page
#
# !!! USED ONLY FOR LPD PAGE !!!
#
# Parameter(s)
# 1. Page (of type a2w::Page)
#
# Returns
# >0 in case of success
# <0 (+ the reason message) in case of error
#
# Remarks
# Processes each text (aka line text) of input and formats it based on
# ASA carriage controls
#
# *** IMPORTANT NOTE ON PROCESSING LINES ***:
# AFP2web will process each line of input and will make a text object
# out of each line of text. So processing a text object mean processing
# a line from input.
#
# Also the text (aka line) will have Carriage Control character at the
# beginning which can be processed in below function for formatting the
# lines
#
#-----------------------------------------------------------------------
sub _formatPage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "_formatPage()\n" );
    }

    #---- Get parameter(s)
    #
    # 1. Page (of type a2w::Page)
    #
    my $a2wCurrentPageTmp = shift;

    #---- Fetch page resolution
    my $iPageResTmp = $a2wCurrentPageTmp->getResolution();

    #---- Set new page width/height
    $a2wCurrentPageTmp->setHeight( $iPageHeight );
    $a2wCurrentPageTmp->setWidth( $iPageWidth );

    #---- Evaluate line height
    my $fFactorTmp = $iPageResTmp / 72;

    if ( lc( $sScriptUnitBase ) eq "mm" ){
        $fFactorTmp = 25.4 / 72;
    }

    #---- Initialize LineId
    my $iLineIdTmp = 0;

    #---- Origin is at the bottom left corner
    my $iLeftMarginTmp = $iLeftMargin * $fFactorTmp;
    my $iTopMarginTmp  = $iTopMargin * $fFactorTmp;

    #---- Modify boundary evaluation based on Top-Left co-ordinate system
    my $iXPosTmp = $iLeftMarginTmp;    # X Position starts from Left Margin
    my $iYPosTmp = $iTopMarginTmp;     # Y Position is "Top Margin"

    #---- Process and format page content ----#
    my $a2wTextTmp = $a2wCurrentPageTmp->getFirstText();

    #---- Format each and every text (aka line text)
    my $sTextTmp  = "";
    my $cCCCharTmp = "";
    my $cTRCharTmp = "";
    my $sTextWithoutCCTmp = "";

    #---- Index Field Insured Name
    my $sInsuredNameTmp = "";
    
    while ( $a2wTextTmp != 0 ){
        $iLineIdTmp++;                 # Increment line number
        $iXPosTmp = $iLeftMarginTmp;   # X Position starts from Left Margin

        #---- Assert Y position
        #---- Modify boundary evaluation based on Top-Left co-ordinate system
        if ( $iYPosTmp > $iPageHeight ){
            if ( $bLog == $TRUE ){
                printf STDERR "\tERROR! Spool File Type (" . $sSpoolFileType . "), Line=" . $iLineIdTmp . ": Not enough space on page.\n";
            }

            return ( -98, "Spool File Type (" . $sSpoolFileType . "), Line=" . $iLineIdTmp . ": Not enough space on page." );
        }

        #---- Process current line and add texts appropriately
        $sTextWithoutCCTmp = "";
        $sTextTmp = $a2wTextTmp->getText();

        #---- Collect current page's index fields line
        if ( $iLineIdTmp == $iIndexFieldsLineNr ){
            $sIndexFieldsLine = $sTextTmp;
        }

        if ( $bTRCExist == $TRUE ){
            if ( $sTextTmp =~ /(.)(.)(.*)/ ){
                $cCCCharTmp        = $1;   # Contains CC
                $cTRCharTmp        = $2;   # Contains TRC
                $sTextWithoutCCTmp = $3;   # Contains line data
            }
        }
        else {
            if ( $sTextTmp =~ /(.)(.*)/ ){
                $cCCCharTmp        = $1;   # Contains CC
                $sTextWithoutCCTmp = $2;   # Contains line data
            }
        }

        #----- ASA Carriage Control action is applied before printing current line
        #      so process apply actions for given CC character
        #
        if ( $cCCCharTmp eq "+" ){                   # Do not space
            # Overwrite existing line
            $iYPosTmp -= $iLineSpacing;
        }
        elsif ( $cCCCharTmp eq " " ){                # Space one line
            # One line space is already evaluated,
            # so nothing to do here
        }
        elsif ( $cCCCharTmp eq "0" ){                # Space two lines
            # One line space is already evaluated,
            # so leave only one line space now
            $iYPosTmp += $iLineSpacing;
        }
        elsif ( $cCCCharTmp eq "-" ){                # Space three lines
            # One line space is already evaluated,
            # so leave only two line space now
            $iYPosTmp += $iLineSpacing;
            $iYPosTmp += $iLineSpacing;
        }
        elsif ( $cCCCharTmp eq "1" ){                # Skip to channel 1/New page
            # A2W kernel itself identifies new page,
            # so nothing to do here
        }
        elsif ( $cCCCharTmp eq "2" ){                # Skip to channel 2
        }
        elsif ( $cCCCharTmp eq "3" ){                # Skip to channel 3
        }
        elsif ( $cCCCharTmp eq "4" ){                # Skip to channel 4
        }
        elsif ( $cCCCharTmp eq "5" ){                # Skip to channel 5
        }
        elsif ( $cCCCharTmp eq "6" ){                # Skip to channel 6
        }
        elsif ( $cCCCharTmp eq "7" ){                # Skip to channel 7
        }
        elsif ( $cCCCharTmp eq "8" ){                # Skip to channel 8
        }
        elsif ( $cCCCharTmp eq "9" ){                # Skip to channel 9
        }
        elsif ( $cCCCharTmp eq "A" ){                # Skip to channel 10
        }
        elsif ( $cCCCharTmp eq "B" ){                # Skip to channel 11
        }
        elsif ( $cCCCharTmp eq "C" ){                # Skip to channel 12
        }

        #---- Skip trailing spaces
        $sTextWithoutCCTmp =~ s/\s+$//g;

        #---- Skip empty lines
        if ( length( $sTextWithoutCCTmp ) <= 0 ){
            #---- Evaluate next YPos (YPos = YPos - FontHeight)
            $iYPosTmp += $iLineSpacing;

            #---- Don't present this text on output
            $a2wTextTmp->remove();
        }
        else {
            if ( $bLog == $TRUE ){
                printf STDERR ( "(" . $iXPosTmp . ", " . $iYPosTmp . ")=>" . $sTextWithoutCCTmp . "<=\n" );
            }

            #---- Fill in text details
            $a2wTextTmp->setText( $sTextWithoutCCTmp ); # Text value
            $a2wTextTmp->setXPos( $iXPosTmp );          # Text X position
            $a2wTextTmp->setYPos( $iYPosTmp );          # Text Y position
            $a2wTextTmp->setFont( $a2wDefaultFont );    # Text font

            #---- Evaluate next YPos
            $iYPosTmp += $iLineSpacing;
        }

        #---- Get next text
        $a2wTextTmp = $a2wCurrentPageTmp->getNextText();
    }

    if ( $bLog == $TRUE ){
        printf STDERR ( "\tIndexFields=" . $sIndexFieldsLine . "\n" );
    }

    return 0;
}
# $V101 End

__END__
