#-------------------------------------------------------------------------------
#  processDocInfo.pm:
#  Scripting Facility Template Module. It is meant to be used as a template to
#  access/process document information like properties, bookmarks, metadata
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -afp -doc_cold -sp:sfsamples\processDocInfo.pm samples\insure.pdf
#
#  On Unix:    ./afp2web   -q -c -afp -doc_cold -sp:sfsamples/rocessDocInfo.pm samples/insure.pdf
#
#  Author  : Fa. OXSEED
#  Date    : 2010-07-19
#  Version : 1.0.0
#
#  $V100   2007-07-15    Initial Release
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );

    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );
}

use a2w::Bookmark;
use a2w::Comment;
use a2w::Config;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;
use a2w::ConfigConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    if ( $bLog == $TRUE ){
        printf STDERR ( "Running $sScriptProcTmp...\n" );
        printf STDERR ( "initialize(): Processing $sSpoolFilename\n" );
        printf STDERR ( "initialize(): Args: $sScriptArgsTmp, IndexFilePath: $sIndexFilePath, OutputFilePath: $sOutputFilePath\n" );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializeDoc(): DocId " . $a2wDocumentPar->getId() . "\n" );
    }

#INFO: This is a sample for processing the document properties
    _processDocumentProperties( $a2wDocumentPar );

#INFO: This is a sample for processing the document metadata
    _processDocumentMetadata( $a2wDocumentPar );

#INFO: This is a sample for processing the document bookmarks
    _processDocumentBookmarks( $a2wDocumentPar );

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){
        printf STDERR ( "initializePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Main entry method
# Return values:
#        < 0:    error
#         0:    append page to Current Document
#         1:    skip page
#         2:    first page / new document
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "afp2web(): PageId " . $a2wPagePar->getParseId() . "\n" );
    }

    #---- Set default return value
    my $iRetTmp = $APPEND; # append page to Current Document

#INFO: If you need to get Information from the Page (Bookmarks etc...), do it here.
#INFO: If you need to add/remove Information to/from the Page, do it here.

#INFO: This is a sample for processing the page bookmarks
    #_processPageBookmarks( $a2wPagePar );

    return $iRetTmp;
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizePage()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalizeDoc()\n" );
    }

#INFO: If you need to write Index Files, do it here.
#INFO: If you need to post process the Output File (for example, sending 
#INFO: it to the archive, do it here.
#INFO: You'll find an example in autosplit.pm and eyecatcher.pm

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){
        printf STDERR ( "finalize()\n" );
    }
    return 0;
}

#-----------------------------------------------------------------------
# Processes document properties
#
# Parameters:
#       a2w::Document instance
#
# Return Value
#       None
#-----------------------------------------------------------------------
sub _processDocumentProperties{
    if ( $bLog == $TRUE ){
        printf STDERR ( "_processDocumentProperties()\n" );
    }

    #---- Get Parameter
    #
    # a2w::Document
    #
    my $a2wDocPar = shift;

    #---- Assert parameter
    if ( $a2wDocPar == 0 ){
        if ( $bLog == $TRUE ){
            printf STDERR ( "\tInvalid document handle to process properties\n" );
        }

        return;
    }

    printf STDERR ( "********* " . $a2wDocPar->getName() . " document properties *********\n" );

    #---- Get property name list (as defined in "a2w::DocumentConstants")
    my @arrConstantListTmp = sort keys( %a2w::DocumentConstants:: );
    my $sPropNameTmp = "";
    my $sPropValueTmp = "";

    #---- Loop through each property and process them ----#
    for ( my $i = 0; $i < @arrConstantListTmp; $i++ ){
        #---- Print only constants whose name starts with "PROP" keyword
        if ( @arrConstantListTmp[ $i ] ne "" ){
            #---- Fetch the property details
            $sPropNameTmp  = ${ %a2w::DocumentConstants::->{ @arrConstantListTmp[ $i ] } };            
            $sPropValueTmp = $a2wDocumentPar->getProperty( $sPropNameTmp );

            if ( $sPropValueTmp ne "" ){
                print STDERR ( $sPropNameTmp
                               . "="
                               . $sPropValueTmp
                               . "\n"
                             );
            }    # if ( $sPropValueTmp ne "" ){
        }    # if ( @arrConstantListTmp[ $i ] =~ /^PROP/ ){
    }

    printf STDERR ( "********* End document properties *********\n\n" );
}

#-----------------------------------------------------------------------
# Processes document metadata
#
# Parameters:
#       a2w::Document instance
#
# Return Value
#       None
#-----------------------------------------------------------------------
sub _processDocumentMetadata{
    if ( $bLog == $TRUE ){
        printf STDERR ( "_processDocumentMetadata()\n" );
    }

    #---- Get Parameter
    #
    # a2w::Document
    #
    my $a2wDocPar = shift;

    #---- Assert parameter
    if ( $a2wDocPar == 0 ){
        if ( $bLog == $TRUE ){
            printf STDERR ( "\tInvalid document handle to process metadata\n" );
        }

        return;
    }

    printf STDERR ( "********* " . $a2wDocPar->getName() . " document metadata *********\n" );

    #---------------- Prrocess document meta data ----------------#
    my $sDocMetadataTmp = $a2wDocumentPar->getMetaDataStream();
    if ( $sDocMetadataTmp ne "" ){
        printf STDERR ( $sDocMetadataTmp . "\n" );
    }

    printf STDERR ( "********* End document metadata *********\n\n" );
}

#-----------------------------------------------------------------------
# Processes document bookmarks
#
# Parameters:
#       a2w::Document instance
#
# Return Value
#       None
#-----------------------------------------------------------------------
sub _processDocumentBookmarks{
    if ( $bLog == $TRUE ){
        printf STDERR ( "_processDocumentBookmarks()\n" );
    }

    #---- Get Parameter
    #
    # a2w::Document
    #
    my $a2wDocPar = shift;

    #---- Assert parameter
    if ( $a2wDocPar == 0 ){
        if ( $bLog == $TRUE ){
            printf STDERR ( "\tInvalid document handle to process bookmarks\n" );
        }

        return;
    }

    printf STDERR ( "********* " . $a2wDocPar->getName() . " document bookmarks *********\n" );

    #---- Fetch first bookmark
    my $a2wBookmarkTmp = $a2wDocPar->getFirstBookmark();

    #---- Loop through all bookmarks
    while( $a2wBookmarkTmp ){
        #---- Print bookmark
        _printBookmark( $a2wBookmarkTmp );

        #---- Process bookmark's child(ren), if any exist
        if ( $a2wBookmarkTmp->getFirstChild() ){
            _processDocumentBookmarkChildren( $a2wBookmarkTmp );
        }

        #---- Fetch next bookmark
        $a2wBookmarkTmp = $a2wDocPar->getNextBookmark();
    }

    printf STDERR ( "********* End document bookmarks *********\n\n" );
}

#-----------------------------------------------------------------------
# Processes page bookmarks
#
# Identifies page bookmarks by the "TargetName" attribute of bookmark
# processes bookmarks whose "TargetName" match with current page id only
#
# Parameters:
#       a2w::Page instance
#
# Return Value
#       None
#-----------------------------------------------------------------------
sub _processPageBookmarks{
    if ( $bLog == $TRUE ){
        printf STDERR ( "_processPageBookmarks()\n" );
    }

    #---- Get Parameter
    #
    # a2w::Page
    #
    my $a2wPageWithBMPar = shift;

    #---- Assert parameter
    if ( $a2wPageWithBMPar == 0 ){
        if ( $bLog == $TRUE ){
            printf STDERR ( "\tInvalid page handle to process bookmarks\n" );
        }

        return;
    }

    #---- Get page parse id
    my $iPageIdTmp = $a2wPageWithBMPar->getParseId();

    printf STDERR ( "********* " . $a2wPageWithBMPar->getName() . "(" . $iPageIdTmp . ") page bookmarks *********\n" );

    #---- Fetch first bookmark
    my $a2wBookmarkTmp = $a2wPageWithBMPar->getFirstBookmark();

    #---- Loop through all bookmarks
    my $sTargetTmp = "";
    while ( $a2wBookmarkTmp ){
        #---- Fetch bookmark target
        $sTargetTmp = $a2wBookmarkTmp->getTargetName();

        #---- Process bookmark that belong to page only
        if ( $sTargetTmp eq "$iPageIdTmp" ){
            #---- Print bookmark
            _printBookmark( $a2wBookmarkTmp );
        }

        #---- Process bookmark's child(ren), if any exist
        if ( $a2wBookmarkTmp->getFirstChild() ){
              _processPageBookmarkChildren( $a2wPageWithBMPar, $a2wBookmarkTmp );
        }

        #---- Fetch next bookmark
        $a2wBookmarkTmp = $a2wPageWithBMPar->getNextBookmark();
    }

    printf STDERR ( "********* End page bookmarks *********\n\n" );
}

#-----------------------------------------------------------------------
# Prints Bookmark
#
# Parameters:
#       a2w::Bookmark instance
#
# Return Value
#       None
#-----------------------------------------------------------------------
sub _printBookmark{
    if ( $bLog == $TRUE ){
        printf STDERR ( "_printBookmark()\n" );
    }

    #---- Get Parameter
    #
    # a2w::Bookmark instance
    #
    my $a2wBookmarkPar = shift;

    #---- Assert parameter
    if ( $a2wBookmarkPar == 0 ){
        if ( $bLog == $TRUE ){
            printf STDERR ( "\tInvalid bookmark to print\n" );
        }
    }

    #---- Fetch bookmark info
    my $sBookmarkTitleTmp = $a2wBookmarkPar->getTitle();    # Fetch bookmark title
    my $iLevelNrTmp = $a2wBookmarkPar->getLevelNumber();    # Fetch bookmark level number
    my $sTargetTmp = $a2wBookmarkPar->getTargetName();      # Fetch bookmark target name

    #---- Print Bookmark details
    #---- Level number is used for alignment of child bookmarks
    if ( $iLevelNrTmp > 0 ){
        printf STDERR ( '%*s%s (%s)%s'
                        , $iLevelNrTmp * 2
                        , " "
                        , $sBookmarkTitleTmp
                        , $sTargetTmp
                        , "\n"
                      );
    }
    else {
        printf STDERR ( '%s (%s)%s'
                        , $sBookmarkTitleTmp
                        , $sTargetTmp
                        , "\n"
                      );    
    }
}

#-----------------------------------------------------------------------
# Processes document bookmark children
#
# Parameters:
#       a2w::Bookmark instance
#
# Return Value
#       None
#-----------------------------------------------------------------------
sub _processDocumentBookmarkChildren{
    if ( $bLog == $TRUE ){
        printf STDERR ( "_processDocumentBookmarkChildren()\n" );
    }

    #---- Get Parameters
    #
    # a2w::Bookmark
    #
    my $a2wBookmarkPar = shift;

    #---- Fetch first child of bookmark
    my $a2wChildBookmarkTmp = $a2wBookmarkPar->getFirstChild(); 
    
    while( $a2wChildBookmarkTmp ){
        #---- Print bookmark
        _printBookmark( $a2wChildBookmarkTmp );
        
        #---- Process bookmark's child(ren), if any exist
        if ( $a2wChildBookmarkTmp->getFirstChild() ){
            _processDocumentBookmarkChildren( $a2wChildBookmarkTmp );
        } 
        
        #---- Fetch next child of bookmark
        $a2wChildBookmarkTmp = $a2wBookmarkPar->getNextChild();
    }
}

#-----------------------------------------------------------------------
# Processes page bookmark children
#
# Identifies page bookmarks by the "TargetName" attribute of bookmark
# processes bookmarks whose "TargetName" match with current page id only
#
# Parameters:
#       a2w::Page instance
#       a2w::Bookmark instance
#
# Return Value
#       None
#-----------------------------------------------------------------------
sub _processPageBookmarkChildren{
    if ( $bLog == $TRUE ){
        printf STDERR ( "_processPageBookmarkChildren()\n" );
    }

    #---- Get Parameter
    #
    # a2w::Page
    # a2w::Bookmark
    #
    my $a2wPageWithBMPar = shift;
    my $a2wBookmarkPar = shift;

    #---- Assert parameter
    if ( $a2wBookmarkPar == 0 ){
        if ( $bLog == $TRUE ){
            printf STDERR ( "\tInvalid bookmark handle to process children\n" );
        }

        return;
    }

    #---- Fetch page parse id
    my $iPageIdTmp = $a2wPageWithBMPar->getParseId();

    #---- Fetch first child of bookmark
    my $a2wChildBookmarkTmp = $a2wBookmarkPar->getFirstChild(); 

    #---- Loop through all bookmark children
    my $sTargetTmp = "";
    while ( $a2wChildBookmarkTmp ){
        #---- Fetch bookmark target
        $sTargetTmp = $a2wChildBookmarkTmp->getTargetName();

        #---- Process bookmark that belong to page only
        if ( $sTargetTmp eq "$iPageIdTmp" ){
            #---- Print bookmark
            _printBookmark( $a2wChildBookmarkTmp );
        }

        #---- Process any child Bookmarks
        if ( $a2wChildBookmarkTmp->getFirstChild() ){
             _processPageBookmarkChildren( $a2wPageWithBMPar, $a2wChildBookmarkTmp ); 
        } 
        
        #---- Fetch next child of bookmark
        $a2wChildBookmarkTmp = $a2wBookmarkPar->getNextChild();
    }
}

__END__
