#-------------------------------------------------------------------------------
#  a2w::core::dm::CompositeBlockFormatter.pm:
#  Data Mining Framework Composite Block Formatter
#
#  Collects sub blocks of current block and writes them with appropriate formatting
#
#  Author  : AFP2web Team, Maas Holding GmbH
#
#  $V100   2018-10-03    Initial Release
#
#-------------------------------------------------------------------------------
package a2w::core::dm::CompositeBlockFormatter;

#-----------------------------------------------------------------------
# Include required modules
#-----------------------------------------------------------------------
use a2w::TypeConstants;
use a2w::core::log::Logger;
use a2w::core::dm::Constants;
use a2w::core::dm::Block;
use a2w::core::dm::BlockFormatter;

#-----------------------------------------------------------------------
# Inherit from base class
#-----------------------------------------------------------------------
our @ISA = qw( a2w::core::dm::BlockFormatter );

#-----------------------------------------------------------------------
# Constructor
#-----------------------------------------------------------------------
sub new{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    #---- Instantiate from base class to inherit base class attributes
    my $this = a2w::core::dm::BlockFormatter->new( @_ );

    #---- Add this derived class specific attributes
    #$this->{ 'someattribute' } = 0;

    bless( $this, $class );

    #---- Get logger
    our $theLogger = a2w::core::log::Logger->getSingleton();
    our $bLog = $theLogger->isRegistered( __PACKAGE__ );
    #if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "new()" ); }

    return $this;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutators
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Accessors
#-----------------------------------------------------------------------

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# writeFormattedBlock
#
# Asserts whether object fall in given block and returns following value
#
#  0, in case of success
# <0, in case of error
#
#-----------------------------------------------------------------------
sub writeFormattedBlock{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "writeFormattedBlock()" ); }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "composite" ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Error! Composite block formatter can not process other type (" . $contDefTmp->getType() . ") of blocks<" ); }
        return -1;
    }

    #---- Assert block has content or not ----#
    my $arefObjsListTmp = $blkCurrentPar->getObjects();
    my @arrObjsListTmp = @{ $arefObjsListTmp };
    my $blkNextTmp = $blkCurrentPar->getNextRef(); # get next block in chain
    if ( @arrObjsListTmp <= 0 ){
        # When current block (or chained block) is empty, ensure to continue with next block in
        # chain (if any exists) else return no error (so that formatting completed properly)
        $blkCurrentPar->setFlushed( $TRUE );
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Skipped writing empty block (" . $blkCurrentPar->getId() . ")" ); }

        #---- Write chained block contents ----#
        if ( $blkNextTmp != undef ){
            #---- Check and create formatter
            my $fmtNextTmp = $blkNextTmp->getFormatter();
            if ( $fmtNextTmp == undef ){
                $blkNextTmp->_createContentFormatter();
                $fmtNextTmp = $blkNextTmp->getFormatter();
            }

            #---- Write next block
            $iRetTmp = $fmtNextTmp->writeFormattedContent( # Block
                                                             $blkNextTmp
                                                           # Visitor
                                                           , $outVisitorPar
                                                           # Line count
                                                           , 0
                                                         );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        return 0;
    }

    #---- Begin Composite block
    my $iRetTmp = $outVisitorPar->beginComposite( $blkCurrentPar );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- Write formatted content of block ----#
    $iRetTmp = $this->_writeFormattedContent( # Block
                                               $blkCurrentPar
                                              # Visitor
                                              , $outVisitorPar
                                              # Line count
                                              , 0
                                            );
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    #---- End Composite block
    if ( $this->isFinalized() == $FALSE ){ $iRetTmp = $outVisitorPar->endComposite( $blkCurrentPar ); }
    if ( $iRetTmp < 0 ){ return $iRetTmp; }

    return 0;
}

#-----------------------------------------------------------------------
# _writeFormattedContent
#
# Writes block content formatted as lines (low level implementation)
#
# Returns
# >=0, in case of successful writing
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub _writeFormattedContent{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_writeFormattedContent()" ); }

    #---- Parameter
    #
    # 1. Block
    # 2. Visitor
    # 3. Line count
    #
    my $blkCurrentPar = shift;
    my $outVisitorPar = shift;
    my $iLineCountPar = shift;
    if ( $bLog == $TRUE ){ $theLogger->logMessage(   "Block: Id=>" . $blkCurrentPar->getId() . "<" ); }

    #---- Set flushed
    $blkCurrentPar->setFlushed( $TRUE );

    #---- Fetch content definition of block
    my $contDefTmp = $blkCurrentPar->getContentDef();
    if ( lc( $contDefTmp->getType() ) ne "composite" ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Error! Composite block formatter can not process other type (" . $contDefTmp->getType() . ") of blocks<" ); }
        return -1;
    }
    my $arefSubBlocksTmp = $contDefTmp->getSubBlocks();
    my @arrSubBlocksTmp = @{ $arefSubBlocksTmp };
    if (    lc( ref( $arefSubBlocksTmp ) ) ne "array"
         || @arrSubBlocksTmp <= 0
       ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Error! Composite block does not have sub blocks defined in it" ); }
        return -2;
    }

    #---- Assert block has content or not ----#
    my $arefObjsListTmp = $blkCurrentPar->getObjects();
    my @arrObjsListTmp = @{ $arefObjsListTmp };
    my $blkNextTmp = $blkCurrentPar->getNextRef(); # get next block in chain
    if ( @arrObjsListTmp <= 0 ){
        # When current block (or chained block) is empty, ensure to continue with next block in
        # chain (if any exists) else return no error (so that formatting completed properly)
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Skipped writing empty block (" . $blkCurrentPar->getId() . ")" ); }

        #---- Write chained block contents ----#
        if ( $blkNextTmp != undef ){
            #---- Check and create formatter
            my $fmtNextTmp = $blkNextTmp->getFormatter();
            if ( $fmtNextTmp == undef ){
                $blkNextTmp->_createContentFormatter();
                $fmtNextTmp = $blkNextTmp->getFormatter();
            }

            #---- Write next block
            $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                              $blkNextTmp
                                                            # Visitor
                                                            , $outVisitorPar
                                                            # Line count
                                                            , $iLineCountPar
                                                          );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }
        return 0;
    }

    #---- Get sub blocks
    my $arefSubBlocksTmp = $contDefTmp->getSubBlockObjects();
    my @arrSubBlocksTmp = @{ $arefSubBlocksTmp };
    if ( @arrSubBlocksTmp <= 0 ){
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "Warning! Missing sub blocks for composite block (" . $blkCurrentPar->getId() . "), skipped writing" ); }

        # When current block (or chained block) is empty, ensure to continue with next block in
        # chain (if any exists) else return no error (so that formatting completed properly)

        #---- Write chained block contents ----#
        if ( $blkNextTmp != undef ){
            #---- Check and create formatter
            my $fmtNextTmp = $blkNextTmp->getFormatter();
            if ( $fmtNextTmp == undef ){
                $blkNextTmp->_createContentFormatter();
                $fmtNextTmp = $blkNextTmp->getFormatter();
            }

            #---- Write next block
            $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                              $blkNextTmp
                                                            # Visitor
                                                            , $outVisitorPar
                                                            # Line count
                                                            , $iLineCountPar
                                                          );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }
        }

        return 0;
    }

    #---- Process composite block objects and classify them sub block specific
    $this->_classifyContentAsSubBlocks( $blkCurrentPar, $arefObjsListTmp, $arefSubBlocksTmp );

    #---- Write sub blocks
    foreach my $subBlockTmp ( @arrSubBlocksTmp ){
        if ( $subBlockTmp == undef ){ next; }

        #---- Write sub block
        $subBlockTmp->write( $outVisitorPar );
    }

    #---- Write chained block contents ----#
    if ( $blkNextTmp != undef ){
        #---- Check and create formatter
        my $fmtNextTmp = $blkNextTmp->getFormatter();
        if ( $fmtNextTmp == undef ){
            $blkNextTmp->_createContentFormatter();
            $fmtNextTmp = $blkNextTmp->getFormatter();
        }

        #---- Assert whether next block is of same type
        # a. If it is same, continue current block state on visitor
        # b. If it is NOT same, finalize current block state on visitor and let next block started properly
        if ( lc( $blkNextTmp->getContentDefType() ) eq "composite" ){
            #---- Continue next block writing
            $iRetTmp = $fmtNextTmp->_writeFormattedContent( # Block
                                                              $blkNextTmp
                                                            # Visitor
                                                            , $outVisitorPar
                                                            # Line count
                                                            , $iLineCountPar
                                                          );
            $this->setFinalized( $fmtNextTmp->isFinalized() ); # Current block is also finalized, if block is finalized on next block
        }
        else {
            #---- Finalize current block
            $this->setFinalized( $TRUE );
            $iRetTmp = $outVisitorPar->endComposite( $blkCurrentPar );
            if ( $iRetTmp < 0 ){ return $iRetTmp; }

            #---- Write next block
            $iRetTmp = $blkNextTmp->write( $outVisitorPar );
        }
        if ( $iRetTmp < 0 ){ return $iRetTmp; }
    }

    return 0;
}

#-----------------------------------------------------------------------
# _classifyContentAsSubBlocks
#
# Process composite block objects and classify them sub block specific
#
# Returns
# >=0, in case of success
# < 0, in case of error
#
#-----------------------------------------------------------------------
sub _classifyContentAsSubBlocks{
    my $this = shift;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( __PACKAGE__, "_classifyContentAsSubBlocks()" ); }

    #---- Parameter
    #
    # 1. Composite block
    # 2. Array of composite block objects
    # 3. Array of sub blocks
    #
    my $compositeBlockPar = shift;
    my $arefObjsListPar = shift;
    my $arefSubBlocksPar = shift;

    my $bStartAnchorTmp = $FALSE;
    my @arrObjsListTmp = @{ $arefObjsListPar };
    my @arrSubBlocksTmp = @{ $arefSubBlocksPar };

    #---- Iterate through composite objects and classify them specific to sub blocks
    my $subBlockTmp = undef;
    foreach my $objTmp ( @arrObjsListTmp ){
        if ( $bLog == $TRUE ){ $this->_logObject( $objTmp->{ 'A2WOBJ' }, $objTmp->{ 'POMOBJ' } ); }

        foreach $subBlockTmp ( @arrSubBlocksTmp ){
            $bStartAnchorTmp = $FALSE;

            #---- Check whether object fall in current sub block, if yes add the object to current sub block
            if ( $subBlockTmp->isStartAnchor(   $objTmp->{ $a2w::core::dm::Constants::AT_PAGEREF }
                                              , $objTmp->{ 'A2WOBJ' }
                                              , $objTmp->{ 'POMOBJ' }
                                            )
               ){
                $bStartAnchorTmp = $TRUE;

                #---- Initialize sub block
                $subBlockTmp->addStartAnchor( $objTmp->{ 'A2WOBJ' }, $objTmp->{ 'POMOBJ' } );
            }

            if (    $bStartAnchorTmp == $FALSE
			     && $subBlockTmp->doesObjectFallIn(   $objTmp->{ $a2w::core::dm::Constants::AT_PAGEREF }
                                                    , $objTmp->{ 'A2WOBJ' }
                                                    , $objTmp->{ 'POMOBJ' }
                                                  )
               ){
                #---- Add object to sub block
                $subBlockTmp->addObject( $objTmp->{ 'A2WOBJ' }, $objTmp->{ 'POMOBJ' } );
            }

            if ( $subBlockTmp->isEndAnchor(   $objTmp->{ $a2w::core::dm::Constants::AT_PAGEREF }
                                            , $objTmp->{ 'A2WOBJ' }
                                            , $objTmp->{ 'POMOBJ' }
                                          )
               ){
                #---- Finalize sub block
                $subBlockTmp->addEndAnchor( $objTmp->{ 'A2WOBJ' }, $objTmp->{ 'POMOBJ' } );
            }
        } # foreach my $subBlockTmp ( @arrSubBlocksTmp )
    } # foreach my $objTmp ( @arrObjsListTmp )

    #---- Iterate sub blocks, mark it as filled (needed for region like blocks) and update all blocks
    foreach $subBlockTmp ( @arrSubBlocksTmp ){
        #---- Mark block as filled if it is region block
        if ( $subBlockTmp->isRegion() == $TRUE && $subBlockTmp->isFilled() == $FALSE ){ $subBlockTmp->setAsFilled(); }

        #---- Update block
        $subBlockTmp->setStartedOn( $compositeBlockPar->getStartedOn() );
        $subBlockTmp->update();
    }

    return 0;
}

#-----------------------------------------------------------------------
# Log object
#-----------------------------------------------------------------------
sub _logObject{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Object
    # 2. POM object wrapper (optional)
    #
    my $a2wObjectPar = shift;
    my $pomObjectPar = undef;
    if ( @_ > 0 ){
        $pomObjectPar = shift;
    }

    #---- Build log message ----#
    my $sObjTypeTmp = $a2wObjectPar->_getType();
    my $sLogMsgTmp  = "";
    my $sLogObjTmp  = "";

    if ( $pomObjectPar->{ $a2w::core::dm::Constants::AT_PAGEFIRST } == $TRUE ){ $sLogObjTmp  = " Page first object"; }
    if ( $pomObjectPar->{ $a2w::core::dm::Constants::AT_PAGELAST } == $TRUE ){
        if ( $sLogObjTmp ne "" ){ $sLogObjTmp = " Page first and last object"; }
        else { $sLogObjTmp = " Page last object"; }
    }

    if ( $sObjTypeTmp eq "image" ){
        if ( $pomObjectPar == undef ){
            $sLogMsgTmp = "img>" . $a2wObjectPar->getName() . "<@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ") W=>" . $a2wObjectPar->getWidth() . "< H=>" . $a2wObjectPar->getHeight() . "<$sLogObjTmp";
        }
        else {
            $sLogMsgTmp = "img>" . $a2wObjectPar->getName() . "<@(" . $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . "," . $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ") W=>" . $a2wObjectPar->getWidth() . "< H=>" . $a2wObjectPar->getHeight() . "<$sLogObjTmp";
        }
    }
    if ( $sObjTypeTmp eq "container" ){
        if ( $pomObjectPar == undef ){
            $sLogMsgTmp = "obj>" . $a2wObjectPar->getName() . "<@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ") W=>" . $a2wObjectPar->getWidth() . "< H=>" . $a2wObjectPar->getHeight() . "<$sLogObjTmp";
        }
        else {
            $sLogMsgTmp = "obj>" . $a2wObjectPar->getName() . "<@(" . $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . "," . $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ") W=>" . $a2wObjectPar->getWidth() . "< H=>" . $a2wObjectPar->getHeight() . "<$sLogObjTmp";
        }
    }
    elsif ( $sObjTypeTmp eq "line" ){
        if ( $pomObjectPar == undef ){
            $sLogMsgTmp = "lin><@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ") W=>" . $a2wObjectPar->getWidth() . "< L=>" . $a2wObjectPar->getLength() . "<$sLogObjTmp";
        }
        else {
            $sLogMsgTmp = "lin><@(" . $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . "," . $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ") W=>" . $a2wObjectPar->getWidth() . "< L=>" . $a2wObjectPar->getLength() . "<$sLogObjTmp";
        }
    }
    elsif ( $sObjTypeTmp eq "text" ){
        if ( $pomObjectPar == undef ){
            $sLogMsgTmp = "txt>" . $a2wObjectPar->getText() . "<@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ")$sLogObjTmp";
        }
        else {
            $sLogMsgTmp = "txt>" . $a2wObjectPar->getText() . "<@(" . $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . "," . $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ")$sLogObjTmp";
        }
    }
    elsif ( $sObjTypeTmp eq "vector" ){
        if ( $pomObjectPar == undef ){
            $sLogMsgTmp = "vec><@(" . $a2wObjectPar->getXPos() . "," . $a2wObjectPar->getYPos() . ")$sLogObjTmp";
        }
        else {
            $sLogMsgTmp = "vec><@(" . $pomObjectPar->{ $a2w::core::dm::Constants::AT_XPOS } . "," . $pomObjectPar->{ $a2w::core::dm::Constants::AT_YPOS } . ")$sLogObjTmp";
        }
    }

    $theLogger->logMessage( $sLogMsgTmp );
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__
