#-------------------------------------------------------------------------------
#  a2w/core/log/Logger.pm:
#
#  SINGLETON logging engine
#
#  Author  : AFP2web Team, Maas Holding GmbH
#  Date    : 2013-10-11
#  Version : 1.0.0
#
#  $V100   2013-10-11    Initial Release
#
#-------------------------------------------------------------------------------
package a2w::core::log::Logger;

use a2w::TypeConstants;   # Global constants and variables

#-----------------------------------------------------------------------
# Constants
#-----------------------------------------------------------------------
$LEVEL_INFO  = 1;
$LEVEL_WARN  = 2;
$LEVEL_ERROR = 4;
$LEVEL_DEBUG = 8;

#-----------------------------------------------------------------------
# Private variables
#-----------------------------------------------------------------------
my $theLogger = undef;

#-----------------------------------------------------------------------
# Get singleton
#-----------------------------------------------------------------------
sub getSingleton{
    my $proto = shift;
    my $class = ref( $proto ) || $proto;

    return $theLogger if defined $theLogger;

    #---- Define boolean values
    $TRUE  = $a2w::TypeConstants::TRUE;     # TRUE  boolean value
    $FALSE = $a2w::TypeConstants::FALSE;    # FALSE boolean value

    my $this = {
          'bLog'                  => $FALSE # Enable/Disable logging in modules
        , 'arefRegisteredClasses' => 0      # Array reference having list of registered class names
        , 'bFileOpen'             => $FALSE # Flag indicating whether log file is opened or not
        , 'sFilename'             => ""     # Log file name
        , 'fHandle'               => -100   # Log file handle
        , 'tStartTime'            => 0      # Start time
        , 'tEndTime'              => 0      # End time
        , 'Level'                 => 0      # Log levels (1 => Info, 2 => Warn, 4 => Error, 8 => Debug)
    };

    $theLogger = bless( $this, $class );

    #---- Get Parameters
    #
    # 1. Log flag (of type boolean)
    #
    $theLogger->{ 'bLog' } = shift;

    return $theLogger;
}

#-----------------------------------------------------------------------
# Destructor
#-----------------------------------------------------------------------
sub DESTROY{
    my $this = shift;
}

#-----------------------------------------------------------------------
# Mutator
#-----------------------------------------------------------------------
sub setLog{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Log flag (of type boolean)
    #
    my $bLogPar = shift;

    #---- Set log flag
    $this->{ 'bLog' } = $bLogPar;
}

sub setFilename{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Log filename (of type string)
    #
    my $sLogFilenamePar = shift;

    #---- Set log filename
    $this->{ 'sFilename' } = $sLogFilenamePar;
}

sub setStartTime{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Start time (of type time)
    #
    $tStartTimePar = shift;

    #---- Set Start time
    $this->{ 'tStartTime' } = $tStartTimePar;
}

sub setEndTime{
    my $this = shift;

    #---- Get parameter
    #
    # 1. End time (of type time)
    #
    $tEndTimePar = shift;

    #---- Set End time
    $this->{ 'tEndTime' } = $tEndTimePar;
}

sub setLevel{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Level
    #
    $iLevelPar = shift;

    #---- Set level
    $this->{ 'Level' } = $iLevelPar;
}

#-----------------------------------------------------------------------
# Accessor
#-----------------------------------------------------------------------
sub getLog{
    my $this = shift;

    #---- Return log flag
    return $this->{ 'bLog' };
}

sub getStartTime{
    my $this = shift;

    #---- Get Start time
    return $this->{ 'tStartTime' };
}

sub getEndTime{
    my $this = shift;

    #---- Get End time
    return $this->{ 'tEndTime' };
}

sub getFilename{
    my $this = shift;

    #---- Get log filename
    return $this->{ 'sFilename' };
}

sub getLevel{
    my $this = shift;

    #---- Get level
    return $this->{ 'Level' };
}

#-----------------------------------------------------------------------
# Workers
#-----------------------------------------------------------------------
#-----------------------------------------------------------------------
# Initialize 
#-----------------------------------------------------------------------
sub initialize{
    my $this = shift;

    #---- Get parameter
    my $tStartTimePar = shift;
    my $sPathPar = shift;
    my $sFilePrefixPar = shift;
    my $bLogPar = shift;

    $this->setLog( $bLogPar );       
    $this->setStartTime( $tStartTimePar );

    #---- Open log file
    $this->open( $sPathPar, $sFilePrefixPar );
}

#-----------------------------------------------------------------------
# Register a class additionally for logging
#-----------------------------------------------------------------------
sub registerAdditionalClass{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Class name (of type string)
    #
    my $sClassnamePar = shift;

    #---- Get registered classes list
    $arefClassListTmp = $this->{ 'arefRegisteredClasses' };

    #---- Add given class
    my $iLenTmp = @{ $arefClassListTmp };
    $arefClassListTmp->[ $iLenTmp ] = lc( $sClassnamePar );
}

#-----------------------------------------------------------------------
# Register collection of classes for logging
#-----------------------------------------------------------------------
sub registerClasses{
    my $this = shift;

    #---- Get parameter
    #
    # 1. List of comma separated class names (of type string)
    #
    my $sClassnameListPar = shift;

    #---- Split and create array
    my @arrClassListTmp = split( ',', $sClassnameListPar );

    #---- Convert all class names to lowercase for easy comparison on later
    for( my $i = 0; $i < @arrClassListTmp; $i++ ){
        @arrClassListTmp[ $i ] = lc( @arrClassListTmp[ $i ] );
    }

    #---- Set registered classes
    $this->{ 'arefRegisteredClasses' } = \@arrClassListTmp;
}

#-----------------------------------------------------------------------
# Is class registered for logging
#-----------------------------------------------------------------------
sub isRegistered{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Class name (of type string)
    #
    my $sClassnamePar = shift;

    #---- Get registered classes
    my @arrClassListTmp = @{ $this->{ 'arefRegisteredClasses' } };
    my $iCountTmp = @arrClassListTmp;

    if ( $iCountTmp == 1
         && lc( @arrClassListTmp[ 0 ] ) eq "all" ){
        return $TRUE;
    }

    my $bRetTmp = $FALSE;
    my $sClassnameTmp = lc( $sClassnamePar );

    for( my $i = 0; $i < $iCountTmp; $i++ ){
        if ( $sClassnameTmp eq @arrClassListTmp[ $i ] ){
            $bRetTmp = $TRUE;

            last;   # break the loop
        }
    }

    #---- Return log flag
    return $bRetTmp;
}

#-----------------------------------------------------------------------
# Open file
#-----------------------------------------------------------------------
sub open{
    my $this = shift;

    #---- Get parameter
    #
    # 1. Log path (of type string)
    # 2. Prefix of the Log filename (of type string), optional
    #
    my $sPathPar = shift;
    my $sPrefixPar = "";
    if ( @_ > 0 ){
        $sPrefixPar = shift;
    }

    if ( $this->{ 'sFilename' } eq "" ){
        #---- Store start time
        if ( $this->{ 'tStartTime' } == 0 ){
            $this->{ 'tStartTime' } = time();
        }

        #---- Build filename
        #   0    1    2     3     4    5     6     7     8
        #($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);

        my @arrCurrentTimeTmp = localtime( $this->{ 'tStartTime' } );
        @arrCurrentTimeTmp[ 5 ] += 1900;        # Year is not complete 4 digit, so adding 1900
        @arrCurrentTimeTmp[ 4 ]++;              # Day of month starts from 0, so adding 1

        if ( $sPrefixPar ne "" ){
            $this->{ 'sFilename' } = $sPrefixPar . "_";
        }

        $this->{ 'sFilename' } .= sprintf(   "%04d%02d%02d_%02d%02d%02d_%06d_%03d.log"
                                           , @arrCurrentTimeTmp[ 5 ]
                                           , @arrCurrentTimeTmp[ 4 ]
                                           , @arrCurrentTimeTmp[ 3 ]
                                           , @arrCurrentTimeTmp[ 2 ]
                                           , @arrCurrentTimeTmp[ 1 ]
                                           , @arrCurrentTimeTmp[ 0 ]
                                           , $$             # Process id
                                           , rand( 999 )    # 3 digit random value
                                         );
    }

    if ( $sPathPar ne "" ){
        $this->{ 'sFilename' } = $sPathPar . $this->{ 'sFilename' };
    }

    #---- Open file
    if ( open( $this->{ 'fHandle' }, ">" . $this->{ 'sFilename' } ) ){
        $this->{ 'bFileOpen' } = $TRUE;

        #---- Turn on auto flush for log file handle
        select( ( select( $this->{ 'fHandle' } ), $| = 1 )[ 0 ] );
    }

    return $this->{ 'bFileOpen' };
}

#-----------------------------------------------------------------------
# Close Log File
#-----------------------------------------------------------------------
sub close{
    my $this = shift;

    if ( $this->{ 'bFileOpen' } == $FALSE ){
        return -1;
    }

    #---- Open file
    close( $this->{ 'fHandle' } );
    $this->{ 'bFileOpen' } = $FALSE;

    return 0;
}

#-----------------------------------------------------------------------
# Log function name
#-----------------------------------------------------------------------
sub logFunctionName{
    my $this = shift;

    if ( $this->{ 'bFileOpen' } == $FALSE ){
        return -1;
    }

    #---- Get parameter
    #
    # 1. Module name (of type string)
    # 2. Function name (of type string)
    #
    my $sModuleNamePar = shift;
    my $sFuncNamePar = shift;

    my $hFileHandleTmp = $this->{ 'fHandle' };
    printf $hFileHandleTmp ( "[%-54s] %s\n"
                             , $sModuleNamePar
                             , $sFuncNamePar
                           );

    return 0;
}

#-----------------------------------------------------------------------
# Log message
#-----------------------------------------------------------------------
sub logMessage{
    my $this = shift;

    if ( $this->{ 'bFileOpen' } == $FALSE ){
        return -1;
    }

    #---- Get parameter
    #
    # 1. Message to log (of type string)
    #
    my $sMessagePar = shift;

    my $hFileHandleTmp = $this->{ 'fHandle' };
    printf $hFileHandleTmp ( "%-54s   %s\n"
                             , " "
                             , $sMessagePar
                           );

    return 0;
}

#-----------------------------------------------------------------------
# Log hash
#-----------------------------------------------------------------------
sub logHashMessage{
    my $this = shift;

    if ( $this->{ bFileOpen } == $FALSE ){
        return -1;
    }

    #---- Get parameter
    #
    # 1. Message to log (of type hash reference)
    # 2. Indent string (optional, of type string)
    #
    my $hrefMsgPar = shift;
    my $sIndentPar = "";
    if ( @_ > 0 ){ $sIndentPar = shift; }

    my $hFileHandleTmp = $this->{ fHandle };

    #---- Process and print hash structure
    my @arrKeysTmp = sort keys( %{ $hrefMsgPar } );
    foreach my $key ( @arrKeysTmp ){
        if ( ref( $hrefMsgPar->{ $key } ) eq "HASH" ){
            printf $hFileHandleTmp ( "%-54s   %s%s:\n", " ", $sIndentPar, $key );
            $this->logHashMessage( $hrefMsgPar->{ $key }, $sIndentPar . " " );
        }
        elsif ( ref( $hrefMsgPar->{ $key } ) eq "ARRAY" ){
            my @arrElemTmp = @{ $hrefMsgPar->{ $key } };
            for ( my $i = 0; $i < @arrElemTmp; $i++ ){
                if ( ref( $arrElemTmp[ $i ] ) eq "HASH" ){
                    printf $hFileHandleTmp ( "%-54s   %s%d:\n", " ", $sIndentPar, $i );
                    $this->logHashMessage( $arrElemTmp[ $i ], $sIndentPar . " " );
                }
                elsif ( ref( $arrElemTmp[ $i ] ) eq "ARRAY" ){
                    my @arrElemL2Tmp = @{ $arrElemTmp[ $i ] };
                    printf $hFileHandleTmp ( "%-54s   %s%d:[%s]\n", " ", $sIndentPar, $i, "@arrElemL2Tmp" );
                }
                else {
                    printf $hFileHandleTmp ( "%-54s   %s%d:>%s<\n", " ", $sIndentPar, $i, $arrElemTmp[ $i ] );
                }
            }
        }
        else{
            printf $hFileHandleTmp ( "%-54s   %s%s:>%s<\n", " ", $sIndentPar, $key, $hrefMsgPar->{ $key } );
        }
    }

    return 0;
}

#-----------------------------------------------------------------------
# Don't remove the following lines !!!
#-----------------------------------------------------------------------
1;
__END__

