#-------------------------------------------------------------------------------
#  mergeShipmentBlocks.pm:
#  Scripting Facility to draw base line for barcode texts
#
#  Call:
#
#  On Windows:    afp2web.exe -q -c -doc_cold -sp:mergeShipmentBlocks.pm -sa:<shipment blocks json filename> samples\afpsample.afp
#
#  On Unix:    ./afp2web   -q -c -doc_cold -sp:mergeShipmentBlocks.pm -sa:<shipment blocks json filename> samples/afpsample.afp
#
#  where
#  <shipment blocks json filename> must have shipment details as given below
#  {
#    "blocks" = [
#     { "filename": <shipment block filename without path>, "h": <height>, "res": <resolution> },
#     { "filename": <shipment block filename without path>, "h": <height>, "res": <resolution> }
#    ],
#    "externalDocs" = [
#     <external doc page 1 filename without path>,
#     <external doc page 2 filename without path>
#    ]
#  }
#
#  Author  : Fa. Maas
#
#  $V100   2018-09-18    Initial Release
#
#-------------------------------------------------------------------------------

#-----------------------------------------------------------------------
# BEGIN block of module
#
# Extends PERL module search path array (@INC) with new element having
# this script modules path in order to have better module portability
#-----------------------------------------------------------------------
BEGIN {
    #---- Fetch script filename
    my $sScriptFilenameTmp = $0;

    #---- Extract script file path from script filename
    my $sScriptFilePathTmp = "";
    if ( $sScriptFilenameTmp =~ /(.*)\/.*\.pm/ ){
        $sScriptFilePathTmp = $1;
    }

    #printf STDERR ( "Script filename: " . $0 . " Script filepath: " . $sScriptFilePathTmp . "\n" );
    if ( $sScriptFilePathTmp eq "" ){
        $sScriptFilePathTmp = ".";
    }
    else {
        my $sScriptFileParentPathTmp = "";
        if ( $sScriptFilePathTmp =~ /(.*)\/sfsamples/ ){
            $sScriptFileParentPathTmp = $1;
        }

        #---- Add script file parent path to module search path
        if ( $sScriptFileParentPathTmp ne "" ){
            unshift( @INC, $sScriptFileParentPathTmp );
        }
    }

    #---- Add script file path to module search path
    unshift( @INC, $sScriptFilePathTmp );

    #---- Add local perl lib path relative to script file parent path to module search path
    unshift( @INC, $sScriptFilePathTmp . "/../../perl/site/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/../../perl/lib" );
    unshift( @INC, $sScriptFilePathTmp . "/a2w" );
}

use a2w::Config;
use a2w::ContentBlock;
use a2w::Document;
use a2w::Font;
use a2w::Index;
use a2w::Kernel;
use a2w::Line;
use a2w::MediumMap;
use a2w::NOP;
use a2w::Overlay;
use a2w::Page;
use a2w::PSEG;
use a2w::Text;

use a2w::ConfigConstants;
use a2w::ContentBlockConstants;
use a2w::DocumentConstants;
use a2w::PageConstants;
use a2w::FontConstants;

use a2w::core::log::Logger; # Log engine

#-----------------------------------------------------------------------
# Initialize once per process
#-----------------------------------------------------------------------
sub initialize(){

    #---- Get Parameter of initialize( Par: a2w::Config, a2w::Kernel )
    ( $a2wConfigPar, $a2wKernelPar ) = @_;

    #---- Define boolean values
    $TRUE  = 1;    # TRUE  boolean value
    $FALSE = 0;    # FALSE boolean value

    #---- Set/Reset Logging
    $bLog = $FALSE;
    if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sf") >= 0 ){
        $bLog = $TRUE;
    }

    my $sScriptProcTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTPROCEDURE );
    my $sScriptArgsTmp = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::SCRIPTARGUMENT );
    $sIndexFilePath    = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::INDEXPATH );
    $sOutputFilePath   = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::OUTPUTFILEPATH );
    $sLogPath          = $a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGPATH );
    $sSpoolFilename    = $a2wKernelPar->getSpoolFilename();

    #---- Instantiate Logger
    $theLogger = undef;
    $bDebugLog = $FALSE;
    if ( $bLog == $TRUE ){
        #---- Set log on log engine
        $theLogger = a2w::core::log::Logger->getSingleton();
        $theLogger->setStartTime( $sStartTime );

        #---- Register modules that has to be logged
        $theLogger->registerClasses( "a2w::Main" );

        #---- Open log file
        my $sLogFilenameTmp = $sSpoolFilename;
        $sLogFilenameTmp =~ s/^.*[\\\/]//;
        $sLogFilenameTmp .= ".cn.msb.log";
        $theLogger->setFilename( $sLogFilenameTmp );
        if ( $theLogger->open( $sLogPath ) == $FALSE ){
            return ( -1, "[ERROR] Unable to open log file (" . $sLogPath . $sLogFilenameTmp . ")" );
        }
        $bLog = $theLogger->isRegistered( "a2w::Main" );

        if (index( lc($a2wConfigPar->getAttribute( $a2w::ConfigConstants::LOGGINGLEVEL )), "sfdbg") >= 0 ){
            $theLogger->setLevel( $a2w::core::log::Logger::LEVEL_DEBUG );
            $bDebugLog = $TRUE;
        }

        $theLogger->logFunctionName( "a2w::Main", "initialize()" );
        $theLogger->logMessage( "Running $sScriptProcTmp..." );
        $theLogger->logMessage( "initialize(): Processing " . $sSpoolFilename );
        $theLogger->logMessage( "initialize(): Args: $sScriptArgsTmp, OutputFilePath: $sOutputFilePath, OutputLanguage: $sOutputLanguage" );
    }

    #---- Load extractdoc arguments JSON
    $sSAJSONFile = $sScriptArgsTmp;
    my $iRcTmp = open( JSON, "<$sSAJSONFile" );
    if ( $iRcTmp == 0 ){
        return ( -1, "Unable to open arguments JSON file $sSAJSONFile. rc=$iRcTmp reason=" . $! );
    }
    my $sShipmentBlocksJSONTmp = do { local $/; <JSON> };
    close( JSON );

    #---- Decode extractdoc arguments
    $hrefShipmentBlocks = undef;
    eval{
        require JSON::Tiny;
        $hrefShipmentBlocks = JSON::Tiny::from_json( $sShipmentBlocksJSONTmp );
    };
    if ( $@ ){
        return ( -2, "Unable to parse arguments JSON file $sSAJSONFile. rc=$iRcTmp reason=" . $@ );
    }
    elsif ( $hrefShipmentBlocks == undef ){
        return ( -2, "Unable to parse arguments JSON file $sSAJSONFile. rc=$iRcTmp" );
    }
    if ( $bLog == $TRUE ){
        $theLogger->logMessage( "Shipment Blocks:" );
        $theLogger->logHashMessage( $hrefShipmentBlocks );
    }

    #---- Page process flags
    $APPEND = 0;    # append page to Current Document
    $SKIP   = 1;    # skip page
    $NEWDOC = 2;    # new document

    #---- Page constants
    $iPageResolution = 300;  # 300 dpi
    $iPageWidth      = int(( 210 / 25.4 ) * $iPageResolution); # A4 width (210 mm) in page resolution
    $iPageHeight     = int(( 297 / 25.4 ) * $iPageResolution); # A4 height (297 mm) in page resolution

    $iTopMargin  = 120;  # ~10 mm in 300 dpi
    $iLeftMargin = 0;    # 0 mm in 300 dpi

    #---- Validate extractdoc arguments
    my ( $iRetTmp, $sMsgTmp ) = _processAndValidateArguments( $hrefShipmentBlocks );
    if ( $iRetTmp < 0 ){ return ( -3, $sMsgTmp ); }

    return 0;
}

#-----------------------------------------------------------------------
# InitializeDoc for each document
#-----------------------------------------------------------------------
sub initializeDoc(){

    #---- Get Parameter of initializeDoc( Par: a2w::Document )
    ($a2wDocumentPar) = @_;

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "initializeDoc()" );
        $theLogger->logMessage( "Name=" . $a2wDocumentPar->getName() . " Id=" . $iDocumentId );
    }

    return 0;
}

#-----------------------------------------------------------------------
# InitializePage for each page
#-----------------------------------------------------------------------
sub initializePage(){

    #---- Get Parameter of initializePage( Par: a2w::Page )
    ($a2wPagePar) = @_;

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "initializePage()" ); }

    return 0;
}

#-----------------------------------------------------------------------
# afp2web
#
# Page processing method
#-----------------------------------------------------------------------
sub afp2web(){

    if ( $bLog == $TRUE ){
        $theLogger->logFunctionName( "a2w::Main", "afp2web()" );
        $theLogger->logMessage( "PageId=" . $a2wPagePar->getParseId() );
    }

    #---- Set default return value
    my $iRetTmp = $SKIP; # default: Skip page

    #---- Get shipment blocks
    my $arefBlocksTmp = $hrefShipmentBlocks->{ 'blocks' };
    my @arrBlocksTmp = @{ $arefBlocksTmp };

    #---- Iterate through shipment blocks and place them on pages
    my $h = 0;
    my $iPageIdTmp = 0;
    my $a2wPageTmp = undef;
    my $hrefBlockTmp = undef;
    my $iCurYTmp = $iTopMargin;
    my $gapTmp = 10; # 10 pixels gap between 2 images

    for ( my $i = 0; $i < @arrBlocksTmp; $i++ ){
        $hrefBlockTmp = $arrBlocksTmp[ $i ];

        #---- Evaluate JSON->blocks[ i ]->h value in page resolution ($iPageResolution)
        $h = ( $hrefBlockTmp->{ 'res' } == $iPageResolution ) ? $hrefBlockTmp->{ 'h' } : int(( $hrefBlockTmp->{ 'h' } / $hrefBlockTmp->{ 'res' } ) * $iPageResolution );

        #---- Check whether block can be presented in current page, if not create new page and present it
        if ( ( $h + $iCurYTmp ) > $iPageHeight ){ $a2wPageTmp = undef; } # force to create new page

        #---- Create new page
        if ( $a2wPageTmp == undef ){
            $a2wPageTmp = new a2w::Page();
            if ( $a2wPageTmp == undef ){ return ( -21, "Unable to create page to write shipment block " . ( $i + 1 ) ); }

            $a2wPageTmp->setResolution( $iPageResolution );
            $a2wPageTmp->setWidth( $iPageWidth );
            $a2wPageTmp->setHeight( $iPageHeight );

            $a2wDocumentPar->addPage( $a2wPageTmp, ++$iPageIdTmp ); # Add page to document
            $iCurYTmp = $iTopMargin;
        }

        #---- Add block image to page
        $a2wPageTmp->addImage( # Filename
                                 $hrefBlockTmp->{ 'filename' }
                               # X position
                               , $iLeftMargin
                               # Y position
                               , $iCurYTmp
                               # Width
                               , $iPageWidth
                               # Height
                               , $h
                               # Rotation
                               , 0
                             );
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "img>" . $hrefBlockTmp->{ 'filename' } . "<@(" . $iLeftMargin . "," . $iCurYTmp . "), W=" . $iPageWidth . ", H=" . $h . ", R=0" ); }
        $iCurYTmp += $h + $gapTmp;
    }

    #---- Get external documents
    my $arefExtDocsTmp = $hrefShipmentBlocks->{ 'externalDocs' };
    my @arrExtDocsTmp = @{ $arefExtDocsTmp };

    #---- Iterate through external documents and place them on pages
    $a2wPageTmp = undef;
    my $sExtDocTmp = "";
    for ( my $i = 0; $i < @arrExtDocsTmp; $i++ ){
        $sExtDocTmp = $arrExtDocsTmp[ $i ];

        #---- Create new page
        $a2wPageTmp = new a2w::Page();
        if ( $a2wPageTmp == undef ){ return ( -21, "Unable to create page to write external document " . $hrefShipmentBlocks->{ '_externalDocs' }[ $i ] ); }

        $a2wPageTmp->setResolution( $iPageResolution );
        $a2wPageTmp->setWidth( $iPageWidth );
        $a2wPageTmp->setHeight( $iPageHeight );

        $a2wDocumentPar->addPage( $a2wPageTmp, ++$iPageIdTmp ); # Add page to document

        #---- Add block image to page
        $a2wPageTmp->addImage( # Filename
                                 $sExtDocTmp
                               # X position
                               , 0
                               # Y position
                               , 0
                               # Width
                               , $iPageWidth
                               # Height
                               , $iPageHeight
                               # Rotation
                               , 0
                             );
        if ( $bLog == $TRUE ){ $theLogger->logMessage( "img>" . $hrefShipmentBlocks->{ 'externalDocs' }[ $i ] . "<@(0,0), W=" . $iPageWidth . ", H=" . $iPageHeight . ", R=0" ); }
    }

    return $SKIP; # Skip input page and use only the pages added above
}

#-----------------------------------------------------------------------
# FinalizePage for each page
#-----------------------------------------------------------------------
sub finalizePage(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizePage()" ); }

    return 0;
}

#-----------------------------------------------------------------------
# FinalizeDoc for each document
#-----------------------------------------------------------------------
sub finalizeDoc(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalizeDoc()" ); }

    return 0;
}

#-----------------------------------------------------------------------
# Finalize once per process
#-----------------------------------------------------------------------
sub finalize(){

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "a2w::Main", "finalize()" ); }

    return 0;
}

#-----------------------------------------------------------------------
# Process and Validate arguments
#
# process and validate shipment blocks json
#
# Returns >=0 on valid arguments else <0, error message
#
#-----------------------------------------------------------------------
sub _processAndValidateArguments{

    if ( $bLog == $TRUE ){ $theLogger->logFunctionName( "main", "_processAndValidateArguments" ); }

    #---- Get parameter
    #
    # 1. Document arguments (of type hash reference)
    #
    my $hrefShipmentBlocksPar = shift;

    #  <shipment blocks json filename> must have shipment details as given below
    #  {
    #    "blocks" = [
    #     { "filename": <shipment block filename>, "h": <height>, "res": <resolution> },
    #     { "filename": <shipment block filename>, "h": <height>, "res": <resolution> }
    #    ],
    #    "externalDocs" = [
    #     <external doc page 1 filename>,
    #     <external doc page 2 filename>
    #    ]
    #  }

    #---- Assert for mandatory fields
    # Rule 1: Ensure "blocks" array is defined
    if ( !defined( $hrefShipmentBlocksPar->{ 'blocks' } ) ){ return ( -11, 'Missing "blocks" on shipment blocks json' ); }

    # Rule 2: Ensure "blocks" array is not empty
    my $arefBlocksTmp = $hrefShipmentBlocksPar->{ 'blocks' };
    my @arrBlocksTmp = @{ $arefBlocksTmp };
    if ( @arrBlocksTmp <= 0 ){
        return ( -12, 'Empty "blocks" list on shipment blocks json' );

    }

    # Rule 3: Ensure "block" entries have filename, height and res attributes
    my $sMsgTmp = "";
    my $iValidTmp = 0;
    my $hrefBlockTmp = undef;
    for ( my $i = 0; $i < @arrBlocksTmp; $i++ ){
        $hrefBlockTmp = $arrBlocksTmp[ $i ];

        #---- Assert filename attribute
        if ( !defined( $hrefBlockTmp->{ 'filename' } ) ){
            $iValidTmp = -13;
            $sMsgTmp = '"filename" is missing on block ' . ( $i + 1 );
            last;
        }
        #---- Assert h attribute
        if ( !defined( $hrefBlockTmp->{ 'h' } ) ){
            $iValidTmp = -13;
            $sMsgTmp = '"h" is missing on block ' . ( $i + 1 );
            last;
        }
        #---- Assert res attribute
        if ( !defined( $hrefBlockTmp->{ 'res' } ) ){
            $iValidTmp = -13;
            $sMsgTmp = '"res" is missing on block ' . ( $i + 1 );
            last;
        }
    }
    if ( $iValidTmp < 0 ){ return ( $iValidTmp, $sMsgTmp ); }

    #---- Check and Get external documents array
    my $arefExtDocsTmp = $hrefShipmentBlocksPar->{ 'externalDocs' };
    if ( $arefExtDocsTmp == undef ){ return 0; } # no external docs, return
    my @arrExtDocsTmp = @{ $arefExtDocsTmp };
    if ( @arrExtDocsTmp <= 0 ){ return 0; } # no external docs, return

    # Rule 4: Ensure "externalDocs" given filenames exists (if given)
    $sMsgTmp = "";
    $iValidTmp = 0;
    my $sExtDocTmp = "";
    for ( my $i = 0; $i < @arrExtDocsTmp; $i++ ){
        #---- Assert external document entry
        $sExtDocTmp = $arrExtDocsTmp[ $i ];
        if ( $sExtDocTmp eq "" ){
            $iValidTmp = -14;
            $sMsgTmp = 'Empty external document at index ' . $i . '.';
            last;
        }
    }
    if ( $iValidTmp < 0 ){ return ( $iValidTmp, $sMsgTmp ); }

    return 0;
}

__END__
