/*
 *   <copyright 
 *   notice="lm-source-program" 
 *   pids="5724-H72," 
 *   years="2005,2012" 
 *   crc="831681648" > 
 *   Licensed Materials - Property of IBM  
 *    
 *   5724-H72, 
 *    
 *   (C) Copyright IBM Corp. 2005, 2012 All Rights Reserved.  
 *    
 *   US Government Users Restricted Rights - Use, duplication or  
 *   disclosure restricted by GSA ADP Schedule Contract with  
 *   IBM Corp.  
 *   </copyright> 
 */

import java.io.*;
import java.util.*;
import java.text.*;

import com.ibm.mq.*;
import com.ibm.mq.soap.transport.jms.*;
import com.ibm.mq.soap.util.*;

/**
 * This class prepares a Java or Microsoft .NET service for access by SOAP/WMQ
 * It requires javac and runmqsc to be available in the current path
 * 
 * The following binaries are required in the PATH for deployment to succeed:
 * 
 * javac
 * runmqsc
 * amqswsdl.exe (.NET services only)
 * chmod (Linux)
 *
 */
public class DeployWMQService
{
	/** SCSS id - expanded when file is extracted from CMVC */
	private static final String SCCSID = "@(#) MQMBID sn=p800-003-150615.2 su=_O7DtFhOPEeWRp7-5NNeHyQ pn=samples/soap/DeployWMQService.java";

	/** The name of the class as supplied with the -f option. This is stripped of its
	 * .java or .asmx extension and converted to package name format (i.e. directory
	 * seperators are replaced with "."s
	 */
	private String className;
  
	/** This is the className in package name format (i.e. with directory separators rather
	 * than "." separators 
	 */
	private String classSourceDir;
  
	/** Package name used for Axis services only. It is derived from the parent directory
	 * of the supplied path name. For .NET we assume the use of dotNetService as the 
	 * package name
	 */
	private String packageName;
  
	/** This is the pathname used to identify the proxy for Axis services. It is based 
	 * on classSourceDir.
	 */
	private String proxyDir;
  
	/** Name of the WSDL file to be generated */
	private String wmq_wsdl;
  
	/** service port name to use when generating Axis WSDL */
	private String servicePort;
  
	/** SOAP/WMQ URI as a String */
	private String uriOption = null;
  
	/** SOAP/WMQ URI as a WMQAddress object */
	private WMQAddress wmqAddress = null;
  
	/** Service queue manager */
	private String queueManager="";
  
	/** Request queue */
	private String queue=null;
  
	/** Which deployment operation we are to perform */
	private String todo="";
  
	/** Numer of server threads to burn into the listener start up configuration */
	private int serverThreads=10;
  
	/** Backout threshold setting for the request queue */
	private int boThresh=3;
  
	/** Name of our script used to start another window */
	private static String START = "amqwstartwin.cmd";
  
	/** Whether or not verbose messages are required from external commands and
	 * runmqsc commands
	 */
	private boolean verbose=false;
  
	/** Whether existing queue definitions are to be replaced or retained and amended
	 * as necessary
	 */
	private boolean replaceQs=false;
  
	/** runmqsc verb to create or replace queue definitions - set to DEFINE or ALTER */
	private String qCmdStr="DEFINE";
	private String trqCmdStr="DEFINE";
  
	/** Set if running on the Windows platform */
	private boolean windows = false;

	/** Set to true if we are deploying for Axis and false if for .NET */
	private boolean generatingAxis=true;  
  
	/** Directory to used for generated listener configuration scripts */
	private final String generatedServerDir = "generated" + File.separatorChar + "server";
  
	/** Directory to be used for client proxy code */
	private final String generatedClientDir = "generated" + File.separatorChar + "client";

	/** Whether or not triggering is required to start SOAP/WMQ listeners */ 
	private boolean triggeringEnabled = false;
  
	/** The name of the trigger monitor initiation queue */
	private String triggerQueueName = null;
  
	/** Name of the trigger monitor program to be used */
	private String triggerProgName = "runmqtrm";
 
	/** This is the process name used on an initiation queue to start a listener */
	private String processName="";
  
	/** Definition of the command to be executed when a listener is to be started 
	 * via triggering 
	 */
	private String applicIDStr = null;

	/** Set to true if we are configuring the listener to start as a service */
	private boolean listenerAsWMQService = false;
 
	/** Set to "passContext" or "ownContext" depending on whether the listener will attempt 
	 * to pass context or not */
	private String idContext = null;
  
	/** Optional transaction options for the listener */
	private String transactionOptions = null;

	/**
	 * Provide a help text on the use of deployWMQService
	 * This is given when the user invokes deployWMQService with an "?" argument
	 */
	private void usage()
	{
		//output usage message here
	}

	/**
	 * This method is used internally to confirm whether deployWMQService is running
	 * on Windows or not. This is called when it is attempted to deploy a .NET service.
	 *  
	 * @return true if running on Windows and false otherwise
	 */
	private boolean asmxOnWindows()
	{
		if (!windows)
		{
			// output error message here
			return false;
		}
		return true;
	}
  

	/**
	 * This method is used internally to confirm whether deployWMQService is running on 
	 * a Linux system or not. This is called on non Windows system to differentiate
	 * between Linux and Unix systems.
	 * 
	 * @return if running on Linux and false otherwise
	 */
	private boolean onLinux()
	{
		if (System.getProperty("os.name").toLowerCase().startsWith("linux"))
			return true;
		return false;
	}

	/**
	 * Method to return a date and time that is returned as a String in the format:
	 * dd-mmm-yyyy hh:mm:ss
	 * This time stamp is inserted into comments into the various cmd files that are 
	 * generated by deployWMQService.
	 * 
	 * @return a String date and time
	 */
	private String getDateTime()
	{
		Date d = new Date();
		return DateFormat.getDateTimeInstance().format(d);
	}


	/**
	 * Method to create a directory. A check is first made to see if the path already
	 * exists. 
	 * 
	 * @param aPath is a String representation of the path to be created
	 * @return true if successful or the path already exists and false otherwise
	 */
	private static boolean mkdir(String aPath)
	{
		File path = new File(aPath);
		if (path.exists()) return true;
		return path.mkdirs();
	}

	/**
	 * Method to delete a directory and its contents. Calls the recursive method 
	 * deletePath.
	 * 
	 * @param aPath is a String representation of the path to be delete
	 * 
	 * @return true if the directory was deleted and false otherwise
	 */
	private static boolean erase(String aPath)
	{
		File path = new File(aPath);
		return deletePath(path);
	}  
   
	/**
	 * Method to delete all files and sub-directories in a directory. If a deletion
	 * files, the method stops attempting to delete and returns false.
	 * 
	 * @param path is a File object representing the path to be deleted
	 * @return true if the operation was successful and false otherwise
	 */
	private static boolean deletePath(File path) 
	{
       
		if (path.isDirectory()) 
		{
			String[] children = path.list();
			for (int i=0; i<children.length; i++) 
			{
				boolean success = deletePath(new File(path, children[i]));
				if (!success) 
				{
					return false;
				}
			}
		}
    
		/** The directory is now empty so delete it */
		return path.delete();
	}

	/**
	 * Method to check on the existence of a file or directory.
	 * 
	 * @param aFileName is a String representation of the pathname
	 * @return true if the file or directory exists and false otherwise
	 */
	private boolean fileExists(String aFileName)
	{
		File f = new File(aFileName);
		return f.exists();
	}
  
	/**
	 * Method that returns the absolute path of the parent directory for a specified
	 * pathname. The path returned is appended with a directory separator.
	 * 
	 * @param aPathName is the pathname
	 * @return a String representation of the absolute 
	 */
	private String getAbsParent(String aPathName)
	{
		File f1 = new File(aPathName);
		File f2 = new File(f1.getParent());
		return f2.getAbsolutePath() + File.separator;
	}

	/**
	 * Method to return a String representation of the current directory.
	 * 
	 * @return the current directory as a String
	 * 
	 */
	private String getCurrentDir()
	{
		File dir = new File(".");
		String curDir="";  
  
		try 
		{
			curDir = dir.getCanonicalPath();
		}
		catch(Exception e) 
		{
			e.printStackTrace();
		}
		return curDir; 
	}

	/**
	 * This method validates all of the arguments that were supplied to deployWMQService.
	 * If the arguments included the "?" option, the method displays a usage statement 
	 * for deployWMQService and exists successfully. Otherwise, it checks for the presence
	 * of each supported argument and validates all that are present. The -f option is 
	 * mandatory and all the others are optional. Optional arguments must however be 
	 * formatted correctly to validate successfully. Note that the method exits with an 
	 * error as soon as it finds illegal syntax on an argument and does not process all
	 * arguments before reporting errors. 
	 * 
	 * @param args are the arguments supplied to deployWMQServices, represented as an array 
	 * of Strings 
	 * @return true if the arguments were validated successfully and false otherwise
	 */
	private boolean processArgs(String[] args)
	{
		String s;
		try
		{
			ParseOptions opts = new ParseOptions(args);

			/** If the help option was given ("?") give the help
                         * message and exit with success */
			if ((opts.isFlagSet('?')))
			{
				usage();
                                System.exit(0);
			}

			/** -f: className this option is mandatory */
			if (null != (s = opts.isValueSet('f')))
				className = s;
			else
			{
				usage();
				return(false);
			}

			/** -u: WMQ URL (optional) */
			wmqAddress = new WMQAddress();
			if (null != (s = opts.isValueSet('u')))
			{
                                uriOption = s;
                                try
                                {
				wmqAddress.setWmqURI(s);
                                }
                                catch (WMQSOAPException wmqe)
                                {
                                  if (wmqe.getMessage().indexOf("RequestQueueName") == -1)
                                  {
				    usage();
                                    throw wmqe;
                                  }
                                }
				queueManager = wmqAddress.getServiceQmgr();
				queue = wmqAddress.getRequestQueueName();
			}
 
			/** -c: Operation to perform (optional) */
			if (null != (s = opts.isValueSet('c')))
                                todo = s;
   
			/** -n: number of threads the listeners are to start with (optional) */
			if (null != (s = opts.isValueSet('n')))
			{
				serverThreads = Integer.parseInt(s);
				if (serverThreads < 0)
				{
					return(false);
				}
			}

			/** -b: backout threshold (optional) */
			if (null != (s = opts.isValueSet('b')))
                                boThresh = Integer.parseInt(s);
    
			/** -tmq: trigger monitor queue name (optional) */
			if (null != (s = opts.isValueSet("tmq"))) 
			{
				triggeringEnabled = true;
				triggerQueueName = s;
			}

			/** -tmp: name of trigger monitor program (optional) */
			if (null != (s = opts.isValueSet("tmp")))
			{
				if (!triggeringEnabled)
				{
					return(false);
				}
				triggerProgName = s;
			}

			/** -s: If specified configures the listener to run as a service (optional) */
			if (opts.isFlagSet('s'))
			{
				if (triggeringEnabled)
				{
					return(false);
				}

				listenerAsWMQService = true;
			}

			/** -i: identity context (optional) */
			if (null != (s = opts.isValueSet('i')))
			{
				if (!s.equalsIgnoreCase("passcontext"))
				{
					return(false);                      
				}
				idContext = s;
			}

			/** -x: transactionality (optional) */
			if (null != (s = opts.isValueSet('x')))
			{
				if (!s.equalsIgnoreCase("none") &&
					!s.equalsIgnoreCase("onePhase") &&
					!s.equalsIgnoreCase("twoPhase"))
				{
					return(false);             
				}
				transactionOptions = s;
			}


			/** -r: request or trigger monitor queue definitions should be replaced if specified.
			 * Otherwise the existing definitions are retained. (optional) */
			if ((opts.isFlagSet('r')))
                                replaceQs=true;

			/** Set verbose output from external commands (optional) */
			if ((opts.isFlagSet('v')))
                                verbose=true;
		}
		catch ( Exception e )
		{
			System.exit( 2 );
		}

		return true;
	}  
  
	/**
	 * This method invokes a BAT/CMD file and ensures the environment variable errorlevel 
	 * set on an exit /b is set into the actual process exit value. This method is Windows
	 * specific.
	 * 
	 * @param aCommand is a String specification of the command to be exeucted
	 * @return true if the command was successfully executed and false otherwise
	 */
	private boolean doBatCommand(String aCommand)
	{
		String fixedCommand="cmd /V:ON /C \"" + aCommand + " && exit /b !errorlevel!\"";
                // Execute command and return
		//return (new RunCommand(fixedCommand)).Execute();
                return true;
	}

	/**
	 * Method to execute an external command
	 * 
	 * @param aCommand is a String specification of the command to be exeucted
	 * @return true if the command was successfully executed and false otherwise
	 */
	private boolean doCommand(String aCommand)
	{
		// Execute command and return
		//return (new RunCommand(aCommand)).Execute();
                return true;
	}
  

	/**
	 * This method executes a command through runmqsc.
	 * 
	 * @param anMQCommand is a String specification of the command to be executed in runmqsc.
	 * 
	 * @return true if the command was successfully executed and false otherwise
	 */
	private boolean doMQCommand(String anMQCommand)
	{
		// Execute command and return
		//return (new RunMQCommand(queueManager)).Execute(anMQCommand);
                return true;
	}

	/**
	 * This method performs additional preparation and validation prior to deployment,
	 * including:
	 * 
	 * - If a request queue name was not specified as an argument, make sure the generated
	 * form is no more than the maxium allowed (48 characters) by truncating
	 * 
	 * - Validate the request queue against any existing lisener configuration
	 * 
	 * - If triggering, checl that the length of the generated APPLICID does not exceed
	 * the limit
	 * 
	 * - Create sub-directories underneath the "generated" directory for the output of
	 * the deployment process
	 *   
	 * @return true if successful and false otherwise
	 * @throws Exception
	 */
	private boolean setup() throws Exception
	{
		/** We don't need to do anything here we're just starting the trigger monitor */
		if (todo.equals("startWMQMonitor"))
		{
			return true;
		}
    
		/** Remove trailing ".java" or ".asmx" from classname supplied on command line */
		if (className.endsWith(".java"))
		{
			generatingAxis=true;
			className = className.substring(0, className.length() - 5);
			if (todo.equals(""))
				todo="allAxis";
		}
		else
		{
			if (className.endsWith(".asmx"))
			{
				generatingAxis=false;
				className = className.substring(0, className.length() - 5);
				if (todo.equals(""))
					todo="allAsmx";
			}
			else
			{
				return false;
			}
		}
       
		/** Convert classname to classname format */
		className = className.replace(File.separatorChar, '.'); 
                /** Allow both forward and backward slash in the classname */
		className = className.replace('/', '.'); 
  
		/** Convert classname to pathname format */
		classSourceDir = className.replace('.', File.separatorChar); 
  
		String classNameStr;
		if (generatingAxis)
			classNameStr = className + ".java";
		else
			classNameStr = className + ".asmx";

		if (wmqAddress.getTargetService() != null)
		{
			if (!wmqAddress.getTargetService().equalsIgnoreCase(classNameStr))
			{
				/** Error - classname and target service are different */
				return false;
			}
		}
		else
		{
			/** Add the target service to both Axis and .NET */
			wmqAddress.setTargetService(classNameStr);
		}

		/** Set the target package name */
		String parentDir = (new File(classSourceDir)).getParent();
		if (parentDir == null)
			packageName = "DefaultNamespace";
		else
			packageName = parentDir.replace(File.separatorChar, '.'); 

		proxyDir = classSourceDir;
		if (classSourceDir.equals(className))
			proxyDir="DefaultNamespace" + File.separator + className;
 
		wmq_wsdl = className + "_Wmq.wsdl";

		/** If no queue name was specified, set the default here to be <currentdir>
		 * but with periods speficying sub-directory names rather than machine specific separators
		 * and special characters (e.g. ':' or '$') removed
		 * The genAsmxWMQBits or getAxisWMQBits methods prefix SOAPJ. or SOAPN. respectively
		 */
		if ((queue == null) || (queue.equals("")))
		{
			boolean colonPresent = false;

			queue = className;
			queue = queue.replace(File.separatorChar, '.'); 
			queue = queue.replace(' ', '_');
			if (windows)
			{
				if (queue.indexOf(':') != -1)
				{
					queue = queue.replace(':', '.'); 
					colonPresent = true;
				}
			}

			if (generatingAxis)
				queue = "SOAPJ." + queue;
			else
				queue = "SOAPN." + queue;

			/** Restrict the queue name to 48 characters. */
			if (48 < queue.length())
			{
				/** The generated name was greater than the maximum allowed. Make the name fit
				 * the limit by taking the rightmost 40 or 42 characters so the total length
				 * of the queue name becomes 48 characters 
				 */
				if (colonPresent)
					queue = queue.substring(0,7) + queue.substring(queue.length() - 40);
				else
					queue = queue.substring(0,5) + queue.substring(queue.length() - 42);
			}

			wmqAddress.setRequestQueueName(queue);
		}
 
		/** Validate the specified/generated request queue */
		String listenScript;
		String scriptQ = null;
		boolean scriptExists = false;
		boolean qmgrExists=false;
		boolean qExists=false;
		MQQueueManager qm = null;
		FileReader file = null;
		BufferedReader buff = null;

		if (generatingAxis)
			listenScript = generatedServerDir + File.separator + "startWMQJListener" + scriptExtn();
		else
			listenScript = generatedServerDir + File.separator + "startWMQNListener" + scriptExtn();

		try
		{
			file = new FileReader(listenScript);
			buff = new BufferedReader(file);
			scriptExists = true;
		}
		catch (Exception e)
		{
		}

		if (scriptExists)
		{
			/** go through the script and get the queue name from the URL */
			try
			{
				boolean eof = false;
				int uriIndex;
				String line;

				while (!eof)
					if ((line = buff.readLine()) ==  null)
						eof = true;
					else
					{
						if (line.startsWith("#"))
							continue;
						if (line.startsWith("@"))
							continue;
						if (line.startsWith("rem"))
							continue;
						if (generatingAxis)
						{
							if ((line.indexOf("SimpleJavaListener") == -1))
								continue;
						}
						else
						{
							if ((line.indexOf("amqwSOAPNETListener") == -1))
								continue;
						}
						/** We have found the command line with the -u option, now get the
						 * queue name
						 */
						if ((uriIndex = line.indexOf("-u")) == -1)
						{
							/** Something wrong with the script file */
							return false;
						}

                                                //Find the queue name in the
                                                //URI
                                                int endPos = 0;
                                                int qPos = line.indexOf("destination");
                                                if (qPos != -1)
                                                  qPos = line.indexOf('=');
                                                if (qPos != -1)
                                                {
                                                  qPos++; 
                                                  int atPos = line.indexOf('@', qPos);

                                                  if(atPos == -1)
                                                  {
                                                    endPos = line.indexOf('&', qPos);
                                                    if (endPos == -1)
                                                    {
                                                      endPos = line.length() - 1;
                                                    }
                                                  }
                                                  else
                                                    endPos = atPos;
                                                  scriptQ = line.substring(qPos, endPos);
                                                }
					}
			}
			catch(Exception e)
			{
			}
		}

                // Exclude any 2085 logs
                MQException.logExclude(new Integer(2085));
		try
		{
			qm = new MQQueueManager(queueManager);
			qmgrExists = true;
		}
		catch (MQException e)
		{
		}

		if(qmgrExists)
		{
			try
			{
				MQQueue q = qm.accessQueue(queue, MQC.MQOO_INPUT_AS_Q_DEF, null, null, null);
				qExists = true;
			}
			catch (MQException e)
			{
			}
		}

		if (qExists)
		{
			if (!scriptExists)
			{
				/** Error - Queue already in use */
				return false;
			}

			if (!queue.equalsIgnoreCase(scriptQ))
			{
				/** Error - Queues are different */
				return false;
			}
		}
		else
		{
			/** Request queue not found */
			if (scriptExists)
			{
				if (!queue.equalsIgnoreCase(scriptQ))
				{
					/** Error - Queues are different */
					return false;
				}
				else
				{
					/** Warning - Possible incomplete deployment */
				}
			}
		}

		/** If triggering, check that the APPLICID will not exceed 256 chars. */
		if (triggeringEnabled)
		{
			if (generatingAxis)
			{
				if (windows)
				{
					applicIDStr = "start \"Java WMQSoapListener - " + queue + "\" /min " + getCurrentDir() + File.separator + generatedServerDir + File.separator + "startWMQJListener" + scriptExtn();
				}
				else
				{
					applicIDStr = "xterm -iconic -T \\\"Java WMQSoapListener_" + queue + "\\\" -e " + getCurrentDir() + File.separator + generatedServerDir + File.separator + "startWMQJListener" + scriptExtn() + " & #";
				}
			}
			else
			{
				applicIDStr = "start \"WMQAsmxListener - " + className + "\" /min " + getCurrentDir() + File.separator + generatedServerDir + File.separator + "startWMQNListener" + scriptExtn();
			}

			if (applicIDStr.length() > 256)
			{
				/** Error - ApplicationID is greater than 256 chars */
				return false;
			}
		}

		processName = queue;

		/** Setup the output uri */
                if (uriOption != null)
                {
		  uriOption = wmqAddress.getWmqURI();
                }
                else
                {
                  // No URI option and set up a dummy uri for the -u switch in
                  // the listener script
                  uriOption = "jms:/queue?initialContextFactory=com.ibm.mq.jms.Nojndi&connectionFactory=()&destination=" + queue + "&targetService=" + wmqAddress.getTargetService();
                }

		servicePort=className + "_Wmq";
  

		/** Create "generated" directories for output of deployment */
		if ( !mkdir("generated") || 
			!mkdir(generatedServerDir) ||
			!mkdir(generatedClientDir) ||
			!mkdir(generatedClientDir + File.separator + "remote") )
		{
			return false;
		}
  
		return true;
	}
  
	/**
	 * This method compiles Java service code prior to generating WSDL
	 * 
	 * @return true if the compilation succeeded and false otherwise
	 */
	private boolean compileJava()
	{
		if (!doCommand("javac -d " + generatedServerDir + " " + classSourceDir + ".java"))
		{
			return false;
		}
		return true;
	}

	/**
	 * This method generates WSDL from the service code by invoking the Axis
	 * Java2WSDL utility
	 * 
	 * @return true if successful and false otherwise
	 */
	private boolean genAxisWsdl()
	{
		boolean status;
  
		String targetWsdl = "generated" + File.separator + wmq_wsdl;

		String wsdlin="";
		if (fileExists(targetWsdl)) wsdlin = "--input " + targetWsdl;
  
		status = doCommand(
			"java org.apache.axis.wsdl.Java2WSDL " + wsdlin + 
			" --output " + targetWsdl + 
			" --namespace " + servicePort + 
			" --location " + uriOption + 
			" --bindingName " + className + "BindingSoap" + 
			" --servicePortName " + servicePort + " " + className);
  
		if (!status || !fileExists(targetWsdl))
		{
			return false;
		}

		return true;
	}
    
	/**
	 * This method performs the bulk of the Axis deployment, including:
	 * 
	 * - generating the server side wsdd file
	 * - patching the deploy and undeploy.wsdd files for the SOAP/WMQ jms: transport
	 * - deploying the service through Axis using the patched deploy .wsdd file
	 * 
	 * The patched deploy and undeploy.wsdd files are created in the temporary
	 * sub-directory temp.server which the method deletes when it has completed
	 * the deployment.
	 * 
	 * NB: java com.ibm.mq.soap.util.RunWSDL2Java is the same as org.apache.axis.wsdl.WSDL2Java,
	 * except that it allows for the jms: URL syntax.
	 * 
	 * @return true if successful and false otherwise
	 */
	private boolean axisDeploy()
	{
		/** Use the RunWSDL2Java -p option to override packagename - otherwise use of 
		 * --namespace earlier results in a packagename the wrong way round
		 */
		if (!doCommand(
			"java com.ibm.mq.soap.util.RunWSDL2Java --server-side " + " " +
			"--timeout -1 " +   // Stop timeout aborting WSDL2Java (esp on AIX)
			"-p " + packageName + " " +
			"--output generated" + File.separator + "temp.server" + " " + 
			"generated" + File.separator + wmq_wsdl))
		{
			return false;
		}

		/** Patch the wsdd -- Axis does not seem to understand that the wsdl was generated bottom up.
		 * This very primitive substitution points the wsdd directly at our original class as implementation,
		 * rather than at the newly generated implementation skeleton.
		 */

		/** Make sure target directory ready */
		String relPath = generatedServerDir + File.separator + classSourceDir;
		if (!mkdir(getAbsParent(relPath)))
		{
			return false;
		}

		/** Find the temporary source directory */
		String tempDir = getAbsParent("generated" + File.separator + "temp.server" + File.separator + proxyDir);
  
		/** Set up the command to create the patched deploy .wsdd file 
		 * Patching is performed with the SOAP/WMQ PatchWsdd utility 
		 * On windows, spaces in directory names causes problems.
                 */

                String patchCommand = null; 
                if (windows)
		  patchCommand = "java com.ibm.mq.soap.util.PatchWsdd " + 
			className + 
			" \"" + tempDir + "deploy.wsdd\" " + 
			generatedServerDir + File.separator + classSourceDir + "_deploy.wsdd";
                else
		  patchCommand = "java com.ibm.mq.soap.util.PatchWsdd " + 
			className + 
			" " + tempDir + "deploy.wsdd " + 
			generatedServerDir + File.separator + classSourceDir + "_deploy.wsdd";
  
		/** On Windows need to execute the patch command with cmd to avoid interactions with 
		 * command redirection */
		if (windows) patchCommand = "cmd /c " + patchCommand;
  
		/** Execute the patch command for deploy.wsdd */
		if (!doCommand(patchCommand))
		{
			return false;
		}
  
		/** Set up the command to create the patched undeploy .wsdd file
		 * On windows, spaces in directory names causes problems.*/
                if (windows)
		  patchCommand = "java com.ibm.mq.soap.util.PatchWsdd " + 
			className + 
			" \"" + tempDir + "undeploy.wsdd\" " + 
			generatedServerDir + File.separator + classSourceDir + "_undeploy.wsdd";
                else
		  patchCommand = "java com.ibm.mq.soap.util.PatchWsdd " + 
			className + 
			" " + tempDir + "undeploy.wsdd " + 
			" " + generatedServerDir + File.separator + classSourceDir + "_undeploy.wsdd";

		/** On Windows need to execute the patch command with cmd to avoid interactions with 
		 * command redirection */
		if (windows) patchCommand = "cmd /c " + patchCommand;

		/** Execute the patch command for undeploy.wsdd */
		if (!doCommand(patchCommand))
		{
			return false;
		}

		/** Remove the temporary temp.server directory */
		if (!erase("generated" + File.separator + "temp.server"))
		{
			return false;
		}
  
		/** Deploy using the generated and patched wsdd file */
		if (!doCommand("java org.apache.axis.utils.Admin server " + generatedServerDir + File.separator + classSourceDir + "_deploy.wsdd"))
		{
			return false;
		}

		return true;
	}

	/**
	 * This method returns the name of the extension for script files on the
	 * host operating system. This will be ".cmd" for Windows systems and ".sh" for
	 * Linux and Unix systems.
	 * 
	 * @return a String containing the appropriate extention na,e
	 */
	private String scriptExtn()
	{
		if (windows) return ".cmd";
		return ".sh";
	}

	/**
	 * Method to prepare a tailored Axis listener with scripts to start and stop it. If
	 * required a script is also generated that will configure the listener to start
	 * as a service. genAxisWMQBits also creates triggering definitions on the request
	 * queue if specified and creates the initiation queue if it doesn't exist.
	 * 
	 * @return true if successful and false otherwise
	 */
	private boolean genAxisWMQBits()
	{
		String listen = "java";

		/** Add any system properties that are specified
		 * if both are specified (system property and URL), take the
		 * URL - don't add the system property to the output
		 */
		String mqChlUrlProp;
		if ((mqChlUrlProp = System.getProperty("com.ibm.mq.soap.transport.jms.mqchlurl")) != null)
			listen = listen + " -Dcom.ibm.mq.soap.transport.jms.mqchlurl=" +  mqChlUrlProp;

		String sslKeyStoreProp = System.getProperty("javax.net.ssl.keyStore");
		if (wmqAddress.getSslKeyStore() == null)
		{
			if (sslKeyStoreProp != null)
				listen = listen + " -Djavax.net.ssl.keyStore=" +  sslKeyStoreProp;
		}
		else
		{
			if (sslKeyStoreProp != null)
			{
				if(!(sslKeyStoreProp.equals(wmqAddress.getSslKeyStore())))
				{
				}
			}
		}

		String sslKeyStorePasswordProp = System.getProperty("javax.net.ssl.keyStorePassword");
		if (wmqAddress.getSslKeyStorePassword() == null)
		{
			if (sslKeyStorePasswordProp != null)
				listen = listen + " -Djavax.net.ssl.keyStorePassword=" +  sslKeyStorePasswordProp;
		}
		else
		{
			if (sslKeyStorePasswordProp != null)
			{
				if(!(sslKeyStorePasswordProp.equals(wmqAddress.getSslKeyStorePassword())))
				{
				}
			}
		}

		String sslTrustStoreProp = System.getProperty("javax.net.ssl.trustStore");
		if (wmqAddress.getSslTrustStore() == null)
		{
			if (sslTrustStoreProp != null)
				listen = listen + " -Djavax.net.ssl.trustStore=" +  sslTrustStoreProp;
		}
		else
		{
			if (sslTrustStoreProp != null)
			{
				if(!(sslTrustStoreProp.equals(wmqAddress.getSslTrustStore())))
				{
				}
			}
		}

		String sslTrustStorePasswordProp = System.getProperty("javax.net.ssl.trustStorePassword");
		if (wmqAddress.getSslTrustStorePassword() == null)
		{
			if (sslTrustStorePasswordProp != null)
				listen = listen + " -Djavax.net.ssl.trustStorePassword=" +  sslTrustStorePasswordProp;
		}
		else
		{
			if (sslTrustStorePasswordProp != null)
			{
				if(!(sslTrustStorePasswordProp.equals(wmqAddress.getSslTrustStorePassword())))
				{
				}
			}
		}

		listen = listen + " com.ibm.mq.soap.transport.jms.SimpleJavaListener" + " -u \"" + uriOption + "\" -n " + serverThreads;

		/** Add the idContext option if specified. */
		if (idContext != null)
			listen = listen + " -i " + idContext;

		/** Add any transaction options if specified */
		if (transactionOptions != null)
			listen = listen + " -x " + transactionOptions;

		String listenbat = generatedServerDir + File.separator + "startWMQJListener" + scriptExtn();
  
		try
		{
			FileWriter fw = new FileWriter(listenbat);
 
			if (windows)
			{
				fw.write("rem - generated by DeployWMQService.java at " + getDateTime() + "\n");
				fw.write("call " + "amqwsetcp" + scriptExtn() + "\n");
				fw.write("set CLASSPATH=" + generatedServerDir + ";%CLASSPATH%\n");
				fw.write("cd /d " + getCurrentDir() + "\n");
				fw.write(listen + "\n");  
				fw.close();
			}
			else
			{
				/** On Linux explicitly start with bash in order to get the shell startup
				 *files (${HOME}/.bashrc) read
				 */
				if (onLinux())
					fw.write("#!/bin/bash\n");
				else
					fw.write("#!/bin/sh\n");

				fw.write("# - generated by DeployWMQService.java at " + getDateTime() + "\n");
				fw.write(". " + "amqwsetcp" + scriptExtn() + "\n");
				fw.write("CLASSPATH=" + generatedServerDir + ":${CLASSPATH}\n");
				fw.write("export CLASSPATH\n");
				fw.write("cd " + getCurrentDir() + "\n");
				fw.write("exec " + listen + "\n");  
				fw.close();
      
				/** Set the permissions on the generated file. */
				if (!doCommand("chmod ugo+x " + listenbat))
				{
					return false;
				}
			}
		}
		catch (Exception e)
		{
			e.printStackTrace();
			return false;
		}

		/** If required generate the script that defines the listener as a service */
		if (listenerAsWMQService)
		{
			if (!writeListenerSvcScript(listenbat))
			{
				return false;
			}
		}

		/** Generate the stop listener script */
		String stopbat = generatedServerDir + File.separator + "endWMQJListener" + scriptExtn();

		if (!writeListenerStopScript(stopbat))
		{
			return false;
		}

		/** Define the request queue to MQ, and set it up to trigger its listener
		 * qCmdStr should be "ALTER" or "DEFINE"
		 */
		if (!triggeringEnabled)
		{
			String qReplaceStr = "REPLACE";
			if (qCmdStr.equals("ALTER")) qReplaceStr = "";
    
			if (!doMQCommand(qCmdStr + " QL('" + queue + "') BOTHRESH(" + boThresh + ") " + qReplaceStr))
			{
				return false;
			}
		}
		else
		{
			/** trqCmdStr should be "DEFINE" or "ALTER" */
			String trqReplaceStr = "REPLACE";
			if (trqCmdStr.equals("ALTER"))
				trqReplaceStr = "";
			if (!doMQCommand(trqCmdStr + " QL('" + triggerQueueName + "') defsopt(excl) " + trqReplaceStr))
			{
				return false;
			}

			String defProcess;
			if (windows)
				defProcess = "DEFINE PROCESS('" + processName + "') APPLICID('" + applicIDStr + "') REPLACE";
			else
				defProcess = "DEFINE PROCESS('" + processName + "') APPLICID('" + applicIDStr + "') REPLACE";

			String qReplaceStr = "REPLACE";
			if (qCmdStr.equals("ALTER"))
				qReplaceStr = "";
			if (!doMQCommand(qCmdStr + " QL('" + queue + "') BOTHRESH(" + boThresh + ") TRIGTYPE(FIRST) TRIGGER PROCESS('" + processName + "') INITQ('" + triggerQueueName + "') TRIGMPRI(0) " + qReplaceStr) ||
				!doMQCommand(defProcess))
			{
				return false;
			}
		}
		return true;
	}

	/**
	 * This method generates the WSDL for a .NET service
	 * 
	 * @return true if successful and false otherwise
	 */
	private boolean genAsmxWsdl()
	{
		boolean status;
  
		if (!asmxOnWindows())
		{
			return false;
		}
  
                // Have to change '&' to '&amp;' in the uri before generating
                // the wsdl
                String escapedUri = "";
                int curPos = 0;
                int ampPos = 0;
                do
                {
                  ampPos = uriOption.indexOf('&', curPos);
                  if (ampPos == -1)
                  {
                    // Copy to the end of the uri into the escaped URI
                    if (!escapedUri.equals(""))
                    {
                      escapedUri = escapedUri + uriOption.substring(curPos, uriOption.length());
                    }
                   break;
                  }
                  escapedUri = escapedUri + uriOption.substring(curPos, ampPos + 1) + "amp;";
                  curPos = ampPos + 1;
                }

                while (curPos < uriOption.length());

                if (escapedUri.equals(""))
                  // No '&'found
                  escapedUri = uriOption;

		status = doCommand("amqswsdl " + escapedUri + " " + className + ".asmx" + " generated" + File.separator + wmq_wsdl); 
		if (!fileExists("generated" + File.separator + wmq_wsdl))
		{
			return false;
		}
     
		if (!status)
		{
			return false;
		}
  
		return true;
	}

	/**
	 * Method to prepare a tailored .NET listener with scripts to start and stop it. If
	 * required a script is also generated that will configure the listener to start
	 * as a service. genAxisWMQBits also creates triggering definitions on the request
	 * queue if specified and creates the initiation queue if it doesn't exist.
	 * 
	 * @return true if successful and false otherwise
	 */
	private boolean genAsmxWMQBits()
	{
		if (!asmxOnWindows())
		{
			return false;
		}

		String listen="amqwSOAPNETListener -u \"" + uriOption + "\" -w " + getCurrentDir() + " " +
			"-n " + serverThreads;

		/** Add the idContext option if specified */
		if (idContext != null)
			listen = listen + " -i " + idContext;

		/** Add any transaction options if specified */
		if (transactionOptions != null)
			listen = listen + " -x " + transactionOptions;

		String listenbat = generatedServerDir + File.separator + "startWMQNListener" + scriptExtn();
  
		try
		{
			FileWriter fw = new FileWriter(listenbat);
			fw.write("rem - generated by DeployWMQService.java at " + getDateTime() + "\n");

			fw.write("call " + "amqwsetcp" +  scriptExtn() + "\n");
			fw.write(listen + "\n");  
			fw.close();
		}
		catch (Exception e)
		{
			return false;
		}

		if (listenerAsWMQService)
		{
			if (!writeListenerSvcScript(listenbat))
			{
				return false;
			}
		}

		/** Generate the stop listener script */
		String stopbat = generatedServerDir + File.separator + "endWMQNListener" + scriptExtn();
		if (!writeListenerStopScript(stopbat))
		{
			return false;
		}

		if (!triggeringEnabled)
		{
			/** qCmdStr should be "ALTER" or "DEFINE" */
			String qReplaceStr = "REPLACE";
			if (qCmdStr.equals("ALTER"))
				qReplaceStr = "";
			if (!doMQCommand(qCmdStr + " QL('" + queue + "') BOTHRESH(" + boThresh + ") " + qReplaceStr))
			{
				return false;
			}
		}
		else
		{
			/** Define queue for MQ, and set it up to trigger its listener
			 * trqCmdStr should be "DEFINE" or "ALTER"
			 */
			String trqReplaceStr = "REPLACE";
			if (trqCmdStr.equals("ALTER"))
				trqReplaceStr = "";
			if (!doMQCommand(trqCmdStr + " QL('" + triggerQueueName + "') defsopt(excl) " + trqReplaceStr))
			{
				return false;
			}
  
			/** qCmdStr should be "DEFINE" or "ALTER" */
			String qReplaceStr = "REPLACE";
			if (qCmdStr.equals("ALTER"))
				qReplaceStr = "";
			if (!doMQCommand(qCmdStr + " QL('" + queue + "') BOTHRESH(" + boThresh + ") TRIGTYPE(FIRST) TRIGGER PROCESS('" + processName + "') INITQ('" + triggerQueueName + "') TRIGMPRI(0) " + qReplaceStr) ||
				!doMQCommand("DEFINE PROCESS('" + processName + "') APPLICID('" + applicIDStr + "') REPLACE"))
			{
				return false;
			}
		}

		return true;
	}
  
	/**
	 * This method write a script to configure an Axis or .NET SOAP/WMQ listener
	 * to start as a service
	 * 
	 * @param listenbat is a String containing the name of the script to be created
	 * @return true if successful and false otherwise
	 */
	private boolean writeListenerSvcScript(String listenbat)
	{
		String definebat;
		if (generatingAxis)
			definebat = generatedServerDir + File.separator + "defineWMQJListener" + scriptExtn();
		else
			definebat = generatedServerDir + File.separator + "defineWMQNListener" + scriptExtn();

		try
		{
			FileWriter fw = new FileWriter(definebat);

			if (windows)
			{
                                fw.write("rem - generated by DeployWMQService.java at " + getDateTime() + "\n");
				fw.write("echo DEFINE SERVICE('" + className + "') STARTCMD('" +
					getCurrentDir() + File.separator + listenbat + "') ");

				fw.write("CONTROL(QMGR) SERVTYPE(SERVER) REPLACE | runmqsc " + queueManager + "\n");
				fw.write("echo START SERVICE('" + className + "') | runmqsc " + queueManager + "\n");
			}
			else
			{
				if (onLinux())
					fw.write("#!/bin/bash\n");
				else
					fw.write("#!/bin/sh\n");

                                fw.write("# - generated by DeployWMQService.java at " + getDateTime() + "\n");
				fw.write("runmqsc " + queueManager + " << eof\n");
				fw.write("DEFINE SERVICE('" + className + "') STARTCMD('" + 
					getCurrentDir() + File.separator + listenbat + "') ");
				fw.write("CONTROL(QMGR) SERVTYPE(SERVER) REPLACE\n");
				fw.write("START SERVICE('" + className + "')\n");
				fw.write("eof\n");
			}
			fw.close();
			if (!windows)
				if (!doCommand("chmod ugo+x " + definebat))
				{
					return false;
				}
		}
		catch (Exception e)
		{
			e.printStackTrace();
			return false;
		}

		return true;
	}

	/**
	 * This method write a script to configure an Axis or .NET SOAP/WMQ listener
	 * to stop as a service
	 * 
	 * @param stopbat is a String containing the name of the script to be created
	 * @return
	 */
	private boolean writeListenerStopScript(String stopbat)
	{
		try
		{
			FileWriter fw = new FileWriter(stopbat);
 

			if (windows)
			{
                                fw.write("rem - generated by DeployWMQService.java at " + getDateTime() + "\n");
				fw.write("call " + "amqwsetcp" + scriptExtn() + "\n");
				fw.write("set CLASSPATH=" + generatedServerDir + ";%CLASSPATH%\n");
			}
			else
			{
				/** On Linux explicitly start with bash in order to get the shell startup
				 * files (${HOME}/.bashrc) read
				 **/
				if (onLinux())
					fw.write("#!/bin/bash\n");
				else
					fw.write("#!/bin/sh\n");

                                fw.write("# - generated by DeployWMQService.java at " + getDateTime() + "\n");
				fw.write(". " + "amqwsetcp" + scriptExtn() + "\n");
				fw.write("CLASSPATH=" + generatedServerDir + ":${CLASSPATH}\n");
				fw.write("export CLASSPATH\n");
				fw.write("cd " + getCurrentDir() + "\n");
			}

			if (queueManager.equals(""))
				fw.write("java com.ibm.mq.soap.util.EndWMQListener -q " + queue +"\n");
			else
				fw.write("java com.ibm.mq.soap.util.EndWMQListener -m " + queueManager + " -q " + queue +"\n");
			fw.close();
		}
		catch (Exception e)
		{
			e.printStackTrace();
			return false;
		}

		/** Set the permissions on the generated file */
		if (!windows)
			if (!doCommand("chmod ugo+x " + stopbat))
			{
				return false;
			}
		return true;
	}

	/**
	 * Method that generates the proxies for a .NET service
	 * 
	 * @return true if successful and false otherwise
	 */
	private boolean genProxiesToDotNet()
	{
		if (!asmxOnWindows())
		{
			return false;
		}

		/** Generate all proxies (C#, VB and Java) and server wsdd file, and compile the Java */
		if (!doBatCommand("amqwcallWSDL" + scriptExtn() + " .." + File.separator + wmq_wsdl) ||
			!doBatCommand("amqwcallWSDL" + scriptExtn() + " .." + File.separator + wmq_wsdl + " /language:VB"))
		{
			return false;
		}

		/* Make sure the proxies are generated under [generatedClientDir]/remote to
		 * prevent any confusions with the service source
		 */ 
		String x = "java com.ibm.mq.soap.util.RunWSDL2Java --output " + generatedClientDir + File.separator + "remote -p dotNetService generated" + File.separator + wmq_wsdl;

		if (!doCommand(x))
		{
			return false;
		}

		String listOfJavaFiles = getJavaFiles(getAbsParent(generatedClientDir + File.separator + "remote" + File.separator + "dotNetService" + File.separator + className));
		if (listOfJavaFiles.equals(""))
		{
			return false;
		}
     
		String outputDir = generatedClientDir + File.separator + "remote";

		String compileCommand = "javac -d " + outputDir + " " + listOfJavaFiles;
		if (!doCommand (compileCommand))
		{
			return false;
		}
    
		return true;
	}
  
  
	/**
	 * This method returns list of all Java files (with full abs directory spec) in the 
	 * specified directory
	 * 
	 * @param aDir is the name of the directory to be scanned
	 * @return if successful and false otherwise
	 */
	private String getJavaFiles(final String aDir)
	{
		String listOfFiles="";
		File dir = new File(aDir);
        
		/**
		 * It is also possible to filter the list of returned files.
		 * This example does not return any files that start with `.'.
		 */
		FilenameFilter filter = new FilenameFilter() 
		{
			public boolean accept(File dir, String name) 
			{
				return name.endsWith(".java");
			}
		};
    
		String[] children = dir.list(filter);
		for (int i=0; i<children.length; i++)
		{
                        /* On windows, spaces in directory names causes problems.*/
                        if (windows)
 			  listOfFiles += "\"" + aDir + children[i] + "\" ";
                        else
 			  listOfFiles += aDir + children[i] + " ";
		}
  
		return listOfFiles;
	}
  
	/**
	 * Method that generates the proxies for an axis service
	 * 
	 * @return true if successful and false otherwise
	 */
	private boolean genProxiesToAxis()
	{
		/** - Generate all proxies (C#, VB and Java) and server wsdd file, and compile the Java */
  
		if (windows)
		{
			if (!doBatCommand("amqwcallWSDL" + scriptExtn() + " .." + File.separator + wmq_wsdl) ||
				!doBatCommand("amqwcallWSDL" + scriptExtn() + " .." + File.separator + wmq_wsdl + " /language:VB"))
			{
				return false;
			}
		}
 
		/** 
		 * Use -p RunWSDL2Java option to override packagename - otherwise use of --namespace 
		 * earlier results in a packagename the wrong way round
		 * 
		 * Place the proxies under [generatedClientDir]/remote to prevent any confusion 
		 * with out service source
		 */

		if (!doCommand("java com.ibm.mq.soap.util.RunWSDL2Java --timeout -1 --output " + generatedClientDir +File.separator + "remote" + " -p " + packageName + " " + "generated" + File.separator + wmq_wsdl))
		{
			return false;
		}
     
		String listOfJavaFiles = getJavaFiles(getAbsParent(generatedClientDir + File.separator + "remote" + File.separator + proxyDir));

		if (listOfJavaFiles.equals(""))
		{
			return false;
		}
     
		String outputDir = generatedClientDir + File.separator + "remote";

		String compileCommand = "javac -d " + outputDir + " " + listOfJavaFiles;
		if (!doCommand(compileCommand))

		{
			return false;
		}
 

		return true;  
	}

	/**
	 * Method to execute all of the Axis deployment steps. This will be called if not
	 * executing a single deployment step with the -c option.
	 * 
	 * @return true if successful and false otherwise
	 */
	private boolean allAxis()
	{

		/** Compile service code */
		if (!compileJava()) return false;
 
		/** Generate the correct wsdl, do not override any old versions */
		if (!genAxisWsdl()) return false;
  
		/** Generate the server side wsdd file */
		if (!axisDeploy()) return false;
  
		/** Setup WMQ queues, listeners and triggers */
		if (!genAxisWMQBits()) return false;
  
		/** Generate proxies */
		if (!genProxiesToAxis()) return false;

		return true;
	}

	/**
	 * Method to execute all of the Asmx deployment steps. This will be called if not
	 * executing a single deployment step with the -c option.
	 * 
	 * @return true if successful and false otherwise
	 */
	private boolean allAsmx()
	{
		/** Generate WSDL */
		if (!genAsmxWsdl()) return false;

		/** Setup WMQ queues, listeners and triggers */
		if (!genAsmxWMQBits()) return false;

		/** Generate and compile proxies */
		if (!genProxiesToDotNet()) return false;

		return true;
	}
  
	/**
	 * Method to start a new command window
	 * 
	 * @param command is a String specification of the command to execute
	 * @param winTitle is a String specification of the Window title
	 * @return true if successful and false otherwise
	 */
	private boolean StartWindow(String command, String winTitle)
	{
		return doCommand(START + " " + winTitle + " " + command);
	}
  
	private boolean startWMQMonitor()
	{
		if (!triggeringEnabled)
		{
			return false;
		}

		String startTriggerMonitorCommand = triggerProgName + " -m " + queueManager + " -q " + triggerQueueName;
		return StartWindow(startTriggerMonitorCommand, "MQSoap_Trigger_Monitor");
	}

	/**
	 * Constructor
	 * 
	 * Validate command line arguments, perform setup and further validation,
	 * set required verbosity, set up use of "replace" mode if required with "-r" 
	 * and execute the required deployment process.
	 * 
	 * @param args is an array of Strings specifying the arguments supplied to the program
	 */
	DeployWMQService(String[] args)
	{
		/** Determine if we are on a Windows system */
		if (System.getProperty("os.name").toLowerCase().startsWith("win"))
			windows =  true;

		if (!processArgs(args))
		{
			System.exit(2);
		}
     
		if (todo.equals("usage")) 
		{
			usage();  
			return;
		}
     
		/** Specify verbose output from external commands if required */
		//RunCommand.setQuiet(!verbose);
		//RunMQCommand.setQuiet(!verbose);
     
		try
		{
			if (!setup())
                        {
                                System.exit(2);
                        }
		}
		catch (Exception e)
		{
                        System.exit(2);
		}

		START="amqwstartwin" + scriptExtn();

		/** If '-r' has been specified check if the queues already exist */
		if (replaceQs)
		{
			boolean qmgrExists=true;
			MQQueueManager qm = null;

			try
			{
				qm = new MQQueueManager(queueManager);
			}
			catch (MQException e)
			{
				qmgrExists = false;
			}

			if(qmgrExists)
			{
				try
				{
					MQQueue q = qm.accessQueue(queue, MQC.MQOO_INPUT_AS_Q_DEF, null, null, null);
					qCmdStr = "ALTER";
				}
				catch (MQException e)
				{
				}

				if (triggerQueueName != null)
				{
					try
					{
						MQQueue q = qm.accessQueue(triggerQueueName, MQC.MQOO_INPUT_AS_Q_DEF, null, null, null);
						trqCmdStr = "ALTER";
					}
					catch (MQException e)
					{
					}
				}
			}
		}

		if (todo.equalsIgnoreCase("allAxis"))
                {
                        if (!allAxis())
                        {
                                System.exit(2);
                        }
                }
		else if (todo.equalsIgnoreCase("allAsmx"))
                {
                        if (!allAsmx())
                        {
                                System.exit(2);
                        }
                }
		else if (todo.equalsIgnoreCase("compileJava"))
                {
                        if (!compileJava())
                        {
                                System.exit(2);
                        }
                }
		else if (todo.equalsIgnoreCase("genAxisWsdl"))
                {
                        if (!genAxisWsdl())
                        {
                                System.exit(2);
                        }
                }
		else if (todo.equalsIgnoreCase("axisDeploy"))
                {
                        if (!axisDeploy())
                        {
                                System.exit(2);
                        }
                }
		else if (todo.equalsIgnoreCase("genAxisWMQBits"))
                {
                        if (!genAxisWMQBits())
                        {
                                System.exit(2);
                        }
                }
		else if (todo.equalsIgnoreCase("genProxiesToAxis"))
                {
                        if (!genProxiesToAxis())
                        {
                                System.exit(2);
                        }
                }
		else if (todo.equalsIgnoreCase("genAsmxWsdl"))
                {
                        if (!genAsmxWsdl())
                        {
                                System.exit(2);
                        }
                }
		else if (todo.equalsIgnoreCase("genProxiesToDotNet"))
                {
                        if (!genProxiesToDotNet())
                        {
                                System.exit(2);
                        }
                }
		else if (todo.equalsIgnoreCase("genAsmxWMQBits"))
                {
                        if (!genAsmxWMQBits())
                        {
                                System.exit(2);
                        }
                } 
		else if (todo.equalsIgnoreCase("startWMQMonitor"))
                {
                        if (!startWMQMonitor())
                        {
                                System.exit(2);
                        }
                }
  
	}

	/**
	 * main program
	 * 
	 * @param args is an array of Strings which are the arguments passed from the command line
	 */
	public static void main(String[] args) 
	{

		/** Execute the deployment process */
		new DeployWMQService(args);
    	}
}
