       IDENTIFICATION DIVISION.
      ****************************************************************
      *                                                              *
      * Program name: AMQ0XAG0                                       *
      *                                                              *
      * Description:  Sample CBL program for MQ coordinating         *
      *               XA-compliant database managers.                *
      *                                                              *
      * Note:         Useful information and diagrams about this     *
      *               sample are in the Application Programming      *
      *               Guide.                                         *
      *   <copyright                                                 *
      *   notice="lm-source-program"                                 *
      *   pids="5724-H72,"                                           *
      *   years="1998,2012"                                          *
      *   crc="2355577207" >                                         *
      *   Licensed Materials - Property of IBM                       *
      *                                                              *
      *   5724-H72,                                                  *
      *                                                              *
      *   (C) Copyright IBM Corp. 1998, 2012 All Rights Reserved.    *
      *                                                              *
      *   US Government Users Restricted Rights - Use, duplication or  *
      *   disclosure restricted by GSA ADP Schedule Contract with    *
      *   IBM Corp.                                                  *
      *   </copyright>                                               *
      ****************************************************************
      *                                                              *
      * Function:                                                    *
      *                                                              *
      *    AMQ0XAG0 is a sample CBL program for MQ to coordinate     *
      *    getting a message off a queue and updating two databases  *
      *    within an WebSphere MQ unit of work. This sample calls    *
      *    functions that are in AMQ0XAB0.SQB and AMQ0XAF0.SQB,      *
      *    these contain the SQL to update the two databases and     *
      *    hence must be prepared with the appropriate database.     *
      *                                                              *
      * Note: If you only have one XAResourceManager stanza in your  *
      *       QM.INI file for a database program and hence are only  *
      *       concerned with one database within that database       *
      *       program, it is not necessary to issue the "EXEC SQL    *
      *       CONNECT TO XXXX" statements in AMQ0XAB0.SQB or         *
      *       AMQ0XAF0.SQB, as the connection will be implicitly     *
      *       made for you. As anybody could modify the QM.INI file  *
      *       at sometime in the future, it is probably worth        *
      *       leaving in.                                            *
      *                                                              *
      *    -- A message is read from a queue (under sync point), it  *
      *       must be in the form:                                   *
      *                                                              *
      *       UPDATE Balance change=snnnnnnnn WHERE Account=nnnnnnnn *
      *                                                              *
      *       For simplicity, the balance change must be a signed    *
      *       eight character number and the account number must be  *
      *       an eight character number.                             *
      *                                                              *
      *       The sample AMQSPUT can be used to put the messages on  *
      *       the queue.                                             *
      *                                                              *
      *    -- Information from the databases is obtained and updated *
      *       with the information in the message.                   *
      *                                                              *
      *    -- The new status of the database is printed.             *
      *                                                              *
      * Program logic:                                               *
      *                                                              *
      *    MQCONN connect to default queue manager                   *
      *    MQOPEN open queue for input (using supplied parameter)    *
      *    while no failures                                         *
      *    .  MQBEGIN start a unit of work                           *
      *    .  MQGET get next message from queue under sync point     *
      *    .  get information from databases                         *
      *    .  update information from databases                      *
      *    .  MQCMIT commit changes                                  *
      *    .  print updated information                              *
      *    .  (no message available counts as failure, and loop ends)*
      *    MQCLOSE close queue                                       *
      *    MQDISC disconnect from queue manager                      *
      *                                                              *
      ****************************************************************
      *                                                              *
      * AMQ0XAG0 has no parameters                                   *
      *                                                              *
      ****************************************************************

       PROGRAM-ID. 'AMQ0XAG0'.

      ****************************************************************
       DATA DIVISION.
       WORKING-STORAGE SECTION.

      *
      **  Declare MQI structures needed
      * MQI named constants
       01 MY-MQ-CONSTANTS.
          COPY CMQV.
      * Object Descriptor
       01 OBJECT-DESCRIPTOR.
          COPY CMQODV.
      * Message Descriptor
       01 MESSAGE-DESCRIPTOR.
          COPY CMQMDV.
      * Get message options
       01 GMOPTIONS.
          COPY CMQGMOV.
      * Begin options
       01 BOPTIONS.
          COPY CMQBOV.

      ** note, sample uses defaults where it can

       01 OK                       PIC S9(9) BINARY VALUE 0.
       01 NOT-OK                   PIC S9(9) BINARY VALUE 1.
       01 QM-NAME                  PIC X(48) VALUE SPACES.
       01 HCONN                    PIC S9(9) BINARY.
       01 Q-HANDLE                 PIC S9(9) BINARY.
       01 OPTIONS                  PIC S9(9) BINARY.
       01 COMP-CODE                PIC S9(9) BINARY.
       01 OPEN-COMP-CODE           PIC S9(9) BINARY.
       01 REASON                   PIC S9(9) BINARY.
       01 CONN-REASON              PIC S9(9) BINARY.
       01 MSG-BUF.
          02 UPDATE-TEXT           PIC X(22).
          02 BALANCE-CHANGE        PIC S9(8) SIGN IS LEADING SEPARATE.
          02 ACCOUNT-TEXT          PIC X(15).
          02 ACCOUNT-NUM           PIC 9(8).
          02 FILLER                PIC X(45).
       01 MSG-BUF-LEN              PIC S9(9) BINARY.
       01 MSG-LEN                  PIC S9(9) BINARY.
       01 GOT-MSG                  PIC S9(9) BINARY.
       01 COMMITTED-UPDATE         PIC S9(9) BINARY.
       01 SOURCE-QUEUE             PIC X(48).
       01 ACC-NAME                 PIC X(48) VALUE SPACES.
       01 ACCOUNT                  PIC S9(9) BINARY.
       01 BALANCE                  PIC S9(9) BINARY.
       01 TRANSACTIONS             PIC S9(9) BINARY.
       01 TEMP                     PIC S9(9) BINARY.
       01 FEE-DUE                  PIC S9(9) BINARY.
       01 TRAN-FEE                 PIC S9(9) BINARY.

      ****************************************************************
       PROCEDURE DIVISION.
       P0.
      ** indicate that sample program has started
           DISPLAY 'AMQ0XAG0 start'.

      ****************************************************************
      * Declare the cursor for locking of reads from database        *
      ****************************************************************

           CALL 'DECLAREMQBANKDBCURSOR'.
           IF RETURN-CODE IS NOT EQUAL TO OK
             GOBACK
           END-IF.

           CALL 'DECLAREMQFEEDBCURSOR'.
           IF RETURN-CODE IS NOT EQUAL TO OK
             GOBACK
           END-IF.

      ****************************************************************
      * Display prompt for the name of the source queue              *
      ****************************************************************
           DISPLAY 'Please enter the name of the source queue '

      ** get the source queue from StdIn.
           ACCEPT SOURCE-QUEUE FROM CONSOLE.

      ****************************************************************
      * Connect to default queue manager                             *
      ****************************************************************
           CALL 'MQCONN' USING QM-NAME, HCONN, COMP-CODE, CONN-REASON.

      *      report reason and stop if it failed
           IF COMP-CODE IS EQUAL TO MQCC-FAILED
             DISPLAY 'MQCONN ended with reason code ' CONN-REASON
             MOVE CONN-REASON TO RETURN-CODE
             GOBACK
           END-IF.
      *
      ****************************************************************
      * Open the message queue (and fail if MQM is quiescing)        *
      ****************************************************************
       OPENS.
           MOVE SOURCE-QUEUE TO MQOD-OBJECTNAME.
           ADD MQOO-INPUT-AS-Q-DEF MQOO-FAIL-IF-QUIESCING
                     GIVING OPTIONS.
           CALL 'MQOPEN'
            USING HCONN, OBJECT-DESCRIPTOR,
            OPTIONS, Q-HANDLE,
            OPEN-COMP-CODE, REASON.

      *      report reason, if any; stop if failed
           IF REASON IS NOT EQUAL TO MQRC-NONE
             DISPLAY 'MQOPEN ended with reason code ' REASON
           END-IF.

           IF OPEN-COMP-CODE IS EQUAL TO MQCC-FAILED
             DISPLAY 'unable to open queue for input'
           END-IF.

      ****************************************************************
      * Set up some things for the MQGET                             *
      ****************************************************************
           ADD MQGMO-WAIT MQGMO-CONVERT MQGMO-SYNCPOINT
                GIVING MQGMO-OPTIONS.
           MOVE 15000 TO MQGMO-WAITINTERVAL.
           MOVE 64 to MSG-BUF-LEN.

      ****************************************************************
      * Get messages from the message queue                          *
      ****************************************************************
       GETS.
           MOVE OK TO RETURN-CODE.
           MOVE OPEN-COMP-CODE TO COMP-CODE.
           PERFORM STARTLOOP THRU BACKO WITH TEST BEFORE
             UNTIL COMP-CODE IS EQUAL TO MQCC-FAILED OR
                   RETURN-CODE IS NOT EQUAL TO OK.

      ****************************************************************
      * Close the queue if opened                                    *
      ****************************************************************
       CLOSES.
           IF OPEN-COMP-CODE IS NOT EQUAL TO MQCC-FAILED
             MOVE MQCO-NONE TO OPTIONS
             CALL 'MQCLOSE'
              USING HCONN, Q-HANDLE, OPTIONS,
              COMP-CODE, REASON

      *        report reason, if any
             IF REASON IS NOT EQUAL TO MQRC-NONE
               DISPLAY 'MQCLOSE ended with reason code ' REASON
             END-IF
           END-IF.

      ****************************************************************
      * Disconnect from queue manager if not previously connected    *
      ****************************************************************
       DISCS.
           IF CONN-REASON IS NOT EQUAL TO MQRC-ALREADY-CONNECTED
             CALL 'MQDISC' USING HCONN, COMP-CODE, REASON

      *      report reason, if any
             IF REASON IS NOT EQUAL TO MQRC-NONE
               DISPLAY 'MQDISC ended with reason code ' REASON
             END-IF
           END-IF.

       OVER.
      ** indicate that sample program has finished
           DISPLAY 'AMQ0XAG0 end'.
           MOVE OK TO RETURN-CODE.
           GOBACK.

      ****************************************************************
      * Start of loop to get a message off the queue and update bank *
      * account.                                                     *
      ****************************************************************
       STARTLOOP.
      ****************************************************************
      * Set flags so that we can back out if something goes wrong    *
      * and not lose the message.                                    *
      ****************************************************************
           MOVE ZERO TO GOT-MSG.
           MOVE ZERO TO COMMITTED-UPDATE.

      ****************************************************************
      * Start a unit of work                                         *
      ****************************************************************
       BEGINS.
           CALL 'MQBEGIN' USING HCONN, BOPTIONS, COMP-CODE, REASON.

      *    ***********************************************************
      *    * If we get a reason code and only a warning on the comp  *
      *    * code, there is something wrong with one or more of the  *
      *    * resource managers so stop looping and sort it out,      *
      *    * whatever the comp code.                                 *
      *    ***********************************************************
           IF REASON IS EQUAL TO MQRC-NONE
             DISPLAY 'Unit of work started'
             MOVE OK TO RETURN-CODE
           ELSE
             DISPLAY 'MQBEGIN ended with reason code ' REASON
               IF COMP-CODE IS EQUAL TO MQCC-FAILED
                 DISPLAY 'Unable to start a unit of work'
               END-IF
             MOVE NOT-OK TO RETURN-CODE
           END-IF.

      ****************************************************************
      * Get message off queue                                        *
      ****************************************************************
       GETR.
           IF RETURN-CODE IS EQUAL TO OK
      *      *********************************************************
      *      * In order to read the messages in sequence, MsgId and  *
      *      * CorrelID must have the default value. MQGET sets them *
      *      * to the values for the message it returns, so re-      *
      *      * initialise them before every call.                    *
      *      *********************************************************
             MOVE MQMI-NONE TO MQMD-MSGID
             MOVE MQCI-NONE TO MQMD-CORRELID
             MOVE SPACES TO MSG-BUF

             CALL 'MQGET'
              USING HCONN, Q-HANDLE,
              MESSAGE-DESCRIPTOR, GMOPTIONS,
              MSG-BUF-LEN, MSG-BUF, MSG-LEN,
              COMP-CODE, REASON

             IF REASON IS NOT EQUAL TO MQRC-NONE
               IF REASON IS EQUAL TO MQRC-NO-MSG-AVAILABLE
                 DISPLAY 'no more messages'
               ELSE
                 DISPLAY 'MQGET ended with reason code ' REASON
                 IF REASON IS EQUAL TO MQRC-TRUNCATED-MSG-FAILED
                   MOVE MQCC-FAILED TO COMP-CODE
                 END-IF
               END-IF
             ELSE
               MOVE 1 TO GOT-MSG
               MOVE OK TO RETURN-CODE
             END-IF
           END-IF.

      ****************************************************************
      * Process the message received                                 *
      ****************************************************************
       PROCM.
           IF COMP-CODE IS NOT EQUAL TO MQCC-FAILED AND
              RETURN-CODE IS EQUAL TO OK
             IF UPDATE-TEXT IS NOT EQUAL TO 'UPDATE Balance change='
                OR ACCOUNT-TEXT IS NOT EQUAL TO ' WHERE Account='
               DISPLAY 'Invalid string received'
               MOVE NOT-OK TO RETURN-CODE
             ELSE
               MOVE ACCOUNT-NUM TO ACCOUNT
             END-IF

      ****************************************************************
      * Note only actively connected to one database at a time       *
      ****************************************************************
      ****************************************************************
      * Get details from database                                    *
      ****************************************************************
             IF RETURN-CODE IS EQUAL TO OK
               CALL 'CONNECTTOMQBANKDB'
             END-IF

             IF RETURN-CODE IS EQUAL TO OK
               CALL 'GETMQBANKTBDETAILS'
                USING BY VALUE ACCOUNT
                BY REFERENCE ACC-NAME BALANCE TRANSACTIONS
             END-IF

             IF RETURN-CODE IS EQUAL TO OK
               CALL 'CONNECTTOMQFEEDB'
             END-IF

             IF RETURN-CODE IS EQUAL TO OK
               CALL 'GETMQFEETBDETAILS'
                USING BY VALUE ACCOUNT
                BY REFERENCE FEE-DUE TRAN-FEE TEMP
             END-IF

      ****************************************************************
      * The number of transactions to the two databases should be    *
      * identical, stop if not.                                      *
      ****************************************************************
             IF RETURN-CODE IS EQUAL TO OK
               IF TEMP IS NOT EQUAL TO TRANSACTIONS
                 DISPLAY 'Databases are out of step !'
                 MOVE NOT-OK TO RETURN-CODE
               END-IF
             END-IF

      ****************************************************************
      * Update the bank balance                                      *
      ****************************************************************
             IF RETURN-CODE IS EQUAL TO OK
               ADD 1 TO TRANSACTIONS
               MOVE BALANCE TO TEMP
               ADD BALANCE-CHANGE TO BALANCE
               ADD TRAN-FEE TO FEE-DUE

               CALL 'UPDATEMQFEETBFEEDUE'
                USING BY VALUE FEE-DUE TRANSACTIONS

               IF RETURN-CODE IS EQUAL TO OK
                 CALL 'CONNECTTOMQBANKDB'
               END-IF

               IF RETURN-CODE IS EQUAL TO OK
                 CALL 'UPDATEMQBANKTBBALANCE'
                  USING BY VALUE BALANCE TRANSACTIONS
               END-IF

               IF RETURN-CODE IS EQUAL TO OK

                 DISPLAY 'Account No:' ACCOUNT
                         ' Balance updated from ' TEMP
                         ' to ' BALANCE
                         ' ' ACC-NAME
                 COMPUTE TEMP = FEE-DUE - TRAN-FEE
                 DISPLAY 'Fee Due updated from ' TEMP
                         ' to ' FEE-DUE

      *          *****************************************************
      *          * We are going to commit the update so even if      *
      *          * something goes wrong now, the message has been    *
      *          * used so don't back out.                           *
      *          *****************************************************
                 MOVE 1 TO COMMITTED-UPDATE

      *          *****************************************************
      *          * Note: the cursor will be implicitly closed by the *
      *          * MQCMIT.                                           *
      *          *****************************************************
                 CALL 'MQCMIT' USING HCONN, COMP-CODE, REASON

                 IF REASON IS EQUAL TO MQRC-NONE
                   DISPLAY 'Unit of work successfully completed'
                   MOVE OK TO RETURN-CODE
                 ELSE
                   DISPLAY 'MQCMIT ended with reason code ' REASON
                           ' compleation code ' COMP-CODE
                   MOVE NOT-OK TO RETURN-CODE
                 END-IF
               END-IF
             END-IF
           END-IF.

      ****************************************************************
      * If we got the message but something went wrong, back out so  *
      * that we don't lose the message.                              *
      ****************************************************************
       BACKO.
           IF GOT-MSG IS EQUAL TO 1 AND
              COMMITTED-UPDATE IS EQUAL TO ZERO
             CALL 'MQBACK' USING HCONN, COMP-CODE, REASON
             IF REASON IS EQUAL TO MQRC-NONE
               DISPLAY 'MQBACK successfully issued'
             ELSE
               DISPLAY 'MQBACK ended with reason code ' REASON
             END-IF
      *      *********************************************************
      *      * We have issued MQBACK so we must stop looping         *
      *      *********************************************************
             MOVE NOT-OK TO RETURN-CODE
           END-IF.

      ****************************************************************
      *                                                              *
      * END OF AMQ0XAG0                                              *
      *                                                              *
      ****************************************************************
